﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <memory>

#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/htcs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <Utils/GTestUtil.h>


namespace // unnamed, for constants accessible only to this file
{

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }
}




// send the first data received back to the the user, unchanged
bool EchoBack(int sock)
{
    char buf[128];
    nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
    if (receivedBytes < 0)
    {
        NN_LOG("ERROR: recv failed, err=%d\n", nn::htcs::GetLastError());
        return false;
    }
    else if (receivedBytes == 0)
    {
        NN_LOG("ERROR: connection closed by peer\n");
        return false;
    }

    // this should never happen, as this is limited by the
    // value passed into Recv
    NN_ASSERT( receivedBytes < sizeof(buf) );

    buf[receivedBytes] = '\0';
    NN_LOG("Received \"%s\".\n", buf);

    nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, buf, receivedBytes, 0);
    if (sentBytes < 0)
    {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
        return false;
    }
    return true;
}



TEST(natf, HTCSLib_Target)
{
    int result;
    NN_LOG("\n Starting: HTCSLib_Target\n");
    nn::htcs::Initialize(&Allocate, &Deallocate);

    // register the service
    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serviceAddr.portName.name, "ServerInTarget");

    int serviceSocket = nn::htcs::Socket();
    NATF_EXPECT_TRUE( serviceSocket >= 0,
                      "ERROR: Socket create failed, err=%d\n",  nn::htcs::GetLastError() );

    result = nn::htcs::Bind(serviceSocket, &serviceAddr);
    NATF_EXPECT_TRUE( result == 0,
                      "ERROR: Bind failed, err=%d\n",  nn::htcs::GetLastError() );

    result = nn::htcs::Listen(serviceSocket, 1);
    NATF_EXPECT_TRUE( result == 0,
                      "ERROR: Listen failed, err=%d\n",  nn::htcs::GetLastError() );

    // wait for a connection
    NN_LOG("Waiting for connection from host.\n");
    int sock = nn::htcs::Accept(serviceSocket, nullptr);
    NATF_EXPECT_TRUE( sock >= 0,
                      "ERROR: Accept failed, err=%d\n",  nn::htcs::GetLastError() );

    // got a connection!!
    NN_LOG("Accepted client on host.\n");

    // run a one-time echo-back test
    NATF_EXPECT_TRUE( EchoBack(sock), "Error communicating, sock=%d\n", sock  );

    // break the connection
    result = nn::htcs::Close(sock);
    NATF_EXPECT_TRUE( result == 0,
                      "ERROR: Close data socket failed, err=%d\n",  nn::htcs::GetLastError() );

    result = nn::htcs::Close(serviceSocket);
    NATF_EXPECT_TRUE( result == 0,
                      "ERROR: Close listen socket failed, err=%d\n",  nn::htcs::GetLastError() );

    NN_LOG("\n Finished: HTCSLib_Target\n");

}


