﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Tests/CurlHttpDownloadMultiTest.h"
#include "Utils/CommandLineParser.h"
#include "../Common/ParseHelpers.h"

#include <nnt/nntest.h>
#include <curl/curl.h>

namespace
{
    const uint32_t WaitTypeBufLen = 16;
    const uint32_t ProxyAuthBufLen = 32;
    const uint32_t CacheModeBufLen = 8;
}

// CurlHttpDownloadMulti
TEST(natf, CurlHttpDownloadMulti)
{
    bool isSuccess = true;
    NATF::Utils::ParserGroup parser;
    NATF::Modules::LibCurl::Params params;
    int argc;
    int32_t defaultHttpResponse = 200;
    uint32_t defaultConnectionCount = 1;
    uint32_t connectionCount = defaultConnectionCount;
    uint32_t defaultMultiApiConnectionCount = 0;
    uint32_t timeoutMs = 0;
    uint32_t defaultVerifyArrayLen = 4;
    int32_t defaultCurlReturn = 0;
    uint16_t defaultProxyPort = 0;
    int32_t defaultCurlHandleLoop = 1;
    int32_t defaultCurlPerformLoop = 1;
    int32_t defaultSendBuf = 32 * 1024;
    int32_t defaultRecvBuf = 256 * 1024;
    int defaultAuthError = 0;
    bool defaultDoVerifyPeer = true;
    bool defaultDoVerifyHostName = true;
    bool defaultDoVerifyDate = true;
    bool defaultUseInternalPki = false;
    bool defaultTeamCityDown = false;
    bool defaultExpectBufferTooShort = false;
    bool defaultCheckHashEnabled = true;
    bool defaultUseAutoUrl = false;
    nn::util::Uuid netProfile;
    char pTestName[NATF::BaseTest::NameBufferLen];
    char pWaitType[WaitTypeBufLen];
    char pProxyAuth[ProxyAuthBufLen];
    char pCacheModeStr[CacheModeBufLen];
    const char* pDefaultWaitType = "Select";
    const char* pDefaultProxyServer = "";
    const char* pDefaultProxyCredentials = "";
    const char* pDefaultProxyAuth = "NEGOTIATE";
    const char* pDefaultCacheMode = "NONE";
    const char * const * pArgv;

    NN_NETTEST_LOG("\nEnter NATF Process\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(NATF::Utils::StringParser  ("--Name", nullptr, pTestName, sizeof(pTestName)));
    parser.AddParser(NATF::Utils::UInt32Parser  ("--ConnectionCount", &defaultConnectionCount, connectionCount));
    parser.AddParser(NATF::Utils::UInt32Parser  ("--MultiApiConnectionCount", &defaultMultiApiConnectionCount, params.multiApiConnectionCount));
    parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfile));
    parser.AddParser(NATF::Utils::StringParser  ("--Url", nullptr, params.pUrl, sizeof(params.pUrl)));
    parser.AddParser(NATF::Utils::Int32Parser   ("--HttpResponse", &defaultHttpResponse, params.expectedHttpResponse));
    parser.AddParser(NATF::Utils::Int32Parser   ("--SslAuthError", &defaultAuthError, params.expectedAuthReturn));
    parser.AddParser(NATF::Utils::UInt32Parser  ("--TimeoutMs", nullptr, timeoutMs));
    parser.AddParser(NATF::Utils::Md5HashParser ("--Md5Hash", nullptr, params.expectedHash));
    parser.AddParser(NATF::Utils::StringParser  ("--WaitType", pDefaultWaitType, pWaitType, sizeof(pWaitType)));
    parser.AddParser(NATF::Utils::BoolParser    ("--VerifyPeer", &defaultDoVerifyPeer, params.doVerifyPeer));
    parser.AddParser(NATF::Utils::BoolParser    ("--VerifyHostName", &defaultDoVerifyHostName, params.doVerifyHostname));
    parser.AddParser(NATF::Utils::BoolParser    ("--VerifyDate", &defaultDoVerifyDate, params.doVerifyDate));
    parser.AddParser(NATF::Utils::BoolParser    ("--UseInternalPki", &defaultUseInternalPki, params.useInternalPki));
    parser.AddParser(NATF::Utils::BoolParser    ("--LogTeamCityDownSpeed", &defaultTeamCityDown, params.publishDownSpeedToTeamCity));
    parser.AddParser(NATF::Utils::BoolParser    ("--ExpectBufferTooShort", &defaultExpectBufferTooShort, params.isExpectingBufferTooShort));
    parser.AddParser(NATF::Utils::BoolParser    ("--CheckHash", &defaultCheckHashEnabled, params.checkHash));
    parser.AddParser(NATF::Utils::BoolParser    ("--AutoUrl", &defaultUseAutoUrl, params.isAutoUrlEnabled));
    parser.AddParser(NATF::Utils::StringParser  ("--ProxyServer", pDefaultProxyServer, params.pProxyServer, sizeof(params.pProxyServer)));
    parser.AddParser(NATF::Utils::StringParser  ("--ProxyCredentials", pDefaultProxyCredentials, params.pProxyUserPwd, sizeof(params.pProxyUserPwd)));
    parser.AddParser(NATF::Utils::StringParser  ("--ClientCert", "", params.pClientCertPath, sizeof(params.pClientCertPath)));
    parser.AddParser(NATF::Utils::StringParser  ("--ServerCert", "", params.pServerCertPath, sizeof(params.pServerCertPath)));
    parser.AddParser(NATF::Utils::UInt16Parser  ("--ProxyPort", &defaultProxyPort, params.proxyPort));
    parser.AddParser(NATF::Utils::StringParser  ("--ProxyAuth", pDefaultProxyAuth, pProxyAuth, sizeof(pProxyAuth)));
    parser.AddParser(NATF::Utils::StringParser  ("--SessionCacheMode", pDefaultCacheMode, pCacheModeStr, sizeof(pCacheModeStr)));
    parser.AddParser(NATF::Utils::StringParser  ("--HttpHeaderOption", "", params.pHttpHeaderOption, sizeof(params.pHttpHeaderOption)));
    parser.AddParser(NATF::Utils::Int32Parser   ("--CurlReturn", &defaultCurlReturn, params.expectedCurlReturn));
    parser.AddParser(NATF::Utils::Int32Parser   ("--CurlHandleLoop", &defaultCurlHandleLoop, params.curlHandleLoop));
    parser.AddParser(NATF::Utils::Int32Parser   ("--CurlPerformLoop", &defaultCurlPerformLoop, params.curlPerformLoop));
    parser.AddParser(NATF::Utils::Int32Parser   ("--SendBuffer", &defaultSendBuf, params.sendBuffer));
    parser.AddParser(NATF::Utils::Int32Parser   ("--RecvBuffer", &defaultRecvBuf, params.recvBuffer));
    parser.AddParser(NATF::Utils::UInt32Parser  ("--VerifyArrayLen", &defaultVerifyArrayLen, params.verifyResultArrayLen));
    parser.AddParser(NATF::Utils::StringParser  ("--ExpectedCipher", "", params.pExpectedCipherName, sizeof(params.pExpectedCipherName)));
    parser.AddParser(NATF::Utils::StringParser  ("--ExpectedSslVersion", "", params.pExpectedSslVersion, sizeof(params.pExpectedSslVersion)));

    if (!parser.Parse(argc, pArgv))
    {
        NN_NETTEST_LOG(" * Failed to parse command line arguements!\n\n");
        EXPECT_EQ(false, true);
        return;
    }

    NetTest::StrUpr(pWaitType);
    if(nnt::ParserHelpers::GetWaitTypeFromString(pWaitType, WaitTypeBufLen, params.useSelect) == false)
    {
        FAIL();
        return;
    }

    NetTest::StrUpr(pCacheModeStr);
    if(nnt::ParserHelpers::GetCacheModeFromString(pCacheModeStr, CacheModeBufLen, params.sessionCacheMode) == false)
    {
        FAIL();
        return;
    }

    if( strncmp(pProxyAuth, "", ProxyAuthBufLen) != 0 )
    {
        NetTest::StrUpr(pProxyAuth);
        if(nnt::ParserHelpers::GetProxyAuthFromString(pProxyAuth, ProxyAuthBufLen, params.proxyAuthMethod) == false)
        {
            FAIL();
            return;
        }
    }

    NATF::Tests::CurlHttpDownloadMulti curlHttpDownloadMulti(pTestName, netProfile, timeoutMs, connectionCount, params);

    isSuccess = curlHttpDownloadMulti.Run();
    EXPECT_EQ(isSuccess, true);

    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(2000) );
    NN_NETTEST_LOG("\nExit NATF Process\n\n");
} // NOLINT(impl/function_size)
