﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nnt.h>

#include <nn/socket/socket_Api.h>
#include <nn/bsdsocket/cfg/cfg.h>

namespace nnt { namespace bsdsocket { namespace cfg {


#define COMPARE_RESULT(a, b)  \
    ((((a).GetInnerValueForDebug()) == ((b).GetInnerValueForDebug())) ? true : false)

class TestClientThread
{
public:
    typedef nn::Result(*TestClientThreadEntry)(const char *ifName, void* arg0);
    void Initialize(TestClientThreadEntry funcPtr, const char *ifName, void *arg0)
    {
        m_pIfName = ifName;
        m_Args[0] = arg0;
        m_Entry   = funcPtr;
        m_Result = nn::os::CreateThread(&m_Thread, ThreadEntryStatic,
                                        this, m_Stack, sizeof(m_Stack),
                                        nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
        NNT_EXPECT_RESULT_SUCCESS(m_Result);
        nn::os::StartThread(&m_Thread);
    }
    nn::Result Finalize()
    {
        WaitThread(&m_Thread);
        DestroyThread(&m_Thread);
        return m_Result;
    }
private:
    static void ThreadEntryStatic(void *arg)
    {
        TestClientThread *pThis = reinterpret_cast<TestClientThread *>(arg);
        pThis->m_Result = pThis->m_Entry(pThis->m_pIfName, pThis->m_Args[0]);
    }
    const char            *m_pIfName;
    void                  *m_Args[1];
    nn::Result             m_Result;
    TestClientThreadEntry  m_Entry;
    uint8_t                m_Stack[16384]__attribute__((aligned(nn::os::StackRegionAlignment)));
    nn::os::ThreadType     m_Thread;
};

static nn::bsdsocket::cfg::IfSettings s_DefaultCfg;
static const char *s_IfName = "usb0";
static TestClientThread s_ClientThreads[2];

void ShowIfState(const char *ifName)
{
    nn::bsdsocket::cfg::IfState ifState;
    NNT_EXPECT_RESULT_SUCCESS(nn::bsdsocket::cfg::GetIfState(ifName, &ifState));
    NN_SDK_LOG("GetIfState(%s):\n", ifName);
    NN_SDK_LOG("  mode          = %s\n",
               (ifState.mode == nn::bsdsocket::cfg::IfIpAddrMode_Dhcp) ? "DHCP" : "STATIC");
    NN_SDK_LOG("  metric        = %d\n", ifState.metric);
    NN_SDK_LOG("  mtu           = %d\n", ifState.mtu);
    NN_SDK_LOG("  addr          = %s\n", nn::socket::InetNtoa(ifState.addr));
    NN_SDK_LOG("  subnetMask    = %s\n", nn::socket::InetNtoa(ifState.subnetMask));
    NN_SDK_LOG("  gatewayAddr   = %s\n", nn::socket::InetNtoa(ifState.gatewayAddr));
    NN_SDK_LOG("  broadcastAddr = %s\n", nn::socket::InetNtoa(ifState.broadcastAddr));
    NN_SDK_LOG("  dnsAddrs[0]   = %s\n", nn::socket::InetNtoa(ifState.dnsAddrs[0]));
    NN_SDK_LOG("  dnsAddrs[1]   = %s\n", nn::socket::InetNtoa(ifState.dnsAddrs[1]));
    NN_SDK_LOG("  state         = %s\n", ifState.u.modeDhcp.currentState);
}

static nn::bsdsocket::cfg::IfSettings TestGetDefaultCfg(bool useDhcp)
{
    nn::bsdsocket::cfg::IfSettings bsdCfg;
    memset(&bsdCfg, 0, sizeof(bsdCfg));
    // Pick address configuration
    bsdCfg.mtu = 1500;
    if (useDhcp)
    {
        bsdCfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
    }
    else
    {
        bsdCfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
        nn::socket::InetAton("10.254.239.11", &bsdCfg.u.modeStatic.addr);
        nn::socket::InetAton("10.254.239.1", &bsdCfg.u.modeStatic.gatewayAddr);
        nn::socket::InetAton("255.255.255.224", &bsdCfg.u.modeStatic.subnetMask);
        bsdCfg.u.modeStatic.broadcastAddr.S_addr =
            (bsdCfg.u.modeStatic.addr.S_addr & bsdCfg.u.modeStatic.subnetMask.S_addr) |
                ~bsdCfg.u.modeStatic.subnetMask.S_addr;
    }
    return bsdCfg;
}

static nn::Result TestSetIfUp(const char *ifName, void *arg0)
{
    nn::bsdsocket::cfg::IfSettings *pCfg =
        reinterpret_cast<nn::bsdsocket::cfg::IfSettings *>(arg0);
    return nn::bsdsocket::cfg::SetIfUp(ifName, pCfg);
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, BasicCycle)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(true);

    nn::bsdsocket::cfg::SetIfDown(s_IfName);

    for (int i = 0; (i < 100) && result.IsSuccess(); i++)
    {
        // try diferent metrics for fun
        bsdCfg.metric = i;
        result = nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg);
        if (result.IsSuccess())
        {
            ShowIfState(s_IfName);
            result = nn::bsdsocket::cfg::SetIfDown(s_IfName);
            if (!result.IsSuccess())
            {
                NN_SDK_LOG("Call to SetIfDown() failed\n");
            }
        }
        else
        {
            NN_SDK_LOG("Call to SetIfUp() failed\n");
        }
    }
    result.IsSuccess() ? SUCCEED() : ADD_FAILURE();
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, BadIfName)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(true);
    nn::bsdsocket::cfg::SetIfDown(s_IfName);

    // Try to bring up with bad name
    result = nn::bsdsocket::cfg::SetIfUp("eth1", &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultIfInvalid()))
    {
        NN_SDK_LOG("BadIfName test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }

    // Try to bring up with another name
    result = nn::bsdsocket::cfg::SetIfUp("Eth0", &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultIfInvalid()))
    {
        NN_SDK_LOG("BadIfName test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }

    // Try to bring down a bad name
    result = nn::bsdsocket::cfg::SetIfDown("Eth0");
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultIfInvalid()))
    {
        NN_SDK_LOG("BadIfName test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, AlreadyUp)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(true);
    nn::bsdsocket::cfg::SetIfDown(s_IfName);
    NNT_EXPECT_RESULT_SUCCESS(nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg));

    // Try to bring it up again, when already up
    result = nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultBusy()))
    {
        NN_SDK_LOG("AlreadyUp test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }

    // Try to cancel when SetIfUp() clearly no longer in progress
    result = nn::bsdsocket::cfg::CancelIf(s_IfName);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultIfIsUp()))
    {
        NN_SDK_LOG("AlreadyUp test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }

}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, DhcpBadMTU)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(true);
    bsdCfg.mtu = 65536;
    nn::bsdsocket::cfg::SetIfDown(s_IfName);
    result = nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultMtuSizeError()))
    {
        NN_SDK_LOG("DhcpBadMTU test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, StaticIpBadMTU)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(false);
    bsdCfg.mtu = 65536;
    nn::bsdsocket::cfg::SetIfDown(s_IfName);
    result = nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultMtuSizeError()))
    {
        NN_SDK_LOG("StaticIpBadMTU test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, BadStaticIpRoute)
{
    nn::Result result = nn::ResultSuccess();
    nn::bsdsocket::cfg::IfSettings bsdCfg = TestGetDefaultCfg(false);
    nn::socket::InetAton("0.0.0.0", &bsdCfg.u.modeStatic.subnetMask);
    nn::bsdsocket::cfg::SetIfDown(s_IfName);
    result = nn::bsdsocket::cfg::SetIfUp(s_IfName, &bsdCfg);
    if (!COMPARE_RESULT(result, nn::bsdsocket::ResultRouteError()))
    {
        NN_SDK_LOG("BadStaticIpRoute test encounted unexpected status of %d\n", result);
        ADD_FAILURE();
    }
}

//-----------------------------------------------------------------------------
TEST(BsdsocketCfg, InterruptedCycle)
{
    s_DefaultCfg = TestGetDefaultCfg(true);
    TestClientThread *pClientA = s_ClientThreads + 0;
    int dwellPeriodInMs = 1;

    nn::bsdsocket::cfg::SetIfDown(s_IfName);

    for (int i = 0; i < 6; i++)
    {
        nn::Result upResult;

        NN_SDK_LOG("########## Cycle %d, dwell %d ms ############\n",
                   i, dwellPeriodInMs);

        // Initiate process of setting UP in different thread
        pClientA->Initialize(TestSetIfUp, s_IfName, &s_DefaultCfg);

        // Dwell for variable time
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(dwellPeriodInMs));
        dwellPeriodInMs = dwellPeriodInMs * 7;

        // Try to cancel
        nn::Result cancelResult = nn::bsdsocket::cfg::CancelIf(s_IfName);

        // Get SetIfUp() completion status
        if ((upResult = pClientA->Finalize()).IsSuccess())
        {
            // SetIfUp() succeeded, therefore we must bring it down now
            NN_SDK_LOG("Call to SetIfUp() completed, so now bring it down.\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::bsdsocket::cfg::SetIfDown(s_IfName));
        }
        else
        {
            // SetIfUp() failed, and for test to pass it must be because it was cancelled
            if (COMPARE_RESULT(upResult, nn::bsdsocket::ResultCancelled()) && cancelResult.IsSuccess())
            {
                NN_SDK_LOG("Call to SetIfUp() completed with ResultCancelled()\n");
            }
            else
            {
                NN_SDK_LOG("SetIfUp()=%d, CancelIf()=%d\n",
                           upResult.GetDescription(), cancelResult.GetDescription());
                ADD_FAILURE();
                break;
            }
        }
    }
}




} //cfg
} //bsdsocket
} //nnt

