﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ssl.h>
#include <nn/socket.h>

#include <Common/testInfraInfo.h>
#include <Common/testCommonUtil.h>

#include <Utils/CommandLineParser.h>

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
#define RUN_INVALID_SOCKET       // SIGLONTD-3041
#define RUN_INVALID_SOCKET_CLOSE // SIGLONTD-3046

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define RESULT_EXPECT_SUCCESS(func) \
do { \
    nn::Result expResult = func; \
    if(expResult.IsFailure()) \
    { \
        NN_LOG("Result is not ResultSuccess: (desc:%d)\n", expResult.GetDescription()); \
    } \
    EXPECT_TRUE(expResult.IsSuccess()); \
}while(NN_STATIC_CONDITION(false))

// ------------------------------------------------------------------------------------------------
// Params
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    EXPECT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    EXPECT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// Initialize necessary libraries (must be the first test case)
// ------------------------------------------------------------------------------------------------
namespace
{
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

TEST(InitializeLibrary, Success)
{
    nn::Result result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    EXPECT_TRUE(result.IsSuccess());
    RESULT_EXPECT_SUCCESS(nn::ssl::Initialize());
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// SIGLONTD-3041: NOT COMPLETED YET
//     Testing to pass invalid socket descriptor
// ----------------------------------------------------------------------------
#if defined(RUN_INVALID_SOCKET)
TEST(InvalidSocket, Success)
{
    int                 invalidSockFd = 15;
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;
    nn::Result          result;

    RESULT_EXPECT_SUCCESS(sslContext.Create(nn::ssl::Context::SslVersion_Auto));
    RESULT_EXPECT_SUCCESS(sslConnection.Create(&sslContext));

    // TODO: compare with expected Result: SIGLONTD-3041
    result = sslConnection.SetSocketDescriptor(invalidSockFd);
    NN_LOG("Got result (desc:%d)\n", result.GetDescription());

    RESULT_EXPECT_SUCCESS(sslConnection.Destroy());
    RESULT_EXPECT_SUCCESS(sslContext.Destroy());
}
#endif // RUN_INVALID_SOCKET

// ----------------------------------------------------------------------------
// SIGLONTD-3046:
//     Testing that closing the socket outside the SSL library doesn't cause
//     catastrophic failure.
// ----------------------------------------------------------------------------
#if defined(RUN_INVALID_SOCKET_CLOSE)
TEST(InvalidSocketClose, Success)
{
    SslTestCommonUtil   commonUtil;
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;

    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(commonUtil.SetupNetwork(netProfile));

    int sockFd = commonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    ASSERT_TRUE(sockFd >= 0);

    RESULT_EXPECT_SUCCESS(sslContext.Create(nn::ssl::Context::SslVersion_Auto));
    RESULT_EXPECT_SUCCESS(sslConnection.Create(&sslContext));

    RESULT_EXPECT_SUCCESS(sslConnection.SetSocketDescriptor(sockFd));

    int rval = nn::socket::Close(sockFd);
    EXPECT_TRUE(rval == 0);

    RESULT_EXPECT_SUCCESS(sslConnection.Destroy());
    RESULT_EXPECT_SUCCESS(sslContext.Destroy());

    commonUtil.FinalizeNetwork();
}
#endif // RUN_INVALID_SOCKET_CLOSE

// ------------------------------------------------------------------------------------------------
// Finalize libraries (must be the last test case)
// ------------------------------------------------------------------------------------------------
TEST(FinalizeLibrary, Success)
{
    RESULT_EXPECT_SUCCESS(nn::ssl::Finalize());
    RESULT_EXPECT_SUCCESS(nn::socket::Finalize());
}

