﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>
#include <nnt.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

#include <Utils/CommandLineParser.h>

//-------------------------------------------------------------------------------------------------
//  Build flags
//-------------------------------------------------------------------------------------------------

namespace
{
//-------------------------------------------------------------------------------------------------
//  Params
//-------------------------------------------------------------------------------------------------
const int                TestLoopCount = 5;
const int                PollReadMaxRetry = 10;
const int                PollWriteMaxRetry = 100;
const char*              DefaultHost = "natf.com";

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

nn::util::Uuid           g_NetProfile = nn::util::InvalidUuid;
int                      g_SendBuffSize = 1024;
int                      g_RecvBuffSize = 8192;
char                     g_HostString[128] = {0};
uint32_t                 g_DataSize     = 65536;
//size_t                   g_DataSize     = 65536;
int                      g_PortNumber   = 443;

//-------------------------------------------------------------------------------------------------
//  PostData
//-------------------------------------------------------------------------------------------------
class PostData
{
    char*  pData;
    char*  pDataItr;
    size_t leftBytes;
public:
    PostData() : pData(nullptr), pDataItr(nullptr), leftBytes(0){}
    ~PostData(){}

    size_t GetLeftSize() {return leftBytes;}
    char*  GetDataPtr()  {return pData;}
    char*  GetDataItr()  {return pDataItr;}
    void   UpdateDataItr(size_t updateSize)
    {
        pDataItr  += updateSize;
        leftBytes -= updateSize;
    }

    bool Initialize(size_t dataSize)
    {
        pData = new char[dataSize];
        if (pData)
        {
            for (uint64_t i = 0; i < dataSize - 2; i++)
            {
                pData[i] = '0' + i % 10;
            }
            pData[dataSize - 1] = '\0';
            pDataItr = pData;
            leftBytes = dataSize;
        }

        return (pData)?true:false;
    }

    void Finalize()
    {
        if (pData)
        {
            delete pData;
        }
        pData    = nullptr;
        pDataItr = nullptr;
    }
};

int CreateTcpSocket(
    bool bEstablishConn,
    uint16_t portNumber,
    const char* pInHostName,
    uint32_t ipAddress)
{
    int                tcpSocket;
    nn::socket::InAddr     inetAddr;
    nn::socket::SockAddrIn serverAddr;

    memset(&inetAddr, 0x00, sizeof(inetAddr));

    if(pInHostName == nullptr && ipAddress == 0)
    {
        NN_LOG("Host name or IP address was not passed.\n");
        return -1;
    }

    tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if(tcpSocket < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    NN_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    do
    {
        if (g_RecvBuffSize > 0)
        {
            int buffSize = g_RecvBuffSize;
            nn::socket::SockLenT optLen = sizeof(buffSize);
            int rval = nn::socket::SetSockOpt(tcpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &buffSize, optLen);
            if(rval < 0)
            {
                NN_LOG("SetSockOpt - nn::socket::Option::So_RcvBuf failed (err:%d)\n", nn::socket::GetLastError());
                break;
            }

            rval = nn::socket::GetSockOpt(tcpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &buffSize, &optLen);
            if(rval < 0)
            {
                NN_LOG("GetSockOpt - nn::socket::Option::So_RcvBuf failed (err:%d)\n", nn::socket::GetLastError());
                break;
            }
            NN_LOG("Socket recv buff size:%d\n", buffSize);
        }

        if (g_SendBuffSize > 0)
        {
            int buffSize = g_SendBuffSize;
            nn::socket::SockLenT optLen = sizeof(buffSize);
            int rval = nn::socket::SetSockOpt(tcpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, optLen);
            if(rval < 0)
            {
                NN_LOG("SetSockOpt - nn::socket::Option::So_SndBuf failed (err:%d)\n", nn::socket::GetLastError());
                break;
            }

            rval = nn::socket::GetSockOpt(tcpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &buffSize, &optLen);
            if(rval < 0)
            {
                NN_LOG("GetSockOpt - nn::socket::Option::So_SndBuf failed (err:%d)\n", nn::socket::GetLastError());
                break;
            }
            NN_LOG("Socket send buff size:%d\n", buffSize);
        }

        if (bEstablishConn != true)
        {
            return tcpSocket;
        }

        if(ipAddress == 0)
        {
            NN_LOG("Resolving %s\n", pInHostName);
            nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(pInHostName);
            if(pHostEnt == nullptr)
            {
                NN_LOG("Failed to resolve host name (errno:%d)\n", errno);
                break;
            }

            // Just pick the first one
            memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(nn::socket::InAddr));
            serverAddr.sin_addr.S_addr = inetAddr.S_addr;
        }
        else
        {
            NN_LOG("Use 0x%x for server IP.\n", ipAddress);
            serverAddr.sin_addr.S_addr = ipAddress;
        }

        serverAddr.sin_family = nn::socket::Family::Af_Inet;
        serverAddr.sin_port   = nn::socket::InetHtons(portNumber);
        int rval = nn::socket::Connect(tcpSocket, (nn::socket::SockAddr*)&serverAddr, sizeof(serverAddr));
        if(rval < 0)
        {
            NN_LOG("Failed to create TCP socket (errno: %d).\n", errno);
            break;
        }

        NN_LOG("Established TCP connection (addr:0x%x on port :%d).\n",
             serverAddr.sin_addr.S_addr, nn::socket::InetNtohs(serverAddr.sin_port));
        return tcpSocket;
    } while (NN_STATIC_CONDITION(false));

    nn::socket::Close(tcpSocket);
    return -1;
}

bool SendHttpGet(nn::ssl::Connection* pInConnection, const char* pInHostName)
{
    char httpReqBuff[128] = {0};
    sprintf(httpReqBuff, "GET /kb32 HTTP/1.0\r\nHost: %s\r\n\r\n", pInHostName);

    int sentBytes     = 0;
    nn::Result result = pInConnection->Write(httpReqBuff, &sentBytes, strlen(httpReqBuff));
    if(result.IsFailure())
    {
        NN_LOG("Failed to write data. (Description:%d)\n", result.GetDescription());
    }
    else
    {
        NN_LOG("Sent HTTP request via SSL (%d bytes).\n", sentBytes);
    }

    return (result.IsSuccess())?(true):(false);
}

bool SendHttpPost(nn::ssl::Connection* pInConnection, const char* pInHostName, size_t dataSize)
{
    char httpReqBuff[256] = {0};
#if defined (NN_BUILD_CONFIG_OS_WIN)
    sprintf(httpReqBuff, "POST /d4,1d,8c,d9,8f,00,b2,04,e9,80,09,98,ec,f8,42,7e HTTP/1.1\r\nHost: %s\r\nAccept: */*\r\nContent-Length: %lu\r\nContent-Type: application/x-www-form-urlencoded\r\n\r\n",
            pInHostName, (unsigned long)dataSize);
#else
    sprintf(httpReqBuff, "POST /d4,1d,8c,d9,8f,00,b2,04,e9,80,09,98,ec,f8,42,7e HTTP/1.1\r\nHost: %s\r\nAccept: */*\r\nContent-Length: %zu\r\nContent-Type: application/x-www-form-urlencoded\r\n\r\n",
            pInHostName, dataSize);
#endif
    int sentBytes     = 0;
    nn::Result result = pInConnection->Write(httpReqBuff, &sentBytes, strlen(httpReqBuff));
    if(result.IsFailure())
    {
        NN_LOG("Failed to write data. (Description:%d)\n", result.GetDescription());
    }
    else
    {
        NN_LOG("Sent HTTP request via SSL (%d bytes).\n", sentBytes);
    }

    return (result.IsSuccess())?(true):(false);
}

bool WaitWithPoll(nn::ssl::Connection* pInConnection, nn::ssl::Connection::PollEvent inPollEvent, int maxRetry)
{
    uint32_t msecPollTimeout = 100;
    nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    pollEvent |= inPollEvent;

    nn::Result result;
    bool isSuccess = true;
    do
    {
        if(maxRetry-- == 0)
        {
            NN_LOG("Finished polling for max retry.\n");
            isSuccess = false;
            break;
        }

        result = pInConnection->Poll(&pollOutEvent, &pollEvent, msecPollTimeout);
        if(result.IsFailure())
        {
            if(nn::ssl::ResultIoTimeout::Includes(result))
            {
                continue;
            }
            else
            {
                NN_LOG("Failed due to Poll error (Description:%d)\n", result.GetDescription());
                isSuccess = false;
                break;
            }
        }

        if((pollOutEvent & pollEvent) != pollEvent)
        {
            NN_LOG("Poll succeeded but somehow specified event is not ready (pollEvent:0x%x pollOutEvent:0x%x).\n",
                pollEvent, pollOutEvent);
            continue;
        }
        else
        {
            break; // There's an event to process
        }
    } while (NN_STATIC_CONDITION(true));

    return isSuccess;
}
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser   ("--NetProfile", &nn::util::InvalidUuid, g_NetProfile));
    parser.AddParser(NATF::Utils::StringParser ("--HostName", DefaultHost, g_HostString, sizeof(g_HostString)));
    parser.AddParser(NATF::Utils::Int32Parser  ("--Port", &g_PortNumber, g_PortNumber));
    parser.AddParser(NATF::Utils::Int32Parser  ("--RecvBufLen", &g_RecvBuffSize, g_RecvBuffSize));
    parser.AddParser(NATF::Utils::Int32Parser  ("--SendBufLen", &g_SendBuffSize, g_SendBuffSize));
    parser.AddParser(NATF::Utils::UInt32Parser ("--DataLen", &g_DataSize, g_DataSize));

    int    argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(g_NetProfile));
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
}

TEST(SslGetBlocking, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_Blocking);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.DoHandshake();
        ASSERT_TRUE(result.IsSuccess());
        ASSERT_TRUE(SendHttpGet(&sslConnection, g_HostString));

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            EXPECT_TRUE(result.IsSuccess());
            if(result.IsFailure())
            {
                NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
                break;
            }
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
}

TEST(SslGetBlockingWithPoll, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_Blocking);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.DoHandshake();
        ASSERT_TRUE(result.IsSuccess());
        ASSERT_TRUE(SendHttpGet(&sslConnection, g_HostString));

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            ASSERT_TRUE(WaitWithPoll(
                &sslConnection,
                nn::ssl::Connection::PollEvent::PollEvent_Read,
                PollReadMaxRetry));

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            if(result.IsFailure())
            {
                NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
                break;
            }
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
}

TEST(SslGetNonBlocking, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        ASSERT_TRUE(result.IsSuccess());

        uint32_t maxAsyncDoHandshakeRetry = 30;
        do
        {
            result = sslConnection.DoHandshake();
            if(result.IsSuccess())
            {
                NN_LOG("SSL Handshake completed.\n");
                break;
            }
            else if(nn::ssl::ResultIoWouldBlock::Includes(result))
            {
                if(--maxAsyncDoHandshakeRetry > 0)
                {
                    NN_LOG("Handshake would block, try again after 100 msec (retry remain:%d)\n",
                        maxAsyncDoHandshakeRetry);
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LOG("Handshake failed for max retry.\n");
                    break;
                }
            }
            else
            {
                NN_LOG("SSL Handshake failed (Description:%d)\n", result.GetDescription());
                if(nn::ssl::ResultVerifyCertFailed::Includes(result))
                {
                    nn::Result verifyCertError;
                    result = sslConnection.GetVerifyCertError(&verifyCertError);
                    if(result.IsFailure())
                    {
                        NN_LOG("Failed to get verify cert error (Description:%d)\n",
                            result.GetDescription());
                    }
                    else
                    {
                        NN_LOG("Certificate validation failed (Description:%d)\n",
                            verifyCertError.GetDescription());
                        result = verifyCertError;
                    }
                }
                break;
            }
        }
        while (NN_STATIC_CONDITION(true));
        ASSERT_TRUE(result.IsSuccess());

        ASSERT_TRUE(SendHttpGet(&sslConnection, g_HostString));

        int receivedTotalBytes = 0;
        int readRetryMax = 10;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            if(result.IsFailure())
            {
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    if (--readRetryMax == 0)
                    {
                        NN_LOG("Finished receiving data for max retry.\n");
                        EXPECT_TRUE(NN_STATIC_CONDITION(false));
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
                    break;
                }
            }
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
} // NOLINT(impl/function_size)

TEST(SslGetNonBlockingWithPoll, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        ASSERT_TRUE(result.IsSuccess());

        uint32_t maxAsyncDoHandshakeRetry = 30;
        do
        {
            result = sslConnection.DoHandshake();
            if(result.IsSuccess())
            {
                NN_LOG("SSL Handshake completed.\n");
                break;
            }
            else if(nn::ssl::ResultIoWouldBlock::Includes(result))
            {
                if(--maxAsyncDoHandshakeRetry > 0)
                {
                    NN_LOG("Handshake would block, try again after 100 msec (retry remain:%d)\n",
                        maxAsyncDoHandshakeRetry);
                    ASSERT_TRUE(WaitWithPoll(
                        &sslConnection,
                        nn::ssl::Connection::PollEvent::PollEvent_Read,
                        PollReadMaxRetry));
                    continue;
                }
                else
                {
                    NN_LOG("Handshake failed for max retry.\n");
                    break;
                }
            }
            else
            {
                NN_LOG("SSL Handshake failed (Description:%d)\n", result.GetDescription());
                if(nn::ssl::ResultVerifyCertFailed::Includes(result))
                {
                    nn::Result verifyCertError;
                    result = sslConnection.GetVerifyCertError(&verifyCertError);
                    if(result.IsFailure())
                    {
                        NN_LOG("Failed to get verify cert error (Description:%d)\n",
                            result.GetDescription());
                    }
                    else
                    {
                        NN_LOG("Certificate validation failed (Description:%d)\n",
                            verifyCertError.GetDescription());
                        result = verifyCertError;
                    }
                }
                break;
            }
        }
        while (NN_STATIC_CONDITION(true));
        ASSERT_TRUE(result.IsSuccess());

        ASSERT_TRUE(SendHttpGet(&sslConnection, g_HostString));

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            ASSERT_TRUE(WaitWithPoll(
                &sslConnection,
                nn::ssl::Connection::PollEvent::PollEvent_Read,
                PollReadMaxRetry));

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            NN_STATIC_CONDITION(result.IsSuccess());
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
} // NOLINT(impl/function_size)

TEST(SslPostBlocking, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_Blocking);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.DoHandshake();
        ASSERT_TRUE(result.IsSuccess());

        ASSERT_TRUE(SendHttpPost(&sslConnection, g_HostString, g_DataSize));

        PostData postData;
        ASSERT_TRUE(postData.Initialize(g_DataSize));

        NN_LOG("Size of data to POST = %d bytes \n", postData.GetLeftSize());
        do
        {
            if(postData.GetLeftSize() == 0)
            {
                NN_LOG("Sent all the data.\n");
                break;
            }

            int sentBytes = 0;
            NN_LOG("Calling write: size:%d \n", postData.GetLeftSize());
            result = sslConnection.Write(postData.GetDataItr(), &sentBytes, postData.GetLeftSize());
            EXPECT_TRUE(result.IsSuccess());
            if(result.IsFailure())
            {
                NN_LOG("Failed to write data (Description:%d)\n", result.GetDescription());
                break;
            }
            else
            {
                postData.UpdateDataItr(sentBytes);
                NN_LOG("Sent %d bytes (left:%d bytes)\n", sentBytes, postData.GetLeftSize());
            }
        } while(NN_STATIC_CONDITION(true));
        postData.Finalize();

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            if(WaitWithPoll(&sslConnection, nn::ssl::Connection::PollEvent::PollEvent_Read, PollReadMaxRetry) == false)
            {
                NN_LOG("Finished reading for retry max.\n");
                break;
            }

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            EXPECT_TRUE(result.IsSuccess());
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
}

TEST(SslPostNonBlocking, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.DoHandshake();
        ASSERT_TRUE(result.IsSuccess());

        ASSERT_TRUE(SendHttpPost(&sslConnection, g_HostString, g_DataSize));

        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        ASSERT_TRUE(result.IsSuccess());

        PostData postData;
        ASSERT_TRUE(postData.Initialize(g_DataSize));
        NN_LOG("Size of data to POST = %d bytes \n", postData.GetLeftSize());
        int writeRetlyMax = 200;
        do
        {
            if(postData.GetLeftSize() == 0)
            {
                NN_LOG("Sent all the data.\n");
                break;
            }

            int sentBytes = 0;
            NN_LOG("Calling write: size:%d \n", postData.GetLeftSize());
            result = sslConnection.Write(postData.GetDataItr(), &sentBytes, postData.GetLeftSize());
            if(result.IsFailure())
            {
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    if(--writeRetlyMax == 0)
                    {
                        NN_LOG("Write failed for max retry.\n");
                        break;
                    }
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LOG("Failed to write data (Description:%d)\n", result.GetDescription());
                    break;
                }
            }
            else
            {
                postData.UpdateDataItr(sentBytes);
                NN_LOG("Sent %d bytes (left:%d bytes)\n", sentBytes, postData.GetLeftSize());
            }
        } while(NN_STATIC_CONDITION(true));
        postData.Finalize();

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            if(WaitWithPoll(&sslConnection, nn::ssl::Connection::PollEvent::PollEvent_Read, PollReadMaxRetry) == false)
            {
                NN_LOG("Finished reading for retry max.\n");
                break;
            }

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            EXPECT_TRUE(result.IsSuccess());
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
} // NOLINT(impl/function_size)

TEST(SslPostNonBlockingWithPoll, Success)
{
    for(int i = 0; i < TestLoopCount; i++) {
        int socketFd = CreateTcpSocket(true, static_cast<uint16_t>(g_PortNumber), g_HostString, 0);
        ASSERT_TRUE(socketFd >= 0);

        nn::Result result;
        nn::ssl::Context sslContext;
        result = sslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
        ASSERT_TRUE(result.IsSuccess());

        nn::ssl::Connection sslConnection;
        result = sslConnection.Create(&sslContext);
        ASSERT_TRUE(result.IsSuccess());
        EXPECT_TRUE(sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.SetSocketDescriptor(socketFd);
        ASSERT_TRUE(result.IsSuccess());
        result = sslConnection.DoHandshake();
        ASSERT_TRUE(result.IsSuccess());

        ASSERT_TRUE(SendHttpPost(&sslConnection, g_HostString, g_DataSize));

        result = sslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
        ASSERT_TRUE(result.IsSuccess());

        PostData postData;
        ASSERT_TRUE(postData.Initialize(g_DataSize));
        NN_LOG("Size of data to POST = %d bytes \n", postData.GetLeftSize());
        do
        {
            if(postData.GetLeftSize() == 0)
            {
                NN_LOG("Sent all the data.\n");
                break;
            }

            ASSERT_TRUE(WaitWithPoll(
                &sslConnection,
                nn::ssl::Connection::PollEvent::PollEvent_Write,
                PollWriteMaxRetry));

            int sentBytes = 0;
            NN_LOG("Calling write: size:%d \n", postData.GetLeftSize());
            result = sslConnection.Write(postData.GetDataItr(), &sentBytes, postData.GetLeftSize());
            if(result.IsFailure())
            {
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    // This is possible because SSL write is different from TCP send hence
                    // SSL layer may return WouldBlock error even when there's a space in
                    // socket send buffer which is what nn::ssl::Connection::Poll checks.
                    NN_LOG("Write ResultIoWouldBlock.\n");
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                    continue;
                }
                else
                {
                    NN_LOG("Write failed (desc:%d)\n", result.GetDescription());
                    EXPECT_TRUE(NN_STATIC_CONDITION(false));
                    break;
                }
            }
            EXPECT_TRUE(result.IsSuccess());

            postData.UpdateDataItr(sentBytes);
            NN_LOG("Sent %d bytes (left:%d bytes)\n", sentBytes, postData.GetLeftSize());
        } while(NN_STATIC_CONDITION(true));
        postData.Finalize();

        int receivedTotalBytes = 0;
        do
        {
            char readBuff[1024 * 4];
            int receivedBytes  = 0;

            if(WaitWithPoll(&sslConnection, nn::ssl::Connection::PollEvent::PollEvent_Read, PollReadMaxRetry) == false)
            {
                NN_LOG("Finished reading for retry max.\n");
                break;
            }

            result = sslConnection.Read(readBuff, &receivedBytes, sizeof(readBuff));
            EXPECT_TRUE(result.IsSuccess());
            if(receivedBytes == 0)
            {
                NN_LOG("Connection closed by the server.\n");
                break;
            }

            receivedTotalBytes += receivedBytes;
            NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, receivedTotalBytes);
        } while(NN_STATIC_CONDITION(true));

        result = sslConnection.Destroy();
        ASSERT_TRUE(result.IsSuccess());
        result = sslContext.Destroy();
        ASSERT_TRUE(result.IsSuccess());
    }
}

TEST(FinalizeTest, Success)
{
    NN_LOG("Test parameters\nHost:%s\nport:%d\nRCVBUFF:%d\nSNDBUFF:%d\nData size:%d\n\n",
           g_HostString, g_PortNumber, g_RecvBuffSize, g_SendBuffSize, g_DataSize);

    nn::ssl::Finalize();
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
