﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/nifm.h>

#include <nn/ssl.h>
#include <nn/socket.h>

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
#define RUN_INVALID_LIB_REF_COUNT_SUCCESS // Don't disable this on the tree, needs to be tested manually
#define RUN_INIT_FINALIZE_TWICE
#define RUN_INIT_FINALIZE_MULTIPLE_TIMES  // Does Initialize/Finalize multple times

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define RESULT_EXPECT_SUCCESS(func) \
do { \
    nn::Result expResult = func; \
    if(expResult.IsFailure()) \
    { \
        NN_LOG("Result is not ResultSuccess: (desc:%d)\n", expResult.GetDescription()); \
    } \
    EXPECT_TRUE(expResult.IsSuccess()); \
}while(NN_STATIC_CONDITION(false))

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// SIGLONTD-3305: Other test cases defined in "Build flags" on top of this file
//                need to be disabled to test this. This needs to be tested
//                manually because pefroming this will cause SDK assert.
//                Disable #define RUN_INVALID_LIB_REF_COUNT_SUCCESS to test
//                negative test case
// ----------------------------------------------------------------------------
#ifdef RUN_INVALID_LIB_REF_COUNT_SUCCESS
TEST(InvalidLibReferenceCountShouldSuccess, Success)
{
    nn::Result result;

    result = nn::ssl::Finalize();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    result = nn::ssl::Finalize();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    // 1st Initialize()
    RESULT_EXPECT_SUCCESS(nn::ssl::Initialize());

    // Call Initialize() 99 times
    for (int i = 0; i < 99; i++)
    {
        result = nn::ssl::Initialize();
        EXPECT_TRUE(nn::ssl::ResultLibraryAlreadyInitialized::Includes(result));
    }

    // Call Finalize() 99 times
    for (int i = 0; i < 99; i++)
    {
        RESULT_EXPECT_SUCCESS(nn::ssl::Finalize());
    }

    // Final Finalize()
    RESULT_EXPECT_SUCCESS(nn::ssl::Finalize());

    // This test success without SDK assert
}
#else
TEST(InvalidLibReferenceCountAssertCaused, Fail)
{
    nn::Result result;

    result = nn::ssl::Finalize();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    result = nn::ssl::Finalize();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    // 1st Initialize()
    RESULT_EXPECT_SUCCESS(nn::ssl::Initialize());

    // Call Initialize() 99 times
    for (int i = 0; i < 99; i++)
    {
        result = nn::ssl::Initialize();
        EXPECT_TRUE(nn::ssl::ResultLibraryAlreadyInitialized::Includes(result));
    }

    // Call Finalize() 95 times
    for (int i = 0; i < 95; i++)
    {
        RESULT_EXPECT_SUCCESS(nn::ssl::Finalize());
    }

    // This test must not succeed, instead SDK assert needs to be caused
    // This would happen only on Windows because the mechanism usde in the SSL
    // library wouuldn't work on Horizon
}
#endif // RUN_INVALID_LIB_REF_COUNT_SUCCESS

#if defined(RUN_INIT_FINALIZE_TWICE)
TEST(ShimInitFinalizeTwice, Success)
{
    nn::Result result = nn::ssl::Initialize();
    EXPECT_TRUE(result.IsSuccess());
    result = nn::ssl::Finalize();
    EXPECT_TRUE(result.IsSuccess());
    result = nn::ssl::Initialize();
    EXPECT_TRUE(result.IsSuccess());
    result = nn::ssl::Finalize();
    EXPECT_TRUE(result.IsSuccess());
}
#endif // RUN_INIT_FINALIZE_TWICE

#if defined(RUN_INIT_FINALIZE_MULTIPLE_TIMES)
TEST(ShimInitFinalizeMultipleTimes, Success)
{
    int i = 0;
    while(++i < 128)
    {
        //NN_LOG(" count:%d\n", i);
        nn::Result result = nn::ssl::Initialize();
        EXPECT_TRUE(result.IsSuccess());
        result = nn::ssl::Finalize();
        EXPECT_TRUE(result.IsSuccess());
    }
    NN_LOG(" nn::ssl library Initialize/Finalize done %d times\n", i);
}
#endif // RUN_INIT_FINALIZE_MULTIPLE_TIMES
