﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

#include <Utils/CommandLineParser.h>

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

namespace
{
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

size_t CurlSslContextCallback(CURL* pCurl, void* pSslContext, void* pUserData)
{
    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

    // Create SSL context
    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        return (size_t) - 1;
    }

    if (pUserData != nullptr)
    {
        nn::ssl::CertStoreId certStoreId;
        SimpleCurlHttpsClient* pClient = reinterpret_cast<SimpleCurlHttpsClient*>(pUserData);

        if(pClient->IsImportServerPki())
        {
            result = pContext->ImportServerPki(
                &certStoreId,
                pClient->GetServerCert(),
                pClient->GetServerCertSize(),
                nn::ssl::CertificateFormat_Pem);
            if (result.IsSuccess())
            {
                pClient->SetServerCertStoreId(certStoreId);
                NN_LOG("Server cert imported (certstore id:%d)\n", certStoreId);
            }
            else
            {
                NN_LOG("Importing server cert failed (desc:%d)\n", result.GetDescription());
                return (size_t) - 1;
            }
        }

        certStoreId = 0;
        if(pClient->IsImportClientPki())
        {
            result = pContext->ImportClientPki(
                &certStoreId,
                pClient->GetClientPki(),
                pClient->GetPassword(),
                pClient->GetClientPkiSize(),
                pClient->GetPasswordSize());
            if (result.IsSuccess())
            {
                NN_LOG("Client cert/key imported (certstore id:%d)\n", certStoreId);
                pClient->SetClientCertStoreId(certStoreId);
            }
            else
            {
                NN_LOG("Importing client cert failed (desc:%d)\n", result.GetDescription());
                return (size_t) - 1;
            }
        }
    }

    return 0;
}

} // Un-named namespace

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    // Though Libcurl initializes the SSL library, initialize it here to test
    // the reference counter in the SSL library
    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
    ASSERT_TRUE(curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK);
}

TEST(ExternalCommercial, Success)
{
    SimpleCurlHttpsClient curlSimple(
        "https://www.google.com",
        nullptr, // CA certificate
        nullptr, // client certificate
        nullptr, // password
        0,       // CA certificate buffer size
        0,       // client certificate buffer size
        0,       // password buffer size
        SimpleCurlHttpsClient::InitMode_Manual);
    curlSimple.SetCurlCtxFunction(CurlSslContextCallback);

    curlSimple.Perform();
}

TEST(InternalNormal, Success)
{
    SimpleCurlHttpsClient curlSimple(
        "https://natf.com",
        g_pTestCaCert,
        nullptr,
        nullptr,
        sizeof(g_pTestCaCert),
        0,
        0,
        SimpleCurlHttpsClient::InitMode_Manual);
    curlSimple.SetCurlCtxFunction(CurlSslContextCallback);

    curlSimple.Perform();
}

TEST(InternalClientAuth, Success)
{
    SimpleCurlHttpsClient curlSimple(
        "https://natf.com:444",
        g_pTestCaCert,
        reinterpret_cast<const char*>(g_pTestClientPki),
        g_pTestClientPkiPassword,
        sizeof(g_pTestCaCert),
        g_pTestClientPkiSize,
        g_TestClientPkiPasswordLength,
        SimpleCurlHttpsClient::InitMode_Manual);
    curlSimple.SetCurlCtxFunction(CurlSslContextCallback);

    curlSimple.Perform();
}

TEST(FinalizeTest, Success)
{
    curl_global_cleanup();
    // Though Libcurl finalizes the SSL library, finalize it here to test
    // the reference counter in the SSL library
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
