﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/ssl.h>
#include <nn/socket.h>

#include <Common/testCommonUtil.h>
#include <Utils/CommandLineParser.h>

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
namespace
{
    const char* const TestServerHostname = "www.google.com";
    const uint16_t    TestServerPort     = 443;
    const uint32_t    ServerCertCount    = 3;

    SslTestCommonUtil        g_CommonUtil;
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    void DoHandshakeWithCertChain(nn::ssl::Context* pSslContext, nn::ssl::Connection::SessionCacheMode sessionCacheMode, uint32_t expectedCertCount)
    {
        nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();
        nn::Result result = pSslConnection->Create(pSslContext);
        EXPECT_TRUE(result.IsSuccess());

        int socketFd = g_CommonUtil.CreateTcpSocket(
            true,
            TestServerPort,
            TestServerHostname,
            0
        );

        result = pSslConnection->SetSocketDescriptor(socketFd);
        EXPECT_TRUE(result.IsSuccess());

        result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
        EXPECT_TRUE(result.IsSuccess());

        result = pSslConnection->SetSessionCacheMode(sessionCacheMode);
        EXPECT_TRUE(result.IsSuccess());
        result = pSslConnection->SetHostName(TestServerHostname, strlen(TestServerHostname));
        EXPECT_TRUE(result.IsSuccess());
        result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_GetServerCertChain, true);
        EXPECT_TRUE(result.IsSuccess());

        char buf[4096];
        uint32_t certSize = 0;
        uint32_t certCount = 0;

        result = pSslConnection->DoHandshake(&certSize, &certCount, buf, sizeof(buf));

        NN_LOG("Cert Count: %d, Cert Chain Size: %d\n", certCount, certSize);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_TRUE(certCount == expectedCertCount);
        EXPECT_TRUE((expectedCertCount > 0) ? certSize > 0 : certSize == 0);

        for(uint32_t iCert = 0; iCert < certCount; ++iCert)
        {
            nn::ssl::Connection::ServerCertDetail detail;
            result = pSslConnection->GetServerCertDetail(&detail, buf, iCert);
            EXPECT_TRUE(result.IsSuccess());

            NN_LOG("Cert Index: %d, Cert Size: %d\n", iCert, detail.dataSize);

            EXPECT_TRUE(detail.dataSize > 0);
            EXPECT_TRUE(detail.pDerData > &buf[0]);
            EXPECT_TRUE(detail.pDerData < &buf[certSize]);
        }

        result = pSslConnection->Destroy();
        EXPECT_TRUE(result.IsSuccess());
        pSslConnection = nullptr;
    }
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    EXPECT_TRUE(nn::ssl::Initialize().IsSuccess());
}


TEST(SslInternet, GetChainWithSessionCache)
{
    nn::Result result;

    // Create and configure SSL Context
    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    // Handshake with cert chain and session ID
    DoHandshakeWithCertChain(pSslContext, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId, ServerCertCount);

    // Handshake with cert chain and session ID, this time it will use the
    // previous session cache and not get the cert chain.
    // Current implementation of nn::ssl cannot provide the cert chain when session cache is
    // used because the cert chain used to establish the SSL connection gets flushed every
    // time after handshake is completed for memory constraints.
    DoHandshakeWithCertChain(pSslContext, nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId, 0);

    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
