﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
  Test code for Performance logger
 *---------------------------------------------------------------------------*/


#if defined(_WIN32)
// make sure we test the "pure windows" version, not the Siglo for Windows
#undef NN_NINTENDO_SDK
#define NN_NOEXCEPT
#endif


#include <nn/perflog/PerfLogger.h>
#include <nnt/nntest.h>

namespace
{
    const int maxNumLogEntries = 2048;
    typedef uint64_t LogContext;
    typedef nn::perflog::PerfLogger< LogContext, maxNumLogEntries> TestLoggerType;
}


TEST(LoggerTest, getMaxNumEntries)
{
    TestLoggerType log;
    EXPECT_TRUE( maxNumLogEntries == log.GetMaxNumEntries() );
}

TEST( LoggerTest, iterateEmpty )
{
    uint32_t idx;
    uint32_t nElements;
    TestLoggerType log;
    TestLoggerType::Iterator it;

    // never add any entries then iterate
    const TestLoggerType::LogEntry* e;

    it = log.GetEntries( &nElements );

    idx = 0;
    for( it.Reset(); (e = it.Get()) != NULL; it.Next() )
    {
        // never should run
        ++idx;
    }

    EXPECT_EQ( idx, 0 );
}



TEST( LoggerTest, single_entry )
{
    uint32_t nElements;
    TestLoggerType log;

    const uint32_t expectedId = 0x87654321;
    const uint32_t expectedContext = 0xFFFFFFFF;
    log.Put( expectedId, expectedContext );

    TestLoggerType::LogEntry* entries = new TestLoggerType::LogEntry[ log.GetMaxNumEntries() ];
    nElements = log.Dump( entries, log.GetMaxNumEntries() );

    // the one I added plus the dummy header
    EXPECT_EQ( nElements, 1 );

    EXPECT_NE( log.RawTsToNanos(entries[0].timestamp), 0 );
    EXPECT_EQ( entries[0].id, expectedId );
    EXPECT_EQ( entries[0].context, expectedContext );

    delete[] entries;
}

TEST( LoggerTest, put_max )
{
    uint32_t nElements;
    TestLoggerType log;
    uint64_t firstTimeNanos = 0;
    uint64_t prevTimeNanos = 0;
    nn::perflog::OSRawTimestamp prevTimeRaw;
    TestLoggerType::OSGetRawTimestamp( prevTimeRaw );
    prevTimeNanos = log.RawTsToNanos( prevTimeRaw );

    const uint32_t numEntries = log.GetMaxNumEntries();
    uint32_t id = 1;
    uint32_t context = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < numEntries; ++idx )
    {
        log.Put( id++, context-- );
    }

    TestLoggerType::LogEntry* entries = new TestLoggerType::LogEntry[ log.GetMaxNumEntries() ];
    nElements = log.Dump( entries, log.GetMaxNumEntries() );

    EXPECT_EQ( nElements, log.GetMaxNumEntries() );

    uint32_t expectedId = 1;
    uint32_t expectedContext = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < nElements; ++idx )
    {
        uint64_t curTimeNanos = log.RawTsToNanos(entries[idx].timestamp);
        EXPECT_NE( curTimeNanos, 0 ) << "idx=" << idx;
        EXPECT_GE( curTimeNanos, prevTimeNanos ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].id, expectedId++ ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].context, expectedContext-- ) << "idx=" << idx;
        if( firstTimeNanos == 0 )
        {
            firstTimeNanos = curTimeNanos;
        }
        prevTimeNanos = curTimeNanos;
    }
    EXPECT_GT( prevTimeNanos, firstTimeNanos );


    delete[] entries;
}



// wraps around several times, and not an integer number of times
//  very similar to real life usage
TEST( LoggerTest, put_max_plus_a_lot )
{
    uint32_t nElements;
    TestLoggerType* log = TestLoggerType::GetLogger();
    ASSERT_TRUE( log != NULL );

    const uint32_t numEntries = (log->GetMaxNumEntries() * 5) + (log->GetMaxNumEntries() / 2);
    uint32_t id = 1;
    uint32_t context = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < numEntries; ++idx )
    {
        log->Put( id++, context-- );
    }

    TestLoggerType::LogEntry* entries = new TestLoggerType::LogEntry[ log->GetMaxNumEntries() ];
    nElements = log->Dump( entries, log->GetMaxNumEntries() );

    EXPECT_EQ( nElements, log->GetMaxNumEntries() );


    // rationale for this: first id is this many items ago
    uint32_t expectedId = id - log->GetMaxNumEntries();

    // keep math relationship between id and context
    uint32_t expectedContext = 0xFFFFFFFF - expectedId + 1;

    for( uint32_t idx = 0; idx < nElements; ++idx )
    {
        EXPECT_NE( log->RawTsToNanos(entries[idx].timestamp), 0 ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].id, expectedId++ ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].context, expectedContext-- ) << "idx=" << idx;
    }

    delete[] entries;
}



TEST( LoggerTest, put_max_minus_one )
{
    uint32_t nElements;

    TestLoggerType log;

    const uint32_t numEntries = log.GetMaxNumEntries() - 1;
    uint32_t id = 1;
    uint32_t context = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < numEntries; ++idx )
    {
        log.Put( id++, context-- );
    }

    TestLoggerType::LogEntry* entries = new TestLoggerType::LogEntry[ log.GetMaxNumEntries() ];
    nElements = log.Dump( entries, log.GetMaxNumEntries() );

    // rationale: logger adds a dummy entry at the top
    EXPECT_EQ( nElements, numEntries );

    uint32_t expectedId = 1;
    uint32_t expectedContext = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < nElements; ++idx )
    {
        EXPECT_NE( log.RawTsToNanos(entries[idx].timestamp), 0 ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].id, expectedId++ ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].context, expectedContext-- ) << "idx=" << idx;
    }

    delete[] entries;
}



TEST( LoggerTest, put_max_plus_one )
{
    uint32_t nElements;
    TestLoggerType log;

    const uint32_t numEntries = log.GetMaxNumEntries() + 1;
    uint32_t id = 1;
    uint32_t context = 0xFFFFFFFF;
    for( uint32_t idx = 0; idx < numEntries; ++idx )
    {
        log.Put( id++, context-- );
    }

    TestLoggerType::LogEntry* entries = new TestLoggerType::LogEntry[ log.GetMaxNumEntries() ];
    nElements = log.Dump( entries, log.GetMaxNumEntries() );

    EXPECT_EQ( nElements, log.GetMaxNumEntries() );

    uint32_t expectedId = 2; // first entry should be overwritten, start from 2nd
    uint32_t expectedContext = 0xFFFFFFFE;
    for( uint32_t idx = 0; idx < nElements; ++idx )
    {
        EXPECT_NE( log.RawTsToNanos(entries[idx].timestamp), 0 ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].id, expectedId++ ) << "idx=" << idx;
        EXPECT_EQ( entries[idx].context, expectedContext-- ) << "idx=" << idx;
    }

    delete[] entries;
}

