﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
  Test code for Performance logger
 *---------------------------------------------------------------------------*/

#include <nn/perflog/PerfLogger.h>
#include <nn/perflog/DumpThread.h>
#include <nn/perflog/Daemon.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <cstdio>
#include <cstring>


#ifdef _MSC_VER
#define __customPRIx64 "I64x"
#else
#ifdef __LP64__
#define __customPRIx64  "lx"
#else
#define __customPRIx64 "llx"
#endif
#endif

using namespace nn::perflog;

namespace
{
    const int maxNumLogEntries = 2048;
    typedef uint64_t LogContext;
    typedef nn::perflog::PerfLogger< LogContext, maxNumLogEntries> DumpTestLoggerType;

    const uint32_t expectedMainThreadLogId = 0x87654321;
    const uint32_t expectedMainThreadLogCtx = 0xFFFFFFFF;

    const uint32_t expectedBkgThreadLogId = 0x87654322;
    const uint32_t expectedBkgThreadLogCtx = 0xFFFFFFFE;

    const uint32_t expectedXtrThreadLogId = 0x87224322;
    const uint32_t expectedXtrThreadLogCtx = 0xFFFEFFFE;


    int g_CallbackCallCounter = 0;

    OSThreadId g_MainThreadId = reinterpret_cast<OSThreadId>(0);
    OSThreadId g_XtrThreadId  = reinterpret_cast<OSThreadId>(0);
    OSThreadId g_BkgThreadId  = reinterpret_cast<OSThreadId>(0);

}




static uint32_t ExtraThread( void *pvLog )
{
    DumpTestLoggerType *pLog = static_cast<DumpTestLoggerType *>(pvLog);
    g_XtrThreadId = DumpTestLoggerType::OSGetCurrentThreadId();

    NN_LOG("Xtr Thread ID=%p\n", reinterpret_cast<void*>(g_XtrThreadId) );


    pLog->Put( expectedXtrThreadLogId, expectedXtrThreadLogCtx );
    return 0;
}

static uint32_t SelfSignalThread( void *pvEvent )
{
    nn::Result nnRes;
    OSEvent *e = static_cast<OSEvent *>(pvEvent);


    NN_LOG("v2Self-signal sleeping 5s before signaling remote event\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(8000));
    nnRes = nn::perflog::TriggerDump("SFDumpTest");
    EXPECT_EQ( true, nnRes.IsSuccess() );

    NN_LOG("v2Self-signal sleeping 3s b4 attaching event locally\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));


    NN_LOG("v2Self-signal atttaching eventlocally\n" );
    nn::os::SystemEventType evt;
    nn::os::AttachWritableHandleToSystemEvent(&evt,
                                              nn::os::GetWritableHandleOfSystemEvent(e), false,
                                              nn::os::EventClearMode_AutoClear  );


    NN_LOG("v2Self-signal sleeping 4s before signaling local event\n" );
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(4000));

    nn::os::SignalSystemEvent(&evt);
    nn::os::DetachWritableHandleOfSystemEvent(&evt);


    NN_LOG("Self-signal thread exiting\n" );
    return 0;
}




static void dumplog(DumpTestLoggerType *pLog)
{
    uint32_t idx;
    uint32_t nElements;
    nn::Result nnRes;
    nn::perflog::LogDumpCookie dc;

    DumpTestLoggerType::Iterator it;

    g_BkgThreadId = DumpTestLoggerType::OSGetCurrentThreadId();
    NN_LOG("Bkg Thread ID=%p\n", reinterpret_cast<void*>(g_BkgThreadId) );


    // never add any entries then iterate
    const DumpTestLoggerType::LogEntry* e;

    if( g_CallbackCallCounter == 0 )
    {
       pLog->Put( expectedBkgThreadLogId, expectedBkgThreadLogCtx );
    }

    it = pLog->GetEntries( &nElements );

    if( g_CallbackCallCounter == 0 )
    {
        EXPECT_EQ( 3, nElements );
    }
    else
    {
        EXPECT_EQ( 0, nElements );
    }


    nnRes =  nn::perflog::BeginDump(&dc);
    EXPECT_EQ( true, nnRes.IsSuccess() );

    idx = 0;
    for( it.Reset(); (e = it.Get()) != NULL; it.Next() )
    {
        char strCtx[50];
        sprintf(strCtx, "%" __customPRIx64, e->context ); // my own conversion
        nnRes =  nn::perflog::DumpLine(
            dc,
            reinterpret_cast<uint64_t>( e->thread ),
            pLog->RawTsToNanos( e->timestamp ),
            e->id,
            strCtx );
        EXPECT_EQ( true, nnRes.IsSuccess() );


        if( idx == 0 )
        {
            NN_LOG("checking main thr entry\n");
            EXPECT_EQ( expectedMainThreadLogId , e->id );
            EXPECT_EQ( expectedMainThreadLogCtx, e->context );
            EXPECT_EQ( g_MainThreadId          , e->thread );
        }
        else if( idx == 1 )
        {
            NN_LOG("checking xtr thr entry\n");
            EXPECT_EQ( expectedXtrThreadLogId ,  e->id );
            EXPECT_EQ( expectedXtrThreadLogCtx,  e->context);
            EXPECT_EQ( g_XtrThreadId          ,  e->thread );
        }
        else if( idx == 2 )
        {
            NN_LOG("checking bkg thr entry\n");
            EXPECT_EQ( expectedBkgThreadLogId ,  e->id );
            EXPECT_EQ( expectedBkgThreadLogCtx,  e->context);
            EXPECT_EQ( g_BkgThreadId          ,  e->thread );

        }
        ++idx;
    }
    nnRes =  nn::perflog::EndDump(dc);
    EXPECT_EQ( true, nnRes.IsSuccess() );


    if( g_CallbackCallCounter == 0 )
    {
       EXPECT_EQ( 3, idx );
       NN_LOG("woke up, passed 3-entry test\n");
    }
    else
    {
       EXPECT_EQ( 0, idx );
       NN_LOG("woke up, passed 0-entry test\n");
    }
    ++g_CallbackCallCounter;
}

TEST(SFDump, IPC_Wait)
{
    OSEvent e;
    g_CallbackCallCounter = 0;
    DumpTestLoggerType log;
    OSThread xtrathrhandle;
    OSThread selfsignalhandle;

    bool bTimedOut;
    nn::Result nnRes;

    nnRes =  nn::perflog::Initialize();
    EXPECT_EQ( true, nnRes.IsSuccess() );


    g_MainThreadId = DumpTestLoggerType::OSGetCurrentThreadId();

    NN_LOG("Main Thread ID=%p\n", reinterpret_cast<void*>(g_MainThreadId) );


    DumpThread<decltype(&dumplog), DumpTestLoggerType*>  dthread1(&dumplog, &log ); // requires C++11 !!!
    EXPECT_EQ( dthread1.Start(&e), true );

    nnRes = nn::perflog::RegisterTrigger( "SFDumpTest", &e );
    EXPECT_EQ( true, nnRes.IsSuccess() );

    log.Put( expectedMainThreadLogId, expectedMainThreadLogCtx );

    EXPECT_EQ( true,
        OSCreateThread( &xtrathrhandle, ExtraThread, static_cast<void *>(&log) ) );

    EXPECT_EQ( true,
        OSCreateThread( &selfsignalhandle, SelfSignalThread, static_cast<void *>(&e) ) );

    EXPECT_EQ( true,
        OSJoinThread( &xtrathrhandle, 2000, &bTimedOut  )  );

    EXPECT_EQ( true, OSCloseThread( &xtrathrhandle ) );

    EXPECT_EQ( false, bTimedOut );


    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

    NN_LOG("Waiting for event trigger...\n" );
    while( g_CallbackCallCounter < 2 )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    EXPECT_EQ( true,
        OSJoinThread( &selfsignalhandle, 2000, &bTimedOut  )  );

    dthread1.Stop();

    NN_LOG("Validating threads\n");
    EXPECT_NE( g_BkgThreadId , g_MainThreadId );
    EXPECT_NE( g_MainThreadId, g_XtrThreadId );
    EXPECT_NE( g_XtrThreadId , g_BkgThreadId );
    NN_LOG("Done validating threads\n");

    nnRes =  nn::perflog::UnregisterTrigger("SFDumpTest");
    EXPECT_EQ( true, nnRes.IsSuccess() );

    nnRes =  nn::perflog::Finalize();
    EXPECT_EQ( true, nnRes.IsSuccess() );
    NN_LOG("Done finalize\n");

}

TEST(SFDump, IPC_Signal)
{
    nn::Result nnRes;

    const char **args = (const char **)nn::os::GetHostArgv();
    const char *evtStr = nullptr;
    int ix = 0;
    while( args[ix] != nullptr )
    {
        if( args[ix][0] == 'e' &&
            args[ix][1] == 'v' &&
            args[ix][2] == 't' &&
            args[ix][3] == '='  )
        {
            evtStr = args[ix];
            evtStr += 4; // skip 'evt='
            break;
        }
        ++ix;
    }

    if( evtStr == nullptr )
    {
        NN_LOG("SKIPPING: This test must be passed an argument of 'evt=eventStr'\n");
        return;
    }

    nnRes =  nn::perflog::Initialize();
    EXPECT_EQ( true, nnRes.IsSuccess() );


    NN_LOG("About to signal evt by name %s\n", evtStr  );
    nnRes = nn::perflog::TriggerDump(evtStr);
    if( !nnRes.IsSuccess() )
    {
        NN_LOG("ErrorCode from TriggerDump: module=%d/0x%x, code=%d/0x%x",
                 nnRes.GetModule(),   nnRes.GetDescription() );
    }

    EXPECT_EQ( true, nnRes.IsSuccess() );

    nnRes =  nn::perflog::Finalize();
    EXPECT_EQ( true, nnRes.IsSuccess() );
    NN_LOG("finalize ok\n"  );
}


