﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn\os.h>
#include <nn\nn_Log.h>
#include "util_ThreadManager.h"
namespace nnt { namespace net { namespace manualtests { namespace utils {

#define NN_THREAD_MANAGER_LOG(format, ...) \
    do \
    {  \
        NN_LOG("[SocketUtil - ThreadManager]: " format, ##__VA_ARGS__); \
    } while ( NN_STATIC_CONDITION(false) )

    int32_t ThreadManager::GetThreadIndex(nn::os::ThreadType * pThreadType)
    {

        int32_t threadIndex = (pThreadType - &m_Threads[0]);
        if (threadIndex > MaxThreads || threadIndex < 0)
        {
            NN_THREAD_MANAGER_LOG("Index for Thread %P Not Found.\n", pThreadType);
            return -1;
        }
        return threadIndex;
    }

    ThreadManager::ThreadManager()
    {
        memset(&m_ThreadStatuses, 0, sizeof(m_ThreadStatuses));

        nn::os::InitializeMutex(&m_ThreadManagerMutex, true, nn::os::MutexLockLevelInitial);

        for (uint32_t i = 0; i < MaxThreads; ++i)
        {
            nn::os::InitializeMutex(&m_ThreadMutexes[i], true, nn::os::MutexLockLevelInitial);
        }

        m_CurrentThreadCount = 0;
    }

    ThreadManager::~ThreadManager()
    {

        for (uint32_t i = 0; i < MaxThreads; ++i)
        {
            if (m_ThreadStatuses[i] != ThreadStatus_Empty)
            {
                nn::os::DestroyThread(&m_Threads[i]);
            }
        }

        for (int32_t i = 0; i < MaxThreads; ++i)
        {
            nn::os::FinalizeMutex(&m_ThreadMutexes[i]);
        }

        nn::os::FinalizeMutex(&m_ThreadManagerMutex);

    }

    nn::os::ThreadType * ThreadManager::CreateThread(void(*pThreadFunc) (void *), void * pThreadFuncArg,int priority)
    {
        int32_t threadIndex = 0;
        bool indexFound = false;
        nn::os::LockMutex(&m_ThreadManagerMutex);
        for (; threadIndex < MaxThreads; ++threadIndex)
        {
            if (m_ThreadStatuses[threadIndex] == ThreadStatus_Empty)
            {
                indexFound = true;
                break;
            }
        }
        if (!indexFound)
        {
            NN_THREAD_MANAGER_LOG("Max Thread Limit Reached: %d\n", MaxThreads);
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return 0;
        }

        nn::Result res = nn::os::CreateThread(&m_Threads[threadIndex],
            pThreadFunc,
            pThreadFuncArg,
            m_ThreadStacks[threadIndex],
            ThreadStackSize,
            priority);
        if (res.IsFailure())
        {
            NN_THREAD_MANAGER_LOG("Failed to Create Thread.\n");
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return 0;
        }
        m_ThreadStatuses[threadIndex] = ThreadStatus_Created;
        ++m_CurrentThreadCount;

        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return &m_Threads[threadIndex];
    }

    int ThreadManager::DestroyThread(nn::os::ThreadType * pThreadType)
    {
        int32_t threadIndex = 0;
        nn::os::LockMutex(&m_ThreadManagerMutex);

        if ((threadIndex = GetThreadIndex(pThreadType)) == -1)
        {
            NN_THREAD_MANAGER_LOG("DestroyThread for Thread %P Failed.\n", pThreadType);
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return -1;
        }

        if (m_ThreadStatuses[threadIndex] == ThreadStatus_Running)
        {
            m_ThreadStatuses[threadIndex] = ThreadStatus_Stopping;
        }
        nn::os::UnlockMutex(&m_ThreadManagerMutex);

        nn::os::DestroyThread(pThreadType);

        nn::os::LockMutex(&m_ThreadManagerMutex);
        m_ThreadStatuses[threadIndex] = ThreadStatus_Empty;
        --m_CurrentThreadCount;

        nn::os::UnlockMutex(&m_ThreadManagerMutex);

        return 0;
    }

    int ThreadManager::WaitThreadWithTimeout(nn::os::ThreadType * pThreadType, nn::TimeSpan timeout)
    {
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType threadMultiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&threadMultiWaitHolder, pThreadType);
        nn::os::LinkMultiWaitHolder(&multiWait, &threadMultiWaitHolder);

        if (nn::os::TimedWaitAny(&multiWait, timeout) != NULL)
        {
            nn::os::LockMutex(&m_ThreadManagerMutex);
            {
                ThreadStatus status;
                GetThreadStatus(pThreadType, &status);
                if (status == ThreadStatus_Running || status == ThreadStatus_Stopping)
                {
                    SetThreadStatus(pThreadType, ThreadStatus_Stopped);
                }
            }
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return 0;
        }
        NN_THREAD_MANAGER_LOG("Timed Out Waiting for Thread %P.\n", pThreadType);
        return -1;
    }

    int ThreadManager::WaitThreadsWithTimeout(nn::os::ThreadType ** pThreadTypes, int threadCount, nn::TimeSpan timeout)
    {
        int res = 0;
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType* threadMultiWaitHolders = new nn::os::MultiWaitHolderType[threadCount];
        nn::os::MultiWaitHolderType timerMultiWaitHolder;
        nn::os::TimerEventType timerEvent;

        nn::os::InitializeMultiWait(&multiWait);

        nn::os::InitializeTimerEvent(&timerEvent, nn::os::EventClearMode_AutoClear);
        nn::os::InitializeMultiWaitHolder(&timerMultiWaitHolder, &timerEvent);
        nn::os::SetMultiWaitHolderUserData(&timerMultiWaitHolder, NULL);
        nn::os::LinkMultiWaitHolder(&multiWait, &timerMultiWaitHolder);

        for (int i = 0; i < threadCount; ++i)
        {
            nn::os::InitializeMultiWaitHolder(&threadMultiWaitHolders[i], pThreadTypes[i]);
            nn::os::SetMultiWaitHolderUserData(&threadMultiWaitHolders[i], reinterpret_cast<uintptr_t>(pThreadTypes[i]));
            nn::os::LinkMultiWaitHolder(&multiWait, &threadMultiWaitHolders[i]);
        }
        nn::os::StartOneShotTimerEvent(&timerEvent, timeout);
        for (int i = 0; i < threadCount; ++i)
        {
            nn::os::MultiWaitHolderType * currentHolder = NULL;
            currentHolder = nn::os::WaitAny(&multiWait);
            nn::os::ThreadType * currentThread = reinterpret_cast<nn::os::ThreadType *>(nn::os::GetMultiWaitHolderUserData(currentHolder));

            if (currentThread == NULL)
            {
                NN_THREAD_MANAGER_LOG("Timed Out Waiting for Threads.\n");
                res = -1;
                break;
            }
            nn::os::UnlinkMultiWaitHolder(currentHolder);
            nn::os::LockMutex(&m_ThreadManagerMutex);
            {
                ThreadStatus status;
                GetThreadStatus(currentThread, &status);
                if (status == ThreadStatus_Running || status == ThreadStatus_Stopping)
                {
                    SetThreadStatus(currentThread, ThreadStatus_Stopped);
                }
                NN_THREAD_MANAGER_LOG("Thread Index %d returned.\n", GetThreadIndex(currentThread));

            }
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
        }

        nn::os::UnlinkAllMultiWaitHolder(&multiWait);

        for (int i = 0; i < threadCount; ++i)
        {
            nn::os::FinalizeMultiWaitHolder(&threadMultiWaitHolders[i]);
        }

        nn::os::FinalizeMultiWaitHolder(&timerMultiWaitHolder);
        nn::os::FinalizeTimerEvent(&timerEvent);
        nn::os::FinalizeMultiWait(&multiWait);
        delete[] threadMultiWaitHolders;

        return res;

    }

    int ThreadManager::WaitAllThreadsWithTimeout(nn::TimeSpan timeout)
    {
        int ret = 0;
        nn::os::ThreadType ** ppThreadTypes = new nn::os::ThreadType*[m_CurrentThreadCount];
        nn::os::LockMutex(&m_ThreadManagerMutex);
        for (int i = 0; i < MaxThreads; ++i)
        {
            if (m_ThreadStatuses[i] != ThreadStatus_Empty)
            {
                ppThreadTypes[i] = &m_Threads[i];
            }
        }
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        ret = WaitThreadsWithTimeout(ppThreadTypes, m_CurrentThreadCount, timeout);
        delete[] ppThreadTypes;
        return ret;
    }

    int ThreadManager::DestroyThreadWithTimeout(nn::os::ThreadType * pThreadType, nn::TimeSpan timeout)
    {
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType threadMultiWaitHolder;

        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&threadMultiWaitHolder, pThreadType);
        nn::os::LinkMultiWaitHolder(&multiWait, &threadMultiWaitHolder);
        SetThreadStatus(pThreadType, ThreadStatus_Stopping);
        if (nn::os::TimedWaitAny(&multiWait, timeout) != NULL)
        {
            DestroyThread(pThreadType);
            return 0;
        }
        NN_THREAD_MANAGER_LOG("Timed Out Waiting for Thread %P.\n", pThreadType);
        return -1;
    }

    int ThreadManager::StartThread(nn::os::ThreadType * pThreadType)
    {
        nn::os::LockMutex(&m_ThreadManagerMutex);
        if (SetThreadStatus(pThreadType, ThreadStatus_Running) == -1)
        {
            NN_THREAD_MANAGER_LOG("Failed to Start Thread %P.\n", pThreadType);
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return -1;
        }
        nn::os::StartThread(pThreadType);
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return 0;
    }

    int ThreadManager::StartAllThreads()
    {
        nn::os::LockMutex(&m_ThreadManagerMutex);
        for (int i = 0; i < MaxThreads; ++i)
        {
            if (m_ThreadStatuses[i] == ThreadStatus_Created)
            {
                if (SetThreadStatus(&m_Threads[i], ThreadStatus_Running) == -1)
                {
                    NN_THREAD_MANAGER_LOG("Failed to Start Thread %P.\n", &m_Threads[i]);
                    nn::os::UnlockMutex(&m_ThreadManagerMutex);
                    return -1;
                }
                nn::os::StartThread(&m_Threads[i]);
            }
        }
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return 0;
    }

    int ThreadManager::StartAllThreadsStaggered(nn::TimeSpan staggerDelay)
    {
        nn::os::LockMutex(&m_ThreadManagerMutex);
        for (int i = 0; i < MaxThreads; ++i)
        {
            if (m_ThreadStatuses[i] == ThreadStatus_Created)
            {
                if (SetThreadStatus(&m_Threads[i], ThreadStatus_Running) == -1)
                {
                    NN_THREAD_MANAGER_LOG("Failed to Start Thread %P.\n", &m_Threads[i]);
                    nn::os::UnlockMutex(&m_ThreadManagerMutex);
                    return -1;
                }
                nn::os::StartThread(&m_Threads[i]);
                nn::os::SleepThread(staggerDelay);
            }
        }
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return 0;
    }

    int ThreadManager::GetThreadStatus(nn::os::ThreadType * pThreadType, ThreadStatus * pThreadStatus)
    {
        nn::os::LockMutex(&m_ThreadManagerMutex);
        int threadIndex = 0;
        if ((threadIndex = GetThreadIndex(pThreadType)) == -1)
        {
            NN_THREAD_MANAGER_LOG("GetThreadStatus for Thread %P Failed.\n", pThreadType);
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return -1;
        }

        *pThreadStatus = m_ThreadStatuses[threadIndex];
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return 0;
    }

    int ThreadManager::SetThreadStatus(nn::os::ThreadType * pThreadType, ThreadStatus threadStatus)
    {
        nn::os::LockMutex(&m_ThreadManagerMutex);
        int threadIndex = 0;
        if ((threadIndex = GetThreadIndex(pThreadType)) == -1)
        {
            NN_THREAD_MANAGER_LOG("SetThreadStatus for Thread %P Failed.\n", pThreadType);
            nn::os::UnlockMutex(&m_ThreadManagerMutex);
            return -1;
        }

        m_ThreadStatuses[threadIndex] = threadStatus;
        nn::os::UnlockMutex(&m_ThreadManagerMutex);
        return 0;
    }

    int ThreadManager::SetCurrentThreadStatus(ThreadStatus threadStatus)
    {
        return SetThreadStatus(nn::os::GetCurrentThread(), threadStatus);
    }
    int ThreadManager::GetCurrentThreadStatus(ThreadStatus * pThreadStatus)
    {
        return GetThreadStatus(nn::os::GetCurrentThread(), pThreadStatus);
    }


}}}}
