﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {


//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

#if 0
static bool                 g_startThread = false;
#endif


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Write_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Write_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 }, targetAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( nn::socket::SockAddr );
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     rc = -1;
    int                     idx;
    bool                    isSuccess = true;
    ssize_t                 rc1 = 0;
    size_t                  argValue = 0;

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  || ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::Write( idx, "X", 1 );
        if ( rc > 0 )
        {
            NN_LOG( "\nWrite() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }

        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nWrite() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nWrite() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [TCP] - NULL Send Buffer - rc == 0\n" );

    rc = nn::socket::Write( tcpSock, nullptr, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Write() succeded but should have Failed, got rc: %d! [TCP]\n", rc );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval, "Errno should have been nn::socket::Errno::EInval, but actually got: %d\n",
                                        nn::socket::GetLastError() );

    ////////////
    // [UDP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [UDP] - NULL Send Buffer - rc == 0\n" );

    rc = nn::socket::Write( udpSock, nullptr, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Write() succeded but should have Failed, got rc: %d! [UDP]\n", rc );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval, "Errno should have been nn::socket::Errno::EInval, but actually got: %d\n",
                                        nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (No Address Len)\n" );

    rc = nn::socket::Write( tcpSock, "X", 0 );
    ERROR_IF_AND_COUNT( rc < 0, "[TCPj] Write() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (No Address Len)\n" );

    rc = nn::socket::Write( udpSock, "X", 0 );
    ERROR_IF_AND_COUNT( rc < 0, "[UDP] Write() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL (Negative) Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (Negative Len == -1)\n" );

    argValue = (size_t) 0 - 1;
    rc =  nn::socket::Write( tcpSock, "X", argValue );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Succeeded but should have failed!\n" );

    ////////////
    // [UDP] - Test NULL (Negative) Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (Negative Len == -1)\n" );

    argValue = (size_t) 0 - 1;
    rc = nn::socket::Write( udpSock, "X", argValue );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Succeeded but should have failed!\n" );

    ////////////
    // [UDP] - Exceed UDP Maximum Transmit size - nn::socket::Errno::EMsgSize
    ////////////

    NN_LOG( "Testing: [UDP]- Exceed Maximum Transmit Size - nn::socket::Errno::EMsgSize\n" );

    outAddrLen = 65600;  // Something greater than 65535
    rc = nn::socket::Write( udpSock, "X", outAddrLen );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Suceeded but should have failed!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EMsgSize,
                        "[UDP] Write() expected errno: nn::socket::Errno::EMsgSize, actually got Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP]
    //
    // Client to Server on NIFM LAN Interface
    ////////////

    NN_LOG( "Testing: [UDP] - Send Direct Send to Peer on Lan Iface - Valid)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
        nn::socket::TimeVal myTime = { 0 };
        nn::socket::SockLenT optlen;
        char dnsRequest[512];

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Server Sock
        udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Server - Local Address
        memset( &servAddr, 0, sizeof(servAddr));
        servAddr.sin_port        = nn::socket::InetHtons( 9056 );
        servAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        servAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Server - Don't wait forever..
        myTime.tv_sec  = 5;
        myTime.tv_usec = 0;
        optlen         = sizeof( myTime );

        rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<void *>(&myTime), optlen );
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Server - Local Bind
        rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &servAddr, sizeof( servAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9057 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9056 );   // Point to UDP Server
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN DIRECTED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr;
        NN_LOG( "Connected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send data from NX (LAN Iface) to NX (Lan Iface)
        rc = nn::socket::Write( udpSock, msgs[0], sizeof( msgs[0] ) );
        ERROR_IF_AND_COUNT( rc < 0, "[UDP] Write() failed but should have succeeded!  Errno: %d\n", nn::socket::GetLastError());

        NN_LOG( "Send - Sent %d bytes\n", rc );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // For()ever
        for( ; ; )
        {
            // Recieve DNS Response Messages
            optlen = sizeof( remoteHost );
            rc1 = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       reinterpret_cast<nn::socket::SockAddr *>(&remoteHost), &optlen );
            if ( rc1 < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
                {
                    NN_LOG( "Finished waiting for DNS Responses!\n" );
                    break;
                }

                NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
            }

            NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa(  remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
        }
    }

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

}   // NOLINT(impl/function_size)


TEST(ApiUnit,Win_Write_Blocking_Sndtimeo)
{
    nn::socket::TimeVal     myTime = { 0 };
    int                     cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    bool                    isSuccess = true, rval;
    nn::socket::SockLenT    optlen;
    uint8_t                 noiseVal = 1;
    unsigned char *         buf1 = NULL;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Establish Send Timeout
    ///////////

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( cliSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, reinterpret_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed - Errno: %d\n", nn::socket::GetLastError() );

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc = nn::socket::Write( cliSock, buf1, 32767 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- SNDTIMEO unblocked after 5 seconds\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Write() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}

TEST(ApiUnit,Win_Write_Blocking)
{
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;
    bool                isSuccess = true, rval;
    uint8_t             noiseVal = 1;
    unsigned char *     buf1 = NULL;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( cliSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Send stuff until we would block
    ///////////

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc = nn::socket::Write( cliSock, buf1, 32767 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- unblocked because write would block!\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Write() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }
}

#if 0

namespace {

typedef struct
{
    int   threadId;
    int   myIdx;
    int   sockfds;
    int   count;
    int   desiredCore;

} WRITE_PAYLOAD_TYPE;

}


static void
Win_Write_UDP_To_Clients( void  * inArg )
{
    WRITE_PAYLOAD_TYPE *     payload;
    int                      idx, rc;
    bool                     isSuccess = true;
    char                     buf1[5];

    // Restore Payload
    payload = (WRITE_PAYLOAD_TYPE *) inArg;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    // Fix my output
    memset( buf1, 0, sizeof( buf1 ) );
    snprintf( buf1, 5, "%d", payload->myIdx );

    while( g_startThread == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 500 ) );
    }

    NN_LOG( "Thread %d Starting!\n", payload->myIdx );

    for( idx = 0; idx < payload->count; idx++ )
    {
        rc = nn::socket::Write( payload->sockfds, (unsigned char *) buf1, 1 );
        ERROR_IF( rc != 1, "Failed to write 'A' char at position [%d]\n", idx );

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

out:

    NN_LOG( "Thread %d Exiting!\n", payload->myIdx );
}


TEST(ApiUnit,Win_Write_UDP_To_Clients)
{
    WRITE_PAYLOAD_TYPE      payload[5];
    nn::socket::TimeVal          myTime;
    nn::socket::SockAddrIn      outAddr = { 0 }, recvAddr = { 0 };
    int                     idx = 0, rc = 0;
    int                     udpSock;
    nn::socket::SockLenT               recvAddrLen;
    bool                    isSuccess = true;
    char                    buf1[100];
    int                     core = 1;
    int                     portCount = 1;
    ssize_t                 rc1 = 0;

    // Initialize Payload
    for( idx = 0; idx < 5; idx++ )
    {
        payload[ idx ].threadId = -1;
        payload[ idx ].myIdx    = -1;
        payload[ idx ].sockfds  = -1;
        payload[ idx ].count    = -1;
    }

    ////////////
    // Make UDP Server
    ///////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 7000 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    ////////////
    // Make Threaded Test
    ///////////

    g_startThread = false;
    portCount = 7001;
    core = 1;
    for( idx = 0; idx < 5; idx++ )
    {
        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx].myIdx       = idx;
        payload[idx].count       = 10;
        payload[idx].desiredCore = core;

        payload[idx].sockfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", (uint16_t) portCount, "127.0.0.1", 7000 );
        if ( payload[idx].sockfds < 0 )
        {
            NN_LOG( "Failed to create a new UDP Socket!\n" );
            goto out;
        }
        portCount++;

        payload[idx].threadId = g_pThreadedTest->CreateThread( &Win_Write_UDP_To_Clients, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create Client (thread)!\n" );
            goto out;
        }
    }

    // Catch 50 messages
    g_startThread = true;
    for( idx = 0; idx < 50; idx++ )
    {
        recvAddrLen = sizeof( recvAddr );
        rc1 = nn::socket::Recv( udpSock, buf1, sizeof( buf1 ), 0 );
        ERROR_IF( rc1 < 0, "Recv()  - Errno: %d\n", nn::socket::GetLastError() );

        NN_LOG( "Got [%.*s]\n", rc, buf1 );
    }


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    // Free Payload
    for( idx = 0; idx < 5; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            g_pThreadedTest->WaitThread(    payload[idx].threadId );
            g_pThreadedTest->DestroyThread( payload[idx].threadId );
            payload[idx].threadId = -1;
        }

        if ( payload[ idx ].sockfds > -1 )
        {
            nn::socket::Close( payload[ idx ].sockfds );
            payload[ idx ].sockfds = -1;
        }
    }
}

#endif



TEST(ApiUnit,Win_Write_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
