﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

static bool                 g_startThread = false;
static uint8_t              g_noiseVal = 1;



static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( nullptr );

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, reinterpret_cast<void *>(g_echoServer), kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Write_Initialize)
{
    InitializeTesting();
}

// Anonymous namespace
namespace {

//////////////////////////////////////////
// $ dig google.com
//
// ; <<>> DiG 9.10.3-P4 <<>> google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 43551
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 1
//
// ;; OPT PSEUDOSECTION:
// ; EDNS: version: 0, flags:; udp: 4096
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             255     IN      A       172.217.3.174
//
// ;; Query time: 4 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Wed Oct 26 09:05:02 PDT 2016
// ;; MSG SIZE  rcvd: 55
//
//////////////////////////////////////////

static const char GoogleDNSRequest[] =
{
0xaa, 0x1f, 0x01, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00
};

}; // End - Anonymous namespace



TEST(ApiUnit,Write_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 }, targetAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( nn::socket::SockAddr );
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1;
    int                     rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx, enableFlag = 1;
    bool                    isSuccess = true;

    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  || ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::Write( idx, "X", 1 );
        if ( rc > 0 )
        {
            NN_LOG( "\nWrite() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }

        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nWrite() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nWrite() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );



    ////////////
    // [TCP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [TCP] - NULL Send Buffer - rc == 0\n" );

    rc = nn::socket::Write( tcpSock, nullptr, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Write() succeded but should have Failed, got rc: %d! [TCP]\n", rc );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval, "Errno should have been nn::socket::Errno::EInval, but actually got: %d\n",
                                        nn::socket::GetLastError() );

    ////////////
    // [UDP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [UDP] - NULL Send Buffer - rc == 0\n" );

    rc = nn::socket::Write( udpSock, nullptr, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Write() succeded but should have Failed, got rc: %d! [UDP]\n", rc );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval, "Errno should have been nn::socket::Errno::EInval, but actually got: %d\n",
                                        nn::socket::GetLastError() );



    ////////////
    // [TCP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (No Address Len)\n" );

    rc = nn::socket::Write( tcpSock, "X", 0 );
    ERROR_IF_AND_COUNT( rc < 0, "[TCPj] Write() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (No Address Len)\n" );

    rc = nn::socket::Write( udpSock, "X", 0 );
    ERROR_IF_AND_COUNT( rc < 0, "[UDP] Write() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );



    ////////////
    // [TCP] - Test NULL (Negative) Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (Negative Len == -1)\n" );

    rc = nn::socket::Write( tcpSock, "X", -1 );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Succeeded but should have failed!\n" );
    ERROR_IF_AND_COUNT(nn::socket::GetLastError() != nn::socket::Errno::EFault,
        "[UDP] Write() expected errno: nn::socket::Errno::EFault, actually got Errno: %d\n", nn::socket::GetLastError());

    ////////////
    // [UDP] - Test NULL (Negative) Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (Negative Len == -1)\n" );

    rc = nn::socket::Write( udpSock, "X", -1 );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Succeeded but should have failed!\n" );
    ERROR_IF_AND_COUNT(nn::socket::GetLastError() != nn::socket::Errno::EFault,
        "[UDP] Write() expected errno: nn::socket::Errno::EFault, actually got Errno: %d\n", nn::socket::GetLastError());


    ////////////
    // [UDP] - Exceed UDP Maximum Transmit size - nn::socket::Errno::EMsgSize
    ////////////

    NN_LOG( "Testing: [UDP]- Exceed Maximum Transmit Size - nn::socket::Errno::EMsgSize\n" );

    outAddrLen = 65600;  // Something greater than 65535
    rc = nn::socket::Write( udpSock, "X", outAddrLen );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] Write() Suceeded but should have failed!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EMsgSize,
                        "[UDP] Write() expected errno: nn::socket::Errno::EMsgSize, actually got Errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // [UDP]
    //
    // ECONNNREFUSED - The socket received an ICMP destination unreachable from the last
    // message sent.
    ////////////

    NN_LOG( "Testing: [UDP]- Disconnected UDP Server - nn::socket::Errno::EConnRefused\n" );

    {
        // UDP Server Sock
        udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9050, "127.0.0.1", 9060 );
        ERROR_IF_AND_COUNT( udpServSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Connect to Server
        udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9060, "127.0.0.1", 9050 );
        ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // Write from Client to Server
        nn::socket::Write( udpSock, "x", 1);

        // Close UDP Server
        nn::socket::Close( udpServSock );
        udpServSock = -1;

        // Client keeps writing - but expected to FAIL
        for( idx = 0; idx < 10; idx++ )
        {
            rc = nn::socket::Write( udpSock, "X", 1 );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EConnRefused )
                {
                    NN_LOG( "PASSED!  Send failed with nn::socket::Errno::EConnRefused!\n" );
                    break;
                }

                NN_LOG( "Send failed with errno: %d\n", myError );
            }

            NN_LOG( "Good Send to closed UDP socket...\n" );
        }

        if ( idx >= 10 )
        {
            NN_LOG( "FAILED!  Send succeeded 10 times but should have failed with nn::socket::Errno::EConnRefused!\n" );
            goto out;
        }
    }

    ////////////
    // [UDP] - nn::socket::Errno::ENetUnreach
    //
    // nn::socket::Option::So_DontRoute - If an interface is not directly connected to the NX don't follow
    // the default route.
    ////////////

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };


        NN_LOG( "Testing: [UDP] - nn::socket::Option::So_DontRoute - Errno == nn::socket::Errno::ENetUnreach (101)\n" );

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9004 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Client - IP
        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP - Client - Set Socket Option: nn::socket::Option::So_DontRoute
        rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_DontRoute, reinterpret_cast<void *>(&enableFlag), sizeof( enableFlag ) );
        ERROR_IF( rc < 0, "SetSockOpt() failed to set nn::socket::Option::So_DontRoute, errno: %d\n", nn::socket::GetLastError() );

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 53 );   // DNS == domain == 53
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: (E-ROOT)
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &targetAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data which is not (directly) routable from NX
        rc = nn::socket::Write( udpSock, "Test Data", strlen( "Test Data" ) );
        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Write() succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        ERROR_IF( myError != nn::socket::Errno::ENetUnreach, "Expected Errno: nn::socket::Errno::ENetUnreach (101) but actually got: %d\n", myError );
    }


    ////////////
    // [UDP]
    //
    // Client to Server on NIFM LAN Interface
    ////////////

    NN_LOG( "Testing: [UDP] - Send Direct Send to Peer on Lan Iface - Valid)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
        nn::socket::TimeVal myTime = { 0 };
        nn::socket::SockLenT optlen;
        char dnsRequest[512];

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Server Sock
        udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Server - Local Address
        memset( &servAddr, 0, sizeof(servAddr));
        servAddr.sin_port        = nn::socket::InetHtons( 9056 );
        servAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        servAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Server - Don't wait forever..
        myTime.tv_sec  = 5;
        myTime.tv_usec = 0;
        optlen         = sizeof( myTime );

        rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<void *>(&myTime), optlen );
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Server - Local Bind
        rc = nn::socket::Bind( udpServSock, reinterpret_cast<nn::socket::SockAddr *>(&servAddr), sizeof( servAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9057 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9056 );   // Point to UDP Server
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN DIRECTED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr;
        NN_LOG( "Connected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send data from NX (LAN Iface) to NX (Lan Iface)
        rc = nn::socket::Write( udpSock, GoogleDNSRequest, sizeof( GoogleDNSRequest ) );
        ERROR_IF_AND_COUNT( rc < 0, "[UDP] Write() failed but should have succeeded!  Errno: %d\n", nn::socket::GetLastError());

        NN_LOG( "Send - Sent %d bytes\n", rc );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // For()ever
        for( ; ; )
        {
            // Recieve DNS Response Messages
            optlen = sizeof( remoteHost );
            rc = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       reinterpret_cast<nn::socket::SockAddr *>(&remoteHost), &optlen );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
                {
                    NN_LOG( "Finished waiting for DNS Responses!\n" );
                    break;
                }

                NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
            }

            NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa(  remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
        }
    }

    ////////////
    // [UDP]
    //
    // Send Directed Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Directed Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9005 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9006 );   // Somewhere harmless
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN DIRECTED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
        NN_LOG( "Directed Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data from NX
        rc = nn::socket::Write( udpSock, "Test Data", strlen( "Test Data" ) );

        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Write() Directed Broadcast Address succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] Write() Directed Broadcast Address returns Errno: %s!\n", myError );
    }

    ////////////
    // [UDP]
    //
    // Send Directed Broadcast Address - SetSockOpt - Valid Response
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Directed Broadcast Address - SockOpt - Valid)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
        nn::socket::TimeVal myTime = { 0 };
        nn::socket::SockLenT optlen;
        int enableFlag = 1;
        char dnsRequest[512];

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Server Sock - Close
        nn::socket::Close( udpServSock );

        // UDP Server Sock
        udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Server - Local Address
        memset( &servAddr, 0, sizeof(servAddr));
        servAddr.sin_port        = nn::socket::InetHtons( 9058 );
        servAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        servAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Server - Don't wait forever..
        myTime.tv_sec  = 5;
        myTime.tv_usec = 0;
        optlen         = sizeof( myTime );

        rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<void *>(&myTime), optlen );
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Server - Local Bind
        rc = nn::socket::Bind( udpServSock, reinterpret_cast<nn::socket::SockAddr *>(&servAddr), sizeof( servAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - May send Broadcast Packets
        rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &enableFlag, sizeof(enableFlag));

        // This converts a directed broadcast to a limited broadcast.
        // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
        // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
        rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));
        ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::Ip_OnesBcast failed - Errno: %d\n", nn::socket::GetLastError() );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9057 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9058 );   // Point to UDP Server
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN LIMITED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
        NN_LOG( "Directed Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send Directed data from NX
        rc = nn::socket::Write( udpSock, GoogleDNSRequest, sizeof( GoogleDNSRequest ) );

        ERROR_IF_AND_COUNT( rc < 0,
        "[UDP] Write() Directed Broadcast Address failed but should have succeeded!  Errno:\n", nn::socket::GetLastError());

        NN_LOG( "Send - Sent %d bytes\n", rc );

        // For()ever
        for( ; ; )
        {
            // Recieve DNS Response Messages
            optlen = sizeof( remoteHost );
            rc = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       reinterpret_cast<nn::socket::SockAddr *>(&remoteHost), &optlen );
            if ( rc < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
                {
                    NN_LOG( "Finished waiting for DNS Responses!\n" );
                    break;
                }

                NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
            }

            NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa( remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
        }
    }

    ////////////
    // [UDP]
    //
    // Send Limited Broadcast Address Address - nn::socket::Errno::EAcces
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=177019016
    ////////////

    NN_LOG( "Testing: [UDP] - Send Limited Broadcast Address - No SetSockOpt - nn::socket::Errno::EAcces)\n" );

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };
        int enableFlag = 1;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9010 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 9011 );   // Somewhere harmless
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: LAN LIMITED BROADCAST ADDRESS
        targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;

        NN_LOG( "Limited Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

        // This converts a directed broadcast to a limited broadcast.
        // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
        // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
        nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data which is not (directly) routable from NX
        rc = nn::socket::Write( udpSock, "Test Data", strlen( "Test Data" ) );

        ERROR_IF_AND_COUNT( rc > 0, "[UDP] Write() Directed Broadcast Address succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] Write() Directed Broadcast Address returns Errno: %s!\n", myError );
    }

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

}   // NOLINT(impl/function_size)


TEST(ApiUnit,Write_Blocking_Sndtimeo)
{
    nn::socket::TimeVal myTime = { 0 };
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;

    bool                isSuccess = true, rval;
    nn::socket::SockLenT           optlen;
    uint8_t             noiseVal = 1;
    unsigned char *     buf1 = NULL;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Establish Send Timeout
    ///////////

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( cliSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, reinterpret_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed - Errno: %d\n", nn::socket::GetLastError() );

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc = nn::socket::Write( cliSock, buf1, 32767 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- SNDTIMEO unblocked after 5 seconds\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Write() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}

TEST(ApiUnit,Write_Blocking)
{
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;

    bool                isSuccess = true, rval;
    uint8_t             noiseVal = 1;
    unsigned char *     buf1 = NULL;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( cliSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Send stuff until we would block
    ///////////

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc = nn::socket::Write( cliSock, buf1, 32767 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- unblocked because write would block!\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Write() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }
}

namespace {

struct WRITE_PAYLOAD
{
    int    threadId;
    int    desiredCore;
    int    sockfd;
};

} // End Anonymous namespace


namespace {

typedef struct
{
    int   threadId;
    int   myIdx;
    int   sockfds;
    int   count;
    int   desiredCore;

} WRITE_PAYLOAD_TYPE;

}


static void
Write_UDP_To_Clients( void  * inArg )
{
    WRITE_PAYLOAD_TYPE *     payload;
    int                      idx, rc;
    bool                     isSuccess = true;
    char                     buf1[5];

    // Restore Payload
    payload = (WRITE_PAYLOAD_TYPE *) inArg;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    // Fix my output
    memset( buf1, 0, sizeof( buf1 ) );
    snprintf( buf1, 5, "%d", payload->myIdx );

    while( g_startThread == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 500 ) );
    }

    NN_LOG( "Thread %d Starting!\n", payload->myIdx );

    for( idx = 0; idx < payload->count; idx++ )
    {
        rc = nn::socket::Write( payload->sockfds, (unsigned char *) buf1, 1 );
        ERROR_IF( rc != 1, "Failed to write 'A' char at position [%d]\n", idx );

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

out:

    NN_LOG( "Thread %d Exiting!\n", payload->myIdx );
}


TEST(ApiUnit,Write_UDP_To_Clients)
{
    WRITE_PAYLOAD_TYPE      payload[5];
    nn::socket::TimeVal          myTime;
    nn::socket::SockAddrIn      outAddr = { 0 }, recvAddr = { 0 };
    int                     idx = 0, rc = 0;
    int                     udpSock;
    nn::socket::SockLenT               recvAddrLen;
    bool                    isSuccess = true;
    char                    buf1[100];
    int                     core = 1;
    int                     portCount = 1;

    // Initialize Payload
    for( idx = 0; idx < 5; idx++ )
    {
        payload[ idx ].threadId = -1;
        payload[ idx ].myIdx    = -1;
        payload[ idx ].sockfds  = -1;
        payload[ idx ].count    = -1;
    }

    ////////////
    // Make UDP Server
    ///////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 7000 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, reinterpret_cast<void *>(&myTime), sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&outAddr), sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    ////////////
    // Make Threaded Test
    ///////////

    g_startThread = false;
    portCount = 7001;
    core = 1;
    for( idx = 0; idx < 5; idx++ )
    {
        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx].myIdx       = idx;
        payload[idx].count       = 10;
        payload[idx].desiredCore = core;

        payload[idx].sockfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", portCount, "127.0.0.1", 7000 );
        if ( payload[idx].sockfds < 0 )
        {
            NN_LOG( "Failed to create a new UDP Socket!\n" );
            goto out;
        }
        portCount++;

        payload[idx].threadId = g_pThreadedTest->CreateThread( &Write_UDP_To_Clients, reinterpret_cast<void *>(&payload[idx]), kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create Client (thread)!\n" );
            goto out;
        }
    }

    // Catch 50 messages
    g_startThread = true;
    for( idx = 0; idx < 50; idx++ )
    {
        recvAddrLen = sizeof( recvAddr );
        rc = nn::socket::Recv( udpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
        ERROR_IF( rc < 0, "Recv()  - Errno: %d\n", nn::socket::GetLastError() );

        NN_LOG( "Got [%.*s]\n", rc, buf1 );
    }


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    // Free Payload
    for( idx = 0; idx < 5; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            g_pThreadedTest->WaitThread(    payload[idx].threadId );
            g_pThreadedTest->DestroyThread( payload[idx].threadId );
            payload[idx].threadId = -1;
        }

        if ( payload[ idx ].sockfds > -1 )
        {
            nn::socket::Close( payload[ idx ].sockfds );
            payload[ idx ].sockfds = -1;
        }
    }
}


TEST(ApiUnit,Write_SO_SNDLOWAT)
{
    int              intoptval;
    nn::socket::SockLenT        intoptlen;

    int               clisock = -1, writeCount, highWaterMark = -1;
    int               rc = -1;
    bool              isSuccess = true, rval;
    unsigned char *   workPtr;
    size_t            workPtrLen;
    nn::socket::TimeVal    mytime;
    nn::socket::FdSet            writeSet;

    ////////////////
    // nn::socket::Option::So_SndLoWat [TCP] - Option
    //
    // NOTE: No UDP test because UDP alwasy succeeds with UDP + Write()
    ////////////////

    workPtrLen = 1024;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, workPtrLen, g_noiseVal);
    ERROR_IF( rc < 0, "Make Random Noise Failed for size: %d\n", workPtrLen  );

    // Client: Set nn::socket::Option::So_SndLoWat
    //
    clisock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8053 );
    ERROR_IF( clisock < 0, "Connect failed but should have succeeded!\n" );

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server to (NOT) echo
    g_echoServer->setFalseEcho( true );

    // Find High Water mark...
    writeCount = 0;
    for( ; ; )
    {
         nn::socket::FdSetZero( &writeSet );

         nn::socket::FdSetSet( clisock, &writeSet );

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select( clisock + 1, nullptr, &writeSet, nullptr, &mytime );
         if ( rc == 0  )
         {
             NN_LOG( "Socket not ready for writing after bytes: %d sent\n", writeCount );
             break;
         }
         else if ( rc < 0 )
         {
             NN_LOG( "Select fail with errno: %d\n", nn::socket::GetLastError() );
             goto out;
         }

         rc = nn::socket::Write( clisock, workPtr, workPtrLen );
         if ( rc < 1 )
         {
             if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
             {
                 continue;
             }

             ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n", nn::socket::GetLastError() );
         }

         if ( rc != workPtrLen )
         {
             NN_LOG( "Short Write!  Expected: %d, actually wrote: %d\n", workPtrLen, rc );
         }

         writeCount = writeCount + rc;
    }

    // Close Clinet Socket
    nn::socket::Close( clisock );

    // At High Water mark...
    highWaterMark = writeCount;
    NN_LOG( "=======> High Water Mark: %d\n", writeCount );


    // Client: Set nn::socket::Option::So_SndLoWat to A little over High Water Mark
    //
    intoptval = workPtrLen;
    intoptlen = sizeof( intoptval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect( clisock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndLoWat, &intoptval, intoptlen,
                                   9000, g_echoServer->kInterfaceIPv4Addr,
                                   8053, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    // Set Non Blocking I/O
    rc = nn::socket::Fcntl( clisock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell server to (NOT) echo
    g_echoServer->setFalseEcho( true );

    // For()ever
    writeCount = 0;
    for( ; ; )
    {
         nn::socket::FdSetZero( &writeSet );

         nn::socket::FdSetSet( clisock, &writeSet );

         mytime.tv_sec  = 0;
         mytime.tv_usec = 100;

         rc = nn::socket::Select( clisock + 1, nullptr, &writeSet, nullptr, &mytime );
         if ( rc == 0  )
         {
             NN_LOG( "Socket not ready for writing after bytes: %d sent\n", writeCount );
             break;
         }
         else if ( rc < 0 )
         {
             NN_LOG( "Select fail with errno: %d\n", nn::socket::GetLastError() );
             goto out;
         }

         rc = nn::socket::Write( clisock, workPtr, workPtrLen );
         if ( rc < 1 )
         {
             if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
             {
                 continue;
             }

             ERROR_IF( rc < 0, "Write Failed for size: %d - errno: %d\n", nn::socket::GetLastError() );
         }

         if ( rc != workPtrLen )
         {
             NN_LOG( "Short Write!  Expected: %d, actually wrote: %d\n", workPtrLen, rc );
         }

         writeCount = writeCount + rc;
    }

    // At High Water mark...
    nn::socket::Close( clisock );
    NN_LOG( "=======> High Water Mark: %d\n", highWaterMark );
    NN_LOG( "=======> Low  Water Mark: %d\n", writeCount );
    NN_LOG( "=======> Difference     : %d\n", ( highWaterMark - writeCount ) );

    nTestsPassing++;

    ERROR_IF_AND_COUNT( writeCount > highWaterMark, "Low water mark is higher than High Water mark!  Failed!\n" );

out:

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    // Free Payload memory
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;

}  // NOLINT(impl/function_size)



TEST(ApiUnit,Write_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Write_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
