﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/socket.h>
#include <nn/socket/socket_TypesPrivate.h>

#include <nnt/nntest.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>

#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>

#include <nn/nn_Log.h>


#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions


namespace NATF {
namespace API {


//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32


//*********************
//*  G L O B A L S
//*********************

static ThreadedController * g_pThreadedTest = NULL;
static EchoServer *         g_echoServer = NULL;

static const int            kIPHeaderLen = 20;        // IP Header Len
static const int            kICMPHeaderLen = 8;       // ICMP Header Len

static void
StartEchoServer( void * inArg )
{
    NN_UNUSED(kICMPHeaderLen);
    NN_UNUSED(kIPHeaderLen);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}



static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( 32 );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, 10 );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // Free Threaded Tester
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    delete g_echoServer;
    g_echoServer = NULL;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


TEST(ApiUnit,SetSockOpt_IP_Initialize)
{
    InitializeTesting();
}


#ifdef COMMENTED_OUT_SIGLO_72459
TEST(ApiUnit,SetSockOpt_IP_Port_Scanner)
{
    nn::socket::InAddr      myIp = {0};
    nn::socket::InAddr      myMask = {0};
    nn::socket::InAddr      myGw = {0};
    nn::socket::InAddr      myDns1 = {0};
    nn::socket::InAddr      myDns2 = {0};

    nn::socket::SockAddrIn  tcpAddr, udpAddr, udpTarget;
    uint32_t                idx;
    int                     rc;
    bool                    isSuccess = true;
    int                     tcpSock, udpSock;
    uint16_t                udpLocalPort = 0;
    unsigned char           bufPtr[4] = { 'T','E','S','T' };
    size_t                  bufPtrLen = sizeof(bufPtr);
    nn::socket::SockLenT    udpAddrSize;
    nn::socket::TimeVal     myTime;
    nn::socket::Errno       errNum = nn::socket::Errno::ESuccess;


    ////////////
    // Ask NIFM for (DHCP) Informatiom
    ////////////

    nn::Result result = nn::nifm::GetCurrentIpConfigInfo(&myIp, &myMask, &myGw, &myDns1, &myDns2);
    if ( result.IsFailure()  )
    {
        NN_LOG( "Failed to get my IP Address from NIFM\n" );
        return;
    }

    NN_LOG( "NIFM: My IP Address: %d.%d.%d.%d\n",
         static_cast<uint8_t>( myIp.S_addr ),
         static_cast<uint8_t>( myIp.S_addr >> 8 ),
         static_cast<uint8_t>( myIp.S_addr >> 16 ),
         static_cast<uint8_t>( myIp.S_addr >> 24 ) );


    ////////////
    // Test TCP
    ////////////

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

    memset( &tcpAddr, 0, sizeof( tcpAddr ) );
    tcpAddr.sin_family       = nn::socket::Family::Af_Inet;
    tcpAddr.sin_port         = nn::socket::InetHtons(0);
    tcpAddr.sin_addr.S_addr  = myIp.S_addr;

    ////////////
    // Test UDP
    ////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    memset(&udpAddr, 0, sizeof(udpAddr));
    udpAddr.sin_family        = nn::socket::Family::Af_Inet;
    udpAddr.sin_port          = nn::socket::InetHtons(0);
    udpAddr.sin_addr.S_addr   = myIp.S_addr;

    memset(&udpTarget, 0, sizeof(udpTarget));
    udpTarget.sin_family      = nn::socket::Family::Af_Inet;
    udpTarget.sin_port        = nn::socket::InetHtons(0);
    udpTarget.sin_addr.S_addr = myIp.S_addr;

    myTime.tv_sec  = 0;
    myTime.tv_usec = 100;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt failed with errno: %d\n", nn::socket::GetLastError() );


    NN_LOG( "Port Scanning: Interface[%d.%d.%d.%d: [TCP][UDP]\n",
                    static_cast<uint8_t>( myIp.S_addr ),
                    static_cast<uint8_t>( myIp.S_addr >> 8 ),
                    static_cast<uint8_t>( myIp.S_addr >> 16 ),
                    static_cast<uint8_t>( myIp.S_addr >> 24 ) );

    rc = nn::socket::Bind(udpSock, reinterpret_cast<nn::socket::SockAddr *>(&udpAddr), sizeof(udpAddr));
    ERROR_IF( rc < 0, "Bind failed with errno: %d\n", nn::socket::GetLastError() );

    udpAddrSize = sizeof(udpAddr);
    rc = nn::socket::GetSockName(udpSock,reinterpret_cast<nn::socket::SockAddr *>(&udpAddr), &udpAddrSize);
    ERROR_IF( rc < 0, "GetSockName() failed with errno: %d\n", nn::socket::GetLastError() );

    udpLocalPort = nn::socket::InetNtohs(udpAddr.sin_port);

    for( idx = 1; idx <= IPPORT_MAX; idx++ )
    {
        // Skip (SELF) bind
        if ((uint16_t)idx == udpLocalPort)
            continue;

        // TCP
        tcpAddr.sin_port = nn::socket::InetHtons((uint16_t)idx);

        rc = nn::socket::Connect( tcpSock, reinterpret_cast<nn::socket::SockAddr *>(&tcpAddr), sizeof( tcpAddr ) );
        if ( rc > -1)
        {
            NN_LOG( "[TCP]: %d \n", idx );
            ERROR_IF_AND_COUNT( rc > -1, "Failed!  TCP Connection should not have worked!\n" );
        }
        else if((errNum = nn::socket::GetLastError()) != nn::socket::Errno::EConnRefused)
        {
            NN_LOG("[TCP] Unexpected Error, expected error nn::socket::Errno::EConnRefused (111) and encountered error: %d on port: %d\n", errNum, idx);
            nn::socket::Close(tcpSock);
            ERROR_IF_AND_COUNT(errNum != nn::socket::Errno::EConnRefused, "Failed! expected error nn::socket::Errno::EConnRefused (111) and encountered error: %d on port: %d\n", errNum, idx);
        }
        else
        {
            // socket now in dropped state and will no longer connect. Need to remake the socket.
            // This is currently quite slow to do 65k times. If there is a faster way to reset the socket that would be good.
            nn::socket::Close(tcpSock);
            tcpSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        }
        // UDP
        udpTarget.sin_port = nn::socket::InetHtons((uint16_t)idx);

        rc = nn::socket::SendTo(udpSock, bufPtr, bufPtrLen, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&udpTarget), sizeof( udpTarget ) );
        if (rc < 0)
        {
            NN_LOG("[UDP] Unexpected Error, error num: %d on port: %d\n", nn::socket::GetLastError(), idx);
            ERROR_IF_AND_COUNT(errNum != nn::socket::Errno::EAgain, "Failed! Unexpected error num: %d on port: %d\n", nn::socket::GetLastError(), idx);
        }
        udpAddrSize = sizeof(udpAddr);
        rc = nn::socket::RecvFrom(udpSock, bufPtr, bufPtrLen, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&udpAddr), &udpAddrSize);
        if (rc > -1)
        {
            NN_LOG("[UDP]: %d\n", idx);
            ERROR_IF_AND_COUNT(rc > -1, "Failed! UDP RecvFrom should not have worked!\n");
        }
        else if((errNum = nn::socket::GetLastError()) != nn::socket::Errno::EAgain)
        {
            NN_LOG("[UDP] Unexpected Error, expected error nn::socket::Errno::EAgain (11) and encountered error: %d on port: %d\n", errNum, idx);
            ERROR_IF_AND_COUNT(errNum != nn::socket::Errno::EAgain, "Failed! expected error nn::socket::Errno::EAgain (11) and encountered error: %d on port: %d\n", errNum, idx);
        }

        if ( ( idx % 1000 ) == 0 ) NN_LOG( "." );
    }

    NN_LOG( "Port Scan Complete\n" );

out:


    // Close TCP socket descriptor.
    if ( tcpSock > 0 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    // Close UDP socket descriptor.
    if ( udpSock > 0 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    nTestsPassing++;

    return;

}  // NOLINT(impl/function_size)
#endif


struct SetSockOptIPPROTOPayload
{
    int                sockfds;

    int                ttl;
    nn::socket::Icmp   icmpHeader;
    nn::os::Tick       timeSent;
};


static int
SetSockOpt_Send_Ping( nn::socket::InAddr                  in_srcAddr,
                      const char *                        in_dstAddr,
                      struct SetSockOptIPPROTOPayload *   in_sentPing,
                      int                                 in_sequence )
{
    nn::socket::AddrInfo * addrInfo = NULL;
    nn::socket::AddrInfo   addrInfoHint = {};
    nn::socket::TimeVal    myTime;
    int                 sockfds;

    int                 idx, rc, intopt = -1;
    bool                isSuccess = true;
    uint64_t            unused1;
    unsigned char *     bufPtr = NULL;
    size_t              bufPtrLen;
    uint8_t *           icmpData = NULL;
    nn::socket::Icmp *  sendIcmpHeader = NULL;

    NN_UNUSED(unused1);
    NN_UNUSED(idx);
    NN_UNUSED(sockfds);
    NN_UNUSED(isSuccess);

    memset( in_sentPing, 0, sizeof( struct SetSockOptIPPROTOPayload ) );

    // RAW + IP Socket
    in_sentPing->sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Raw, nn::socket::Protocol::IpProto_Icmp );
    ERROR_IF( in_sentPing->sockfds < 0, "Failed to create a nn::socket::Type::Sock_Raw socket!  errno: %d\n", nn::socket::GetLastError() );

    // Set a Recieve timeout on socket
    myTime.tv_sec  = 0;
    myTime.tv_usec = 500;

    rc = nn::socket::SetSockOpt( in_sentPing->sockfds, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &myTime, sizeof( myTime ) );
    ERROR_IF(rc < 0, "SetSockOpt nn::socket::Level::Sol_Socket nn::socket::Option::So_RcvTimeo ERROR: %d\n", nn::socket::GetLastError() );

    // Allocate an ICMP Buffer
    bufPtrLen = sizeof( icmp );
    bufPtr = (unsigned char *) malloc( bufPtrLen );

    // Set TTL via socket opt..
    intopt = in_sequence;
    rc = nn::socket::SetSockOpt( in_sentPing->sockfds, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Ttl, &intopt, sizeof( intopt ) );
    ERROR_IF(rc < 0,"SetSockOpt nn::socket::Level::Sol_Ip nn::socket::Option::Ip_Ttl ERROR: %d\n", nn::socket::GetLastError() );

    sendIcmpHeader = reinterpret_cast<nn::socket::Icmp*>(bufPtr);
    memset(sendIcmpHeader, 0, bufPtrLen);

    sendIcmpHeader->icmp_type             = nn::socket::IcmpType::Icmp_Echo;
    sendIcmpHeader->icmp_code             = static_cast<nn::socket::IcmpCode>(0);
    sendIcmpHeader->icmp_cksum            = 0;
    sendIcmpHeader->icmp_id               = static_cast<uint64_t>(nn::os::GetSystemTick().GetInt64Value());
    sendIcmpHeader->icmp_seq              = 0;

    // Add in Echo Payload
    icmpData = (uint8_t *)&(sendIcmpHeader->icmp_data);
    icmpData[0] = 'T';
    icmpData[1] = 'E';
    icmpData[2] = 'S';
    icmpData[3] = 'T';

    // ICMP Checksum
    {
        uint32_t checksum = 0;

        int size = bufPtrLen;
        if (size & 1)
        {
            checksum += reinterpret_cast<uint8_t*>(bufPtr)[--size];
        }

        size /= 2;
        while (size--)
        {
            checksum += reinterpret_cast<uint16_t*>(bufPtr)[size];
        }

        checksum = (checksum & 0xFFFF) + (checksum >> 16);
        checksum = (checksum & 0xFFFF) + (checksum >> 16);
        sendIcmpHeader->icmp_cksum = static_cast<uint16_t>(~checksum);
    }


    // Save ICMP Header
    in_sentPing->icmpHeader = *sendIcmpHeader;

    // ICMP lookup target host
    addrInfoHint.ai_family   = nn::socket::Family::Af_Inet;
    addrInfoHint.ai_socktype = nn::socket::Type::Sock_Raw;
    addrInfoHint.ai_protocol = nn::socket::Protocol::IpProto_Icmp;
    addrInfoHint.ai_flags    = nn::socket::AddrInfoFlag::Ai_None;

    nn::socket::GetAddrInfo( in_dstAddr, nullptr, nullptr, &addrInfo);
    // ICMP Echo
    //
    // 0 = No Flags


    rc = nn::socket::SendTo(in_sentPing->sockfds, bufPtr, bufPtrLen, nn::socket::MsgFlag::Msg_None, addrInfo->ai_addr, addrInfo->ai_addrlen);
    ERROR_IF_AND_COUNT(rc < 0, "SENDTO ERROR: %d\n", nn::socket::GetLastError());

    // Free GETAddr Info
    nn::socket::FreeAddrInfo(addrInfo);

    // Free Work Pointer
    free( bufPtr );
    bufPtr = NULL;

    // Sleep and Save Sendtime
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    NN_LOG("Send %d\n", rc);

    in_sentPing->timeSent= nn::os::GetSystemTick();

    return( 0 );

out:

    if ( bufPtr != NULL )
    {
        free(bufPtr );
        bufPtr = NULL;
    }

    if ( addrInfo != NULL )
    {
        nn::socket::FreeAddrInfo(addrInfo);
        addrInfo = NULL;
    }

    if ( bufPtr != NULL )
    {
        free( bufPtr );
        bufPtr = NULL;
    }

    return( -1 );

} // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_IP_IMCP_ECHO)
{
    nn::socket::InAddr      myIp = {0};
    nn::socket::InAddr      myMask = {0};
    nn::socket::InAddr      myGw = {0};
    nn::socket::InAddr      myDns1 = {0};
    nn::socket::InAddr      myDns2 = {0};

    nn::socket::Ip          ipHeader;
    nn::socket::IcmpHdr     icmpHeader;

    struct SetSockOptIPPROTOPayload    sentPing;

    int               rc, bytesRead, idx;
    int64_t           unused1;
    unsigned char *   bufPtr;
    size_t            bufPtrSize = 0;
    int64_t           best, worst, avg[10];

    uint64_t          finalAvg = 0;

    NN_UNUSED(unused1);
    NN_UNUSED(bytesRead);

    ////////////
    // Ask NIFM for (DHCP) Informatiom
    ////////////

    nn::nifm::GetCurrentIpConfigInfo(&myIp, &myMask, &myGw, &myDns1, &myDns2);

    {
        NN_LOG( "NIFM: My IP Address: %d.%d.%d.%d\n",
                    static_cast<uint8_t>( myIp.S_addr ),
                    static_cast<uint8_t>( myIp.S_addr >> 8 ),
                    static_cast<uint8_t>( myIp.S_addr >> 16 ),
                    static_cast<uint8_t>( myIp.S_addr >> 24 ) );
    }

    best  = 100000000000;
    worst = 0;

    for( idx = 0; idx < 10; idx++ )
    {
        memset(&sentPing, 0, sizeof(sentPing));
        memset(&ipHeader, 0, sizeof(ipHeader));
        bufPtrSize = 0;
        rc = SetSockOpt_Send_Ping( myIp, "10.1.1.2", &sentPing, idx );  // idx == IP:TTL
        if ( rc < 0 )
        {
            NN_LOG( "Send Ping failed!  errno: %d\n", nn::socket::GetLastError() );
            return;
        }
        int totalLen = 0;

        // Read Packet..
        for( ; ; )
        {
            if ((nn::os::GetSystemTick() - sentPing.timeSent).ToTimeSpan().GetSeconds() > 5)
            {
                NN_LOG( "Timed out!\n" );
                goto out;
            }

            // Read IP Header..
            if ( bufPtrSize == 0 )
            {
                // Read IP Header..
                int recievedLength = nn::socket::Recv( sentPing.sockfds, &ipHeader, sizeof(ipHeader), nn::socket::MsgFlag::Msg_DontWait | nn::socket::MsgFlag::Msg_Peek);
                if ( recievedLength != sizeof( ipHeader ) )
                {
                     NN_LOG( "Waiting for IPv4 Header!\n" );
                     nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100) );
                     continue;
                }
                bufPtrSize = ipHeader.ip_len;
                bufPtr     = (unsigned char *) malloc(bufPtrSize);
            }
            else
            {
                totalLen += nn::socket::Recv(sentPing.sockfds, bufPtr + totalLen, bufPtrSize - totalLen, nn::socket::MsgFlag::Msg_DontWait);
            }
            if ( static_cast<nn::socket::Protocol>(ipHeader.ip_p) == nn::socket::Protocol::IpProto_Icmp && totalLen == bufPtrSize)
            {
                NN_LOG( "[IPv4]: I have an ICMP response from: [%d.%d.%d.%d]!\n",
                    static_cast<uint8_t>( ipHeader.ip_src.S_addr ),
                    static_cast<uint8_t>( ipHeader.ip_src.S_addr >> 8 ),
                    static_cast<uint8_t>( ipHeader.ip_src.S_addr >> 16 ),
                    static_cast<uint8_t>( ipHeader.ip_src.S_addr >> 24 ) );
                break;
            }
        }

        // Calculate Round Trip Time (RTT)
        int rtt_millis = (nn::os::GetSystemTick() - sentPing.timeSent).ToTimeSpan().GetMilliSeconds();
        NN_LOG( "RTT = %lld ms\n", rtt_millis );

        // Point at ICMP header
        icmpHeader = *reinterpret_cast<nn::socket::IcmpHdr*>(bufPtr + ipHeader.ip_hl * 4);
        // Point at ICMP Data
        // uint8_t* receiveIcmpData = bufPtr + ipHeader.ip_hl;

        bool allIsWell = false;
        switch( icmpHeader.icmp_type )
        {
           case nn::socket::IcmpType::Icmp_Echo:          NN_LOG( "Received an Icmp_Echo, code: %d\n",  icmpHeader.icmp_code );
                                    break;

           case nn::socket::IcmpType::Icmp_EchoReply:     NN_LOG( "Received an Icmp_EchoReply!\n" );
                                    allIsWell = true;
                                    break;

           case nn::socket::IcmpType::Icmp_Unreach:       NN_LOG( "Received: Icmp_Unreach!\n" ); break;
           case nn::socket::IcmpType::Icmp_TimXceed:      NN_LOG( "Received: Icmp_TimXceed!\n" ); break;
           case nn::socket::IcmpType::Icmp_ParamProb:     NN_LOG( "Received: Icmp_ParamProb!\n" ); break;
           case nn::socket::IcmpType::Icmp_SourceQuench:  NN_LOG( "Received: Icmp_SourceQuench!\n" ); break;
           case nn::socket::IcmpType::Icmp_Redirect:      NN_LOG( "Received: Icmp_Redirect!\n" ); break;

           default:                 NN_LOG( "(DID NOT) receive an Icmp_EchoReply!  Got: %d\n", icmpHeader.icmp_type );
                                    break;
        }

        // Calculate Round Trip Time (RTT)
        avg[idx] = (nn::os::GetSystemTick() - sentPing.timeSent).ToTimeSpan().GetMilliSeconds();

        if ( avg[idx] > worst ) worst = avg[idx];
        if ( avg[idx] < best )  best  = avg[idx];

        nn::socket::Close(sentPing.sockfds);

        if (bufPtr != NULL)
        {
            free(bufPtr);
            bufPtr = NULL;
        }
    }

    finalAvg = 0;
    for( idx = 0; idx < 10; idx++ )
    {
        finalAvg += avg[idx];
    }

    NN_LOG( "RTT: Best: %lldms, Worst: %lldms, Avg: %lldms\n", best, worst, finalAvg / 10 );

    nTestsPassing++;

out:

    return;

} // NOLINT(impl/function_size)


TEST(ApiUnit,SetSockOpt_IP_BINDANY)
{
    nn::socket::SockAddrIn outAddr = { 0 };
    bool        isSuccess = true;
    int         udpSock = -1, rc = -1;
    int         optVal;
    nn::socket::SockLenT   optLen;

    ///////////////
    // Create a UDP Socket
    ///////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: I want to have Googles DNS Address!
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "8.8.8.8", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    optVal = 1;
    optLen = sizeof( optVal );
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_BindAny, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed to call nn::socket::Option::Ip_BindAny - errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&outAddr), sizeof( outAddr ) );
    ERROR_IF_AND_COUNT( rc < 0, "Bind failed trying to bind Googles DNS Address locally - nn::socket::Option::Ip_BindAny" );

    NN_LOG( "********************\n" );
    NN_LOG( "PASSED - Successfully bound Googles DNS Address\n" );
    NN_LOG( "********************\n" );

out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_IP_TOS)
{
    nn::socket::SockAddrIn outAddr = { 0 };
    bool        isSuccess = true;
    int         udpSock = -1, rc = -1;
    int         optVal;
    nn::socket::SockLenT   optLen;

    ///////////////
    // Create a UDP Socket
    ///////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: I want to use the Loopback Inferface
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ///////////////////
    // Test: nn::socket::IpTos_LowDelay
    ///////////////////

    optVal = static_cast<int>(nn::socket::IpTos::IpTos_LowDelay);
    optLen = sizeof( optVal );
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Tos, (void *) &optVal, optLen );
    ERROR_IF_AND_COUNT( rc < 0, "SetSockOpt failed to set nn::socket::Option::Ip_Tos: IpTos_LowDelay - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Set IpTos_LowDelay\n");


    ///////////////////
    // Test: nn::socket::IpTos_Throughput
    ///////////////////

    optVal = static_cast<int>(nn::socket::IpTos::IpTos_Throughput);
    optLen = sizeof( optVal );
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Tos, (void *) &optVal, optLen );
    ERROR_IF_AND_COUNT( rc < 0, "SetSockOpt failed to call nn::socket::Option::Ip_Tos: IpTos_Throughput - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Set IpTos_Throughput\n");


    ///////////////////
    // Test: nn::socket::IpTos_Reliability
    ///////////////////

    optVal = static_cast<int>(nn::socket::IpTos::IpTos_Reliability);
    optLen = sizeof( optVal );
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Tos, (void *) &optVal, optLen );
    ERROR_IF_AND_COUNT( rc < 0, "SetSockOpt failed to call nn::socket::Option::Ip_Tos: IpTos_Reliability - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Set IpTos_Reliability\n");


    ///////////////////
    // Test: nn::socket::IpTos_MinCost
    ///////////////////

    optVal = static_cast<int>(nn::socket::IpTos::IpTos_MinCost);
    optLen = sizeof( optVal );
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Tos, (void *) &optVal, optLen );
    ERROR_IF( rc < 0, "SetSockOpt failed to call nn::socket::Option::Ip_Tos: IpTos_MinCost - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Set IpTos_MinCost\n");


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }
}

// Anonymous namespace
namespace {

//////////////////////////////////////////
// $ dig google.com
//
// ; <<>> DiG 9.10.3-P4 <<>> google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 43551
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 1
//
// ;; OPT PSEUDOSECTION:
// ; EDNS: version: 0, flags:; udp: 4096
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
//
// ;; ANSWER SECTION:
// google.com.             255     IN      A       172.217.3.174
//
// ;; Query time: 4 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Wed Oct 26 09:05:02 PDT 2016
// ;; MSG SIZE  rcvd: 55
//
//////////////////////////////////////////

static const char GoogleDNSRequest[] =
{
0xaa, 0x1f, 0x01, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x29, 0x10, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00
};

}; // End - Anonymous namespace


TEST(ApiUnit,SetSockOpt_IP_ONESBCAST)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, targetAddr = { 0 };
    bool                    isSuccess = true;
    int                     udpSock = -1, rc = -1;
    int                     udpServSock = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;

    nn::socket::InAddr myIp = { 0 };
    nn::socket::InAddr myMask = { 0 };
    nn::socket::InAddr myGw = { 0 };
    nn::socket::InAddr myDns1 = { 0 };
    nn::socket::InAddr myDns2 = { 0 };
    nn::socket::SockAddrIn remoteHost = { 0 }, servAddr = { 0 };
    nn::socket::TimeVal myTime = { 0 };
    nn::socket::SockLenT optlen;
    int enableFlag = 1;
    char dnsRequest[512];

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Server Sock
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Server - Local Address
    memset( &servAddr, 0, sizeof(servAddr));
    servAddr.sin_port        = nn::socket::InetHtons( 9058 );
    servAddr.sin_family      = nn::socket::Family::Af_Inet;

    NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
    servAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Server - Don't wait forever..
    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo,(void *) &myTime, optlen );
    ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed - Errno: %d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServSock, reinterpret_cast<nn::socket::SockAddr *>(&servAddr), sizeof( servAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // UDP Client - Create
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Client - May send Broadcast Packets
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &enableFlag, sizeof(enableFlag));

    // This converts a directed broadcast to a limited broadcast.
    // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
    // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));
    ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::Ip_OnesBcast failed - Errno: %d\n", nn::socket::GetLastError() );

    // UDP Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9057 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Client - Local Bind
    rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // UDP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port     = nn::socket::InetHtons( 9058 );   // Point to UDP Server
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    // UDP - Target IP: LAN LIMITED BROADCAST ADDRESS
    targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
    NN_LOG( "Directed Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

    // Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSock,reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

    // Send Directed data from NX
    rc = nn::socket::Write( udpSock, GoogleDNSRequest, sizeof( GoogleDNSRequest ) );

    ERROR_IF_AND_COUNT( rc < 0,
    "[UDP] Write() Directed Broadcast Address failed but should have succeeded!  Errno:\n", nn::socket::GetLastError());

    NN_LOG( "Send - Sent %d bytes\n", rc );

    // For()ever
    for( ; ; )
    {
        // Recieve DNS Response Messages
        optlen = sizeof( remoteHost );
        rc = nn::socket::RecvFrom( udpServSock, dnsRequest, sizeof( dnsRequest ), nn::socket::MsgFlag::Msg_None,
                                       reinterpret_cast<nn::socket::SockAddr *>(&remoteHost), &optlen );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "Finished waiting for DNS Responses!\n" );
                break;
            }

            NN_LOG( "Failed calling RecvFrom() - Errno: %d\n", myError );
        }

        NN_LOG( "DNS Request from Host: [%s:%d]\n", nn::socket::InetNtoa( remoteHost.sin_addr ),
                                                        nn::socket::InetNtohs( remoteHost.sin_port ) );
    }

out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }
}


TEST(ApiUnit,SetSockOpt_IP_OPTIONS)
{
    nn::socket::SockAddrIn localAddr = { 0 }, targetAddr = { 0 };
    int                udpSock = -1, udpServSock = -1, rc = -1;
    int                idx;
    bool               isSuccess = true;
    unsigned char      setOptions[16], getOptions[20];
    nn::socket::SockLenT          optLen;


    //////////////////
    // UDP Sock
    //////////////////

    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9013 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Local Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );


    // Client - Tagrget Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9012 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Target Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client - Connect
    rc = nn::socket::Connect( udpServSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "Connect failed for local socket, but should have succeeded!\n" );


    //////////////////
    // UDP Sock
    //////////////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9012 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Local Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );


    // Client - Tagrget Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9013 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client - Target Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client - Connect
    rc = nn::socket::Connect( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "Connect failed for local socket, but should have succeeded!\n" );


    /////////////////
    // UDP Client - Create New Options
    /////////////////

    memset( setOptions, 0, sizeof( setOptions ) );
    setOptions[0] = 7;         // Record Route
    setOptions[1] = 15;        // Option Length:  3x control bytes + 3 x 4 (12) IP Address slots
    setOptions[2] = 4;         // Point at first IP Address slot

    NN_LOG( "UDP Client Set IP Options [" );
    for( idx = 0; idx < sizeof( setOptions); idx++ )
    {
        NN_LOG( "0x%02x, ", setOptions[ idx ] );
    }
    NN_LOG( "]\n" );

    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Options, (void *) &setOptions[0], sizeof( setOptions ) );
    ERROR_IF( rc < 0, "Failed to Set IP Options (Record Route)  - Errno: %d\n",  nn::socket::GetLastError() );

    ////////////////////
    // UDP Client - Sends Data to Server
    ////////////////////

    rc = nn::socket::Send( udpSock, "X", 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc != 1, "Failed to send 1 byte to UDP Server!\n" );


    ////////////////////
    // UDP Client - Recieve 1 byte
    ////////////////////

    rc = nn::socket::Recv( udpServSock, getOptions, sizeof( getOptions ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc != 1, "Failed to Recieve 1 byte from UDP Client!\n" );


    /////////////////
    // UDP Client - Get Options
    /////////////////

    optLen = sizeof( getOptions );  // +4 for returned IPv4 Addresss (pre-appended)
    rc = nn::socket::GetSockOpt( udpSock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Options, &getOptions, &optLen );
    ERROR_IF( rc < 0, "Failed to Get IP Options from Socket - Errno: %d\n",  nn::socket::GetLastError() );

    ERROR_IF_AND_COUNT( optLen != sizeof( getOptions ), "Returned Options Length %d does not equal set Option Len: %d!\n",
                                             optLen, sizeof( getOptions ) );
    NN_LOG( "UDP Client Get IP Options [" );
    for( idx = 0; idx < optLen; idx++ )
    {
        NN_LOG( "0x%02x, ", getOptions[ idx ] );
    }
    NN_LOG( "]\n" );


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }
}

TEST(ApiUnit,SetSockOpt_IP_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,SetSockOpt_IP_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
