﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Send_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Send_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 }, targetAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( nn::socket::SockAddr );
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1;
    int                     rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    ssize_t                 rc1;
    bool                    isSuccess = true;

    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  ||
              ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        outAddrLen = sizeof( nn::socket::SockAddr );
        rc1 = nn::socket::Send( idx, "X", 1, nn::socket::MsgFlag::Msg_None );
        if ( rc1 > 0 )
        {
            NN_LOG( "\nSend() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }

        if ( rc1 < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nSend() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nSend() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [TCP] - NULL Send Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc1 = nn::socket::Send( tcpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 > -1, "Send() succeeded but should have failed! [TCP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "Send() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );


    ////////////
    // [UDP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [UDP] - NULL Send Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc1 = nn::socket::Send( udpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 > -1, "[UDP]: Send() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "Send() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );

    ////////////
    // [UDP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (No Address Len)\n" );

    outAddrLen = 0;
    rc1 = nn::socket::Send( udpSock, "X", 0, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != 0, "[UDP] Send() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Connect established destination - Same Address
    ///////////

    NN_LOG( "Testing: [UDP]- Connect() establishes destination - Same Address\n" );

    outAddrLen = 0;
    rc1 = nn::socket::Send( udpSock, "X", 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 < 0, "[UDP] Send() failed but should have Suceeded!  Errno: %d\n", nn::socket::GetLastError() );


WRAP_FAILING_TEST( "SIGLO-58216", "Socket API: [Win]: Send() over maximum transmit size does not return nn::socket::Errno::EMsgSize" )
{
    ////////////
    // [UDP] - Exceed UDP Maximum Transmit size - nn::socket::Errno::EMsgSize
    ////////////

    NN_LOG( "Testing: [UDP]- Exceed Maximum Transmit Size - nn::socket::Errno::EMsgSize\n" );

    outAddrLen = 131070;  // Something greater than 65535
    rc1 = nn::socket::Send( udpSock, "X", outAddrLen, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 > -1, "[UDP] Send() Suceeded but should have failed!\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "[UDP] Send() expected errno: nn::socket::Errno::EMsgSize, actually got Errno: %d\n", myError );

} // End Wrap Failing Test


#if 0

    ////////////
    // [UDP]
    //
    // ECONNNREFUSED - The socket received an ICMP destination unreachable from the last
    // message sent.
    ////////////

    NN_LOG( "Testing: [UDP]- Disconnected UDP Server - nn::socket::Errno::EConnRefused\n" );

    {
        // UDP Server Sock
        udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9050, "127.0.0.1", 9060 );
        ERROR_IF_AND_COUNT( udpServSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // UDP Client - Close
        nn::socket::Close( udpSock );

        // UDP Client - Connect to Server
        udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9060, "127.0.0.1", 9050 );
        ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

        // Write from Client to Server
        nn::socket::Send( udpSock, "x", 1, nn::socket::MsgFlag::Msg_None);

        // Close UDP Server
        nn::socket::Close( udpServSock );
        udpServSock = -1;

        // Client keeps writing - but expected to FAIL
        for( idx = 0; idx < 10; idx++ )
        {
            rc1 = nn::socket::Send( udpSock, "X", 1, nn::socket::MsgFlag::Msg_None );
            if ( rc1 < 0 )
            {
                myError = nn::socket::GetLastError();
                if ( myError == nn::socket::Errno::EConnRefused )
                {
                    NN_LOG( "PASSED!  Send failed with nn::socket::Errno::EConnRefused!\n" );
                    break;
                }

                NN_LOG( "Send failed with errno: %d\n", myError );
            }

            NN_LOG( "Good Send to closed UDP socket...\n" );
        }

        if ( idx >= 10 )
        {
            NN_LOG( "FAILED!  Send succeeded 10 times but should have failed with nn::socket::Errno::EConnRefused!\n" );
            goto out;
        }
    }

#endif

WRAP_FAILING_TEST( "SIGLO-58232", "Socket API: [Win] nn::socket::MsgFlag::Msg_DontRoute is not working" )
{

    ////////////
    // [UDP]
    //
    // nn::socket::MsgFlag::Msg_DontRoute - If an interface is not directly connected to the NX don't follow
    // the default route.
    ////////////

    {
        nn::socket::InAddr myIp = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };


        NN_LOG( "Testing: [UDP] - nn::socket::MsgFlag::Msg_DontRoute - (Errno: nn::socket::Errno::EHostDown, nn::socket::Errno::ENetDown, etc)\n" );

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // UDP Client - Close
        if ( udpSock > -1 )
        {
            nn::socket::Close( udpSock );
            udpSock = -1;
        }

        // UDP Client - Create
        udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

        // UDP Client - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port     = nn::socket::InetHtons( 9004 );
        localAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Client - IP
        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // UDP Client - Local Bind
        rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

        // UDP - Target Address
        memset( &targetAddr, 0, sizeof(targetAddr));
        targetAddr.sin_port     = nn::socket::InetHtons( 53 );   // DNS == domain == 53
        targetAddr.sin_family   = nn::socket::Family::Af_Inet;

        // UDP - Target IP: (E-ROOT)
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &targetAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // Call Connect to Bind Servers IP as Peer
        rc = nn::socket::Connect( udpSock,(nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
        ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

        // Send test data which is not (directly) routable from NX
        rc1 = nn::socket::Send( udpSock, "Test Data", strlen( "Test Data" ), nn::socket::MsgFlag::Msg_DontRoute );
        ERROR_IF_AND_COUNT( rc1 > 0, "[UDP] Send() succeeded but should have failed!\n" );

        myError = nn::socket::GetLastError();
        switch( myError )
        {
        case nn::socket::Errno::EHostUnreach:
        case nn::socket::Errno::EHostDown:
        case nn::socket::Errno::ENetDown:
        case nn::socket::Errno::EConnRefused:

            break;   // All is well

        default:
            NN_LOG( "Unexpected Errno: %d received, expected one of the 4x legal network ones\n", myError );
            goto out;

        }
    }

} // End Wrap Failing Test

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

}   // NOLINT(impl/function_size)



TEST(ApiUnit,Win_Send_Blocking_Test)
{
    nn::socket::TimeVal myTime = { 0 };
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1, rc = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;

    bool                isSuccess = true, rval;
    nn::socket::SockLenT    optlen;
    uint8_t             noiseVal = 1;
    unsigned char *     buf1 = NULL;
    ssize_t             rc1 = 0;

    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( cliSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    ////////////
    // Make Noise Buffer
    ///////////

    rc = NATF::API::COMMON::MakeRandomNoise( &buf1, 32767, noiseVal );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );


    ////////////
    // Establish Send Timeout
    ///////////

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( cliSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo,(void *) &myTime, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed - Errno: %d\n", nn::socket::GetLastError() );

    for( ; ; )
    {
        NN_LOG( "[TCP]: Sending 32767\n" );

        rc1 = nn::socket::Send( cliSock, buf1, 32767, nn::socket::MsgFlag::Msg_None );
        if ( rc1 < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                NN_LOG( "================================\n" );
                NN_LOG( "PASSED -- SNDTIMEO unlocked after 5 seconds\n" );
                NN_LOG( "================================\n" );

                break;
            }

            NN_LOG( "Send() failed with Errno: %d\n", myError );
        }
    }

out:

    if ( buf1 != NULL )
    {
        free( buf1 );
        buf1 = NULL;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}

#if 0

TEST( ApiUnit,Win_Send_SO_OOB)
{

    int              cliSock = -1, listenSock = -1, acceptedSock = -1;
    int              rc = -1;
    bool             isSuccess = true, isOOB = false;
    int              disableFlag = 0, count = 0, atMark = 0, myError = 0;
    nn::socket::SockLenT        optlen = sizeof( disableFlag );
    char             buf1[1024];
    ssize_t          rc1 = 0;

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns off OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &disableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns on NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    NN_LOG( "Send(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc1 = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : 0 ) );
        if ( rc1 < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc1 = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc1 < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

} // NOLINT(impl/function_size)

#endif


TEST( ApiUnit,Win_Send_SO_OOBINLINE)
{

WRAP_FAILING_TEST( "SIGLO-58323", "Socket API: [Win] - Send() - OOBLINE is always at the mark" )
{
    int                 cliSock = -1, listenSock = -1, acceptedSock = -1;
    int                 rc = -1;
    bool                isSuccess = true, isOOB = false;
    int                 enableFlag = 1, count = 0, atMark = 0;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT        optlen = sizeof( enableFlag );
    char                buf1[1024] = {};
    ssize_t             rc1 = 0;

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns ON OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &enableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns ON NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "Send(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "Send() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    NN_LOG( "Send(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc1 = nn::socket::Send( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc1 != (ssize_t) strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc1 = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ) );
        if ( rc1 < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc1 = nn::socket::Recv( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc1 < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == (int) strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}  // End Wrap Failing Test

}  // NOLINT(impl/function_size)

TEST(ApiUnit,Win_Send_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
