﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_ConstantsPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions
#include "Unit/testNet_ApiUnitCommon.h"

// // Work around for SIGLONTD-11262
#define _ALIGNBYTES (sizeof(int) - 1) //NOLINT(preprocessor/const)
#define _ALIGN(p)   (((unsigned)(p) + _ALIGNBYTES) & ~_ALIGNBYTES) //NOLINT(preprocessor/const)

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************
namespace
{
    const int            MyThreadPriority = 10;       // Thread Priority
    const int            MyMaxNumOfSockets = 3;       // Max test sockets

    ThreadedController * g_pThreadedControllerTest = NULL;
    int                  g_EchoServerThreadId = -1;
    EchoServer *         g_pEchoServer = NULL;

    void
    StartEchoServer( void * inArg )
    {
        // Tell Log we are done
        NN_LOG( "[Echo Server]: Thread starting\n" );

        EchoServer * echoServer = reinterpret_cast<EchoServer *>(inArg);

        // Start the DNS Server Main Loop
        echoServer->Start( nullptr );

        // Tell Log we are done
        NN_LOG( "[Echo Server]: Thread exiting\n" );
    }


    bool
    InitializeTesting()
    {
        bool        isSuccess = true;

        NN_LOG( "In\n\n" );

        ///////////////////////////
        //// Test Counts: Initialize
        ///////////////////////////

        INITIALIZE_TEST_COUNTS;


        ///////////////////////////
        //// NIFM Library: Initialize
        ///////////////////////////

        ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

        // NIFM: Network Interface is Online
        NN_LOG( "====================================\n" );
        NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
        NN_LOG( "====================================\n" );


        ///////////////////////////
        ////  Allocate Echo Server
        ///////////////////////////

        g_pEchoServer = new EchoServer();
        if ( g_pEchoServer == NULL )
        {
            NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
            goto out;
        }

        NN_LOG( "===> Starting [Echo Server]\n" );

        // Create: ThreadedTest
        g_pThreadedControllerTest = new ThreadedController();
        if ( g_pThreadedControllerTest == NULL )
        {
            NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
            goto out;
        }

        // Allocate 1 slot
        g_pThreadedControllerTest->Initialize( MyMaxNumOfSockets );
        // Create a thread to run the Server in

        g_EchoServerThreadId = g_pThreadedControllerTest->CreateThread( &StartEchoServer, static_cast<void *>(g_pEchoServer), MyThreadPriority );
        if (  g_EchoServerThreadId < 0 )
        {
            NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
            goto out;
        }

        NN_LOG( "Successfully started Echo Server" );

        // Give Thread time to start
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        return( true );

    out:

        // If the loopback DNS server is allocated
        if ( g_pThreadedControllerTest != NULL )
        {
            delete g_pThreadedControllerTest;
            g_pThreadedControllerTest = NULL;
        }

        // If the loopback DNS server is allocated
        if ( g_pEchoServer != NULL )
        {
            delete g_pEchoServer;
            g_pEchoServer = NULL;
        }

        return( false );
    }


    bool
    TeardownTesting()
    {
         bool    isSuccess = true;

        ///////////////////
        //// Stop DNS Server
        ////////////////////

        NN_LOG( "Stopping Echo Server\n" );

        // If the echo server is allocated
        if ( g_pEchoServer != NULL )
        {
            g_pEchoServer->Stop();
        }

        // Wait for Echo Server thread to die
        g_pThreadedControllerTest->WaitThread( g_EchoServerThreadId );

        // Destroy Echo Server Thread
        g_pThreadedControllerTest->DestroyThread( g_EchoServerThreadId );

        // Delete Echo Server
        delete g_pEchoServer;
        g_pEchoServer = NULL;

        // Delete Threaded Test
        delete g_pThreadedControllerTest;
        g_pThreadedControllerTest = NULL;
        g_EchoServerThreadId = -1;

        NN_LOG( "Echo Server successfully stopped!\n" );


        ////////////////////
        ////  Stop Testing
        ////////////////////

        ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

        ////////////////////
        ////  Print Test Counts
        ////////////////////

        PRINT_TEST_COUNTS;

        EXPECT_EQ( isSuccess, true );

        NN_LOG( "Out\n\n" );

        return( true );

    out:

        return( false );
    }

    const int MaxMessages = nn::socket::MaxNumMessages;
    const int MaxMessageSize = nn::socket::MaxMessageSize;
    const int MaxMessagesBytes = nn::socket::MaxMessagesBytes;
    const int OverloadBufDifference = 50;
}

////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,SendMMsg_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,SendMMsg_Various_Tests)
{
    bool                    isSuccess = true;
    int                     tcpSocket = -1, udpSocket = -1, broadcastSocket = -1;
    int                     tcpServerSocket = -1, udpServerSocket = -1, udpServerSocket1 = -1, acceptedSocket = -1;
    int                     rc = -1, idx = 0, msgIdx = 0, enableFlag = 1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    uint64_t                unusedReadWouldBlock = 0;
    nn::socket::SockLenT    optlen = 0;
    nn::socket::SockAddrIn  localAddr = { 0 },
                            targetAddr = { 0 },
                            serverAddr = { 0 },
                            serverAddr1 = { 0 },
                            *pReceivedNames = new nn::socket::SockAddrIn[MaxMessages]();

    nn::socket::SockLenT    targetAddrLen = sizeof( targetAddr );

    nn::socket::MMsgHdr     *pSendMsgs = new nn::socket::MMsgHdr[MaxMessages](),
                            *pSetNameMsg = new nn::socket::MMsgHdr[MaxMessages](),
                            *pNullIovecMsg = new nn::socket::MMsgHdr[MaxMessages](),
                            *pUniqueMsgs = new nn::socket::MMsgHdr[2](),
                            *pBigMsgs = new nn::socket::MMsgHdr[MaxMessages](),
                            *pReceiveMsgs = new nn::socket::MMsgHdr[MaxMessages]();

    nn::socket::Iovec       *pIov1 = new nn::socket::Iovec[2](),
                            *pIov2 = new nn::socket::Iovec[2](),
                            *pReceiveIov = new nn::socket::Iovec[MaxMessages](),
                            *pBigIovs = new nn::socket::Iovec[MaxMessages](),
                            *pBigIov = new nn::socket::Iovec[1]();

    char                    receiveBuf[MaxMessageSize + OverloadBufDifference] = { 0 },
                            addrbuf[16] = { '\0' };

    static char             mmsgReceiveBufs[MaxMessages][1400] = { { 0 } },
                            bigBufs[MaxMessages][MaxMessageSize + OverloadBufDifference] = { { 0 } },
                            bigBuf[MaxMessagesBytes + OverloadBufDifference] = { 0 };

    nn::socket::InAddr      myIp = { 0 }, myMask = { 0 }, myGw = { 0 }, myDns1 = { 0 }, myDns2 = { 0 };
    nn::socket::TimeVal     myTime = { 0 };
    nn::TimeSpan            timeout = nn::TimeSpan::FromSeconds(5);

    const char * msgs[]
    {
        "sendmmsg",
        "test",
        "sendmmsgtest",
        "testsendmmsg"
    };

    pIov1[0].iov_base = const_cast<char *>( msgs[0] );
    pIov1[0].iov_len = strlen(static_cast<const char *>(pIov1[0].iov_base));
    pIov1[1].iov_base = const_cast<char *>( msgs[1] );
    pIov1[1].iov_len = strlen(static_cast<const char *>(pIov1[1].iov_base));

    pIov2[0].iov_base = const_cast<char *>( msgs[1] );
    pIov2[0].iov_len = strlen(static_cast<const char *>(pIov2[0].iov_base));
    pIov2[1].iov_base = const_cast<char *>( msgs[0] );
    pIov2[1].iov_len = strlen(static_cast<const char *>(pIov2[1].iov_base));

    // Fill Message Vector
    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
    {
        pSendMsgs[msgIdx].msg_hdr.msg_iov = pIov1;
        pSendMsgs[msgIdx].msg_hdr.msg_iovlen = 2;
    }


    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_pEchoServer->serverTCP )  ||
              ( idx == g_pEchoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::SendMMsg( idx, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
        myError = nn::socket::GetLastError();
        if ( rc < 0 )
        {

            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nSendMMsg() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno=<%d>\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nSendMMsg() succeeded but fds: %d. Expected to fail.\n" );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf failed\n" );
    isSuccess = true;


    ////////////
    // Make TCP Connection
    ///////////

    tcpSocket = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSocket < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSocket = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSocket < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Message Vector
    ///////////

    NN_LOG( "Testing: [TCP] - Null Message Vector - nn::socket::Errno::EInval\n" );

    rc = nn::socket::SendMMsg( tcpSocket, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: SendMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "SendMMsg() expected nn::socket::Errno::EInval. Errno:%d\n", myError );

    ////////////
    // [UDP] - Test NULL Message Vector
    ///////////

    NN_LOG( "Testing: [UDP] - Null Message Vector - nn::socket::Errno::EInval\n" );

    rc = nn::socket::SendMMsg( udpSocket, nullptr, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: SendMMsg() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "SendMMsg() expected nn::socket::Errno::EInval. Errno:%d\n", myError );


    ////////////
    // [TCP] - Test No Message Vector Length
    ///////////

    NN_LOG( "Testing: [TCP] - No Message Vector Length - nn::socket::Errno::EMsgSize\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, 0, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: SendMMsg() succeded but should have failed.\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // [UDP] - Test No Message Vector Length
    ///////////

    NN_LOG( "Testing: [UDP] - No message Vector Length - nn::socket::Errno::EMsgSize\n" );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, 0, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: SendMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // [TCP] - Test Overload Message Vector Length
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Message Vector Length - nn::socket::Errno::EMsgSize\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages + 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[TCP]: SendMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // [UDP] - Test Overload Message Vector Length
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Message Vector Length - nn::socket::Errno::EMsgSize\n" );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages + 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP]: SendMMsg() succeded but should have failed." );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // Fill nn::socket::MMsgHdr with overloaded messsage and total bytes allowed
    ///////////

    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
    {
        memset( &bigBufs[msgIdx], 78 + msgIdx, sizeof(bigBufs[msgIdx]) );
        pBigIovs[msgIdx].iov_base = bigBufs[msgIdx];
        pBigIovs[msgIdx].iov_len = sizeof(bigBufs[msgIdx]);

        pBigMsgs[msgIdx].msg_hdr.msg_iov = &pBigIovs[msgIdx];
        pBigMsgs[msgIdx].msg_hdr.msg_iovlen = 1;
    }


    ////////////
    // [TCP] - Test Overload Message bytes - No limit to sending
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Message bytes\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 1, "SendMMsg() failed to send 1 message. Result:%d\n", rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    rc = NATF::API::COMMON::ReadData( tcpSocket, reinterpret_cast<unsigned char *>(receiveBuf), sizeof(receiveBuf), unusedReadWouldBlock );
    ERROR_IF( rc != sizeof(bigBufs[0]), "ReadData() returned %d bytes. Expected %d bytes.\n", rc, sizeof(bigBufs[0]) );


    ////////////
    // [UDP] - Test Overload Message bytes - No limit to sending
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Message bytes\n" );

    rc = nn::socket::SendMMsg( udpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 1, "SendMMsg() failed to send 1 message. Result:%d\n", rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    rc = NATF::API::COMMON::ReadData( udpSocket, reinterpret_cast<unsigned char *>(receiveBuf), sizeof(receiveBuf), unusedReadWouldBlock );
    ERROR_IF( rc != sizeof(bigBufs[0]), "ReadData() returned %d bytes. Expected %d bytes.\n", rc, sizeof(bigBufs[0]) );


WRAP_FAILING_TEST( "SIGLO-57733", "Aborts when writing past the end of the buffer" )
{

    ////////////
    // [TCP] - Test Overload Total Message bytes with max messages - Max is 48000
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Total Message bytes - Max Messages\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pBigMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > 0, "SendMMsg() succeeded. Expected to Fail! Returned: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg() expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // [UDP] - Test Overload Total Message bytes with max messages - Max is 48000
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Total Message bytes - Max Messages\n" );

    rc = nn::socket::SendMMsg( udpSocket, pBigMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > 0, "SendMMsg() succeeded. Expected to Fail! Returned: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg() expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );

}


    ////////////
    // Initialize single message with one big buffer
    ///////////

    memset( &bigBuf[0], 78, sizeof(bigBuf) );
    pBigIov[0].iov_base = bigBuf;
    pBigIov[0].iov_len = sizeof(bigBuf);

    pBigMsgs[0].msg_hdr.msg_iov = pBigIov;
    pBigMsgs[0].msg_hdr.msg_iovlen = 1;


WRAP_FAILING_TEST( "SIGLO-58011", "TCP socket passes but UDP fails with nn::socket::Errno::EMsgSize." )
{

    ////////////
    // [TCP] - Test Overload Total Message bytes with one message - Max is 48000
    ///////////

    NN_LOG( "Testing: [TCP] - Overload Total Message bytes - One message\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > 0, "SendMMsg() succeeded. Expected to Fail! Returned: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg() expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );

}

    ////////////
    // [UDP] - Test Overload Total Message bytes with one message - Max is 48000
    ///////////

    NN_LOG( "Testing: [UDP] - Overload Total Message bytes - One message\n" );

    rc = nn::socket::SendMMsg( udpSocket, pBigMsgs, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > 0, "SendMMsg() succeeded. Expected to Fail! Returned: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EMsgSize, "SendMMsg() expected nn::socket::Errno::EMsgSize. Errno:%d\n", myError );


    ////////////
    // [TCP] - Connect established destination - Same Address
    ///////////

    NN_LOG( "Testing: [TCP] -- Connect() establishes destination - Same Address\n" );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT ( rc != MaxMessages, "[TCP] SendMMsg() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    memset( receiveBuf, 0, sizeof(receiveBuf) );

    // Recv used due to SIGLONTD-11082 - TCP sockets causes memory access error with RecvMMsg()
    rc = NATF::API::COMMON::ReadData( tcpSocket, reinterpret_cast<unsigned char *>(receiveBuf), sizeof(receiveBuf), unusedReadWouldBlock );
    ERROR_IF( rc != MaxMessages * strlen(msgs[2]), "ReadData() returned %d bytes. Expected: %d bytes.\n", rc, MaxMessages * strlen(msgs[2]) );
    NN_LOG( "\tRecieved data from server: %.*s\n", rc, receiveBuf );


    ////////////
    // [UDP] - Connect established destination - Same Address
    ///////////

    NN_LOG( "Testing: [UDP] - Connect() establishes destination - Same Address\n" );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT ( rc != MaxMessages, "[UDP] SendMMsg() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( idx = 0; idx < MaxMessages; idx++)
    {
        pReceiveIov[idx].iov_base              = mmsgReceiveBufs[idx];
        pReceiveIov[idx].iov_len               = 1400;
        pReceiveMsgs[idx].msg_hdr.msg_iov     = &pReceiveIov[idx];
        pReceiveMsgs[idx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[idx].msg_hdr.msg_name    = &pReceivedNames[idx];
        pReceiveMsgs[idx].msg_hdr.msg_namelen = sizeof(pReceivedNames[idx]);
    }

    rc = nn::socket::RecvMMsg( udpSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF( rc != MaxMessages, "RecvMMsg() returned %d messages. Expected %d\n", rc, MaxMessages );


    ////////////
    // [TCP] - Connect established destination - ignores msg_name specification
    ///////////

    NN_LOG( "Testing: [TCP] - Connect() establishes destination - but set different destination - nn::socket::Errno::EIsConn\n" );

    // TCP Server: Target Address
    memset(&serverAddr, 0, sizeof(serverAddr));
    serverAddr.sin_port        = nn::socket::InetHtons( 9020 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // TCP Client - IP
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );
    NN_LOG( "\tMy IP Address: %s\n", nn::socket::InetNtoa(myIp) );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    memset(&pSetNameMsg[0], 0, sizeof(pSetNameMsg[0]));
    pSetNameMsg[0].msg_hdr.msg_name = static_cast<void *>(&serverAddr);
    pSetNameMsg[0].msg_hdr.msg_namelen = sizeof(serverAddr);
    pSetNameMsg[0].msg_hdr.msg_iov = pIov1;
    pSetNameMsg[0].msg_hdr.msg_iovlen = 2;

    rc = nn::socket::SendMMsg( tcpSocket, pSetNameMsg, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "[TCP] SendMMsg() failed but should have Succeeded! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg() sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    // Currently socket connected using Connect() takes precedence
    rc = nn::socket::Recv( tcpSocket, receiveBuf, sizeof(receiveBuf), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "Recv() failed. Message was sent using msg_name address.\n" );
    NN_LOG( "\tRecv() passed, existing loop back connection took precedence. Message: %.*s\n", rc, receiveBuf );


    ////////////
    // [UDP] - Connect established destination - nn::socket::Errno::EIsConn
    ///////////

    NN_LOG( "Testing: [UDP]- Connect() establishes destination - but set different destination - nn::socket::Errno::EIsConn\n" );

    // Client: Target Address
    memset(&serverAddr, 0, sizeof(serverAddr));
    serverAddr.sin_port        = nn::socket::InetHtons( 9020 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP - Client - IP
    NN_LOG( "\tMy IP Address: %s\n", nn::socket::InetNtoa(myIp) );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    memset(&pSetNameMsg[0], 0, sizeof(pSetNameMsg[0]));
    pSetNameMsg[0].msg_hdr.msg_name = static_cast<void *>(&serverAddr);
    pSetNameMsg[0].msg_hdr.msg_namelen = sizeof(serverAddr);
    pSetNameMsg[0].msg_hdr.msg_iov = pIov1;
    pSetNameMsg[0].msg_hdr.msg_iovlen = 2;

    rc = nn::socket::SendMMsg( udpSocket, pSetNameMsg, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1 , "[UDP] SendMMsg() succeded but should have Failed!" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EIsConn, "SendMMsg() expected to fail with nn::socket::Errno::EIsConn. Errno:%d", myError );


    ////////////
    // [TCP] - Test Direct Send to Peer on Lan Iface
    // Connected sockets without setting msghdr_name - VALID
    ////////////

    NN_LOG( "Testing: [TCP] - SendMMsg Direct Send to Peer on Lan Iface - Valid\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // TCP server Sock
    tcpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServerSocket < 0, "Failed to create TCP Server socket. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Address
    memset(&serverAddr, 0, sizeof(serverAddr));
    serverAddr.sin_port = nn::socket::InetHtons( 8058 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // TCP Server = Don't wait forever
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof(myTime);

    rc = nn::socket::SetSockOpt( tcpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Bind
    rc = nn::socket::Bind( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // TCP Server - Listen
    rc = nn::socket::Listen( tcpServerSocket, 5 );
    ERROR_IF( rc < 0, "Listen() failed! Errno:%d\n", nn::socket::GetLastError() );

    // TCP Client - Close
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Client - Create
    tcpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

    // TCP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port = nn::socket::InetHtons( 8058 ); // Point to UDP Server
    targetAddr.sin_family = nn::socket::Family::Af_Inet;

    // TCP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr = myIp.S_addr;
    NN_LOG( "\tConnected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // TCP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( tcpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // TCP Server - Accept client connection
    acceptedSocket = nn::socket::Accept( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), &targetAddrLen );
    ERROR_IF( acceptedSocket < 0, "Accept() expected to Succeed. Failed with Errno:%d \n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg( tcpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    // Recv used due to SIGLONTD-11082 - TCP sockets causes memory access error with RecvMMsg()
    rc = nn::socket::Recv( acceptedSocket, receiveBuf, sizeof(msgs[2]), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc < 0, "Recv() Failed but expected to Succeed - Errno: %d\n", myError );

    NN_LOG( "\tLast message from: %.*s\n", rc, receiveBuf );

    // TCP Client for broadcast testing - Close
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Server - Close
    nn::socket::Close( tcpServerSocket );
    tcpServerSocket = -1;

    // TCP Accepted Sock - Close
    nn::socket::Close( acceptedSocket );
    acceptedSocket = -1;


    ////////////
    // [UDP] - Test Direct Send to Peer on Lan Iface
    // Connected sockets without setting msghdr_name - VALID
    ////////////

    NN_LOG( "Testing: [UDP] - SendMMsg Direct Send to Peer on Lan Iface - Valid\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP server Sock
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port               = nn::socket::InetHtons( 9056 );
    serverAddr.sin_family             = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr        = myIp.S_addr;

    // UDP Server = Don't wait forever
    myTime.tv_sec       = 9;
    myTime.tv_usec      = 15000;
    optlen              = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // UDP Client - Close
    nn::socket::Close( udpSocket );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port              = nn::socket::InetHtons( 9057 );
    localAddr.sin_family            = nn::socket::Family::Af_Inet;

    NN_LOG( "\tClient IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr       = myIp.S_addr;

    // UDP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port             = nn::socket::InetHtons( 9056 ); // Point to UDP Server
    targetAddr.sin_family           = nn::socket::Family::Af_Inet;

    // UDP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr      = myIp.S_addr;
    NN_LOG( "\tConnected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // UDP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( idx = 0; idx < MaxMessages; idx++)
    {
        pReceiveIov[idx].iov_base              = mmsgReceiveBufs[idx];
        pReceiveIov[idx].iov_len               = 1400;
        pReceiveMsgs[idx].msg_hdr.msg_iov     = &pReceiveIov[idx];
        pReceiveMsgs[idx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[idx].msg_hdr.msg_name    = &pReceivedNames[idx];
        pReceiveMsgs[idx].msg_hdr.msg_namelen = sizeof(pReceivedNames[idx]);
    }

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if ( rc == -1 )
    {
        isSuccess = false;
        NN_LOG( "\tRecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "\tPackets received: %d\n", rc );
        mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( udpSocket );
    udpSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;


    ////////////
    // [UDP] - Test Direct Send to Peer on Lan Iface
    // Setting msghdr_name without connect - VALID
    ////////////

    NN_LOG( "Testing: [UDP] - SendMMsg using msg_hdr.msg_name only\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Server Socket - Create
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServerSocket < 0, "Failed to create UDP Server socket. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port = nn::socket::InetHtons( 9156 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Server Socket - Receive Timeout
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void*>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSocket < 0, "Failed to created UDP client socket. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client: Target Address
    memset(&targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9156 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;
    NN_LOG( "\tMy IP Address: %s\n", nn::socket::InetNtoa(myIp) );
    targetAddr.sin_addr.S_addr = myIp.S_addr;

    // Setup nn::socket::MMsgHdr
    for ( idx = 0; idx < MaxMessages; idx++ )
    {
        memset(&pSetNameMsg[idx], 0, sizeof(pSetNameMsg[idx]));
        pSetNameMsg[idx].msg_hdr.msg_name = static_cast<void *>(&targetAddr);
        pSetNameMsg[idx].msg_hdr.msg_namelen = sizeof(targetAddr);
        pSetNameMsg[idx].msg_hdr.msg_iov = pIov1;
        pSetNameMsg[idx].msg_hdr.msg_iovlen = 2;
    }

    rc = nn::socket::SendMMsg( udpSocket, pSetNameMsg, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg - Sent %d messages \n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if ( rc == -1 )
    {
        isSuccess = false;
        NN_LOG( "\tRecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "\tMessages received: %d\n", rc );
        mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // UDP Client - Close
    nn::socket::Close( udpSocket );
    udpSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;


    ////////////
    // [TCP] - Test Direct Send to Peer on Lan Iface
    // Sending Null Iovecs
    ////////////

    NN_LOG( "Testing: [TCP] - SendMMsg Direct Send to Peer on Lan Iface - Null Iovecs - nn::socket::Errno::EAgain\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // TCP server Sock
    tcpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServerSocket < 0, "Failed to create TCP Server socket. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Address
    memset(&serverAddr, 0, sizeof(serverAddr));
    serverAddr.sin_port = nn::socket::InetHtons( 8059 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // TCP Server = Don't wait forever
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof(myTime);

    rc = nn::socket::SetSockOpt( tcpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // TCP Server - Local Bind
    rc = nn::socket::Bind( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // TCP Server - Listen
    rc = nn::socket::Listen( tcpServerSocket, 5 );
    ERROR_IF( rc < 0, "Listen() failed! Errno:%d\n", nn::socket::GetLastError() );

    // TCP Client - Create
    tcpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

    // TCP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port = nn::socket::InetHtons( 8059 ); // Point to UDP Server
    targetAddr.sin_family = nn::socket::Family::Af_Inet;

    // TCP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr = myIp.S_addr;
    NN_LOG( "\tConnected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // TCP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( tcpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // TCP Server - Accept client connection
    acceptedSocket = nn::socket::Accept( tcpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), &targetAddrLen );
    ERROR_IF( acceptedSocket < 0, "Accept() expected to Succeed. Failed with Errno:%d \n", nn::socket::GetLastError() );

    // Fill in nullIovec msgs
    for ( idx = 0; idx < MaxMessages; idx++ )
    {
        memset( &pNullIovecMsg[idx], 0, sizeof(pNullIovecMsg[idx]) );
        pNullIovecMsg[idx].msg_hdr.msg_iov = NULL;
        pNullIovecMsg[idx].msg_hdr.msg_iovlen = 0;
    }

    rc = nn::socket::SendMMsg( tcpSocket, pNullIovecMsg, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() Failed! Expected to Succeed. Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( idx = 0; idx < MaxMessages; idx++)
    {
        pReceiveIov[idx].iov_base              = mmsgReceiveBufs[idx];
        pReceiveIov[idx].iov_len               = 1400;
        pReceiveMsgs[idx].msg_hdr.msg_iov     = &pReceiveIov[idx];
        pReceiveMsgs[idx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[idx].msg_hdr.msg_name    = &pReceivedNames[idx];
        pReceiveMsgs[idx].msg_hdr.msg_namelen = sizeof(pReceivedNames[idx]);
    }

    rc = nn::socket::RecvMMsg( acceptedSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF_AND_COUNT( rc > 0, "RecvMMsg() succeeded but expected to fail. Returned:%d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain, "RecvMMsg() expected nn::socket::Errno::EAgain. Acutal Errno:%d", myError );

    // TCP Client for broadcast testing - Close
    nn::socket::Close( tcpSocket );
    tcpSocket = -1;

    // TCP Server - Close
    nn::socket::Close( tcpServerSocket );
    tcpServerSocket = -1;

    // TCP Accepted Sock - Close
    nn::socket::Close( acceptedSocket );
    acceptedSocket = -1;


    ////////////
    // [UDP] - Test Direct Send to Peer on Lan Iface
    // Sending Null Iovecs
    ////////////

    NN_LOG( "Testing: [UDP] - SendMMsg Direct Send to Peer on Lan Iface - Null Iovecs - Empty payload\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP server Sock
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port               = nn::socket::InetHtons( 9056 );
    serverAddr.sin_family             = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr        = myIp.S_addr;

    // UDP Server = Don't wait forever
    myTime.tv_sec       = 9;
    myTime.tv_usec      = 15000;
    optlen              = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed!\n" );

    // UDP Client - Close
    nn::socket::Close( udpSocket );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port             = nn::socket::InetHtons( 9056 ); // Point to UDP Server
    targetAddr.sin_family           = nn::socket::Family::Af_Inet;

    // UDP - Target IP: Lan Interface
    targetAddr.sin_addr.S_addr      = myIp.S_addr;
    NN_LOG( "\tConnected Peer Address: %08x\n", targetAddr.sin_addr.S_addr );

    // UDP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    // Fill in nullIovec msgs
    for ( idx = 0; idx < MaxMessages; idx++ )
    {
        memset( &pNullIovecMsg[idx], 0, sizeof(pNullIovecMsg[idx]) );
        pNullIovecMsg[idx].msg_hdr.msg_iov = NULL;
        pNullIovecMsg[idx].msg_hdr.msg_iovlen = 0;
    }

    rc = nn::socket::SendMMsg( udpSocket, pNullIovecMsg, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() failed. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSendMMsg - Sent %d messages\n", rc );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(3) );

    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( idx = 0; idx < MaxMessages; idx++)
    {
        pReceiveIov[idx].iov_base              = mmsgReceiveBufs[idx];
        pReceiveIov[idx].iov_len               = 1400;
        pReceiveMsgs[idx].msg_hdr.msg_iov     = &pReceiveIov[idx];
        pReceiveMsgs[idx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[idx].msg_hdr.msg_name    = &pReceivedNames[idx];
        pReceiveMsgs[idx].msg_hdr.msg_namelen = sizeof(pReceivedNames[idx]);
    }

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if ( rc == -1 )
    {
        isSuccess = false;
        NN_LOG( "\tRecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "\tPackets received: %d\n", rc );
        mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( udpSocket );
    udpSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;


    ////////////
    // [UDP] - SendMMsg to unique addresses
    ///////////

    NN_LOG( "Testing: [UDP] - Setting 2 Unique addresses for different msghdrs\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Server - Create 2
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServerSocket < 0, "Failed to create first UDP server socket. Errno:%d\n", nn::socket::GetLastError() );
    udpServerSocket1 = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServerSocket1 < 0, "Falied to create second UDP server socket. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port = nn::socket::InetHtons( 9078 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    memset( &serverAddr1, 0, sizeof(serverAddr1) );
    serverAddr1.sin_port = nn::socket::InetHtons( 9079 );
    serverAddr1.sin_family = nn::socket::Family::Af_Inet;
    serverAddr1.sin_addr.S_addr = myIp.S_addr;

    // UDP Servers Recv timeout
    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void*>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );
    rc = nn::socket::SetSockOpt( udpServerSocket1, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void*>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );
    rc = nn::socket::Bind( udpServerSocket1, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr1), sizeof(serverAddr1) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSocket < 0, "Failed to created UDP client socket. Errno:%d\n", nn::socket::GetLastError() );

    // Setup nn::socket::MMsgHdr
    memset(&pUniqueMsgs[0], 0, sizeof(pUniqueMsgs[0]));
    pUniqueMsgs[0].msg_hdr.msg_name = static_cast<void *>(&serverAddr);
    pUniqueMsgs[0].msg_hdr.msg_namelen = sizeof(serverAddr);
    pUniqueMsgs[0].msg_hdr.msg_iov = pIov1;
    pUniqueMsgs[0].msg_hdr.msg_iovlen = 2;

    memset(&pUniqueMsgs[1], 0, sizeof(pUniqueMsgs[1]));
    pUniqueMsgs[1].msg_hdr.msg_name = static_cast<void *>(&serverAddr1);
    pUniqueMsgs[1].msg_hdr.msg_namelen = sizeof(serverAddr1);
    pUniqueMsgs[1].msg_hdr.msg_iov = pIov2;
    pUniqueMsgs[1].msg_hdr.msg_iovlen = 2;

    // UDP Client - SendMMsg
    rc = nn::socket::SendMMsg( udpSocket, pUniqueMsgs, 2, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 2, "SendMMsg failed to send 2 messages. Errno:%d\n", nn::socket::GetLastError() );
    NN_LOG( "\tSendMMsg - Sent %d messages\n", rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, 1, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF( rc != 1, "RecvMMsg() expected to receive 1 message. Actual: %d. Errno:%d\n", rc , nn::socket::GetLastError() );
    NN_LOG( "\tMessages received: %d\n", rc );
    mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
    nn::socket::InetNtop(
        nn::socket::Family::Af_Inet,
        &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
        addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
    NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    ERROR_IF_AND_COUNT( strncmp(mmsgReceiveBufs[rc - 1], msgs[2], strlen(msgs[2])) != 0, "Expected last message: %s\n", msgs[2] );

    rc = nn::socket::RecvMMsg( udpServerSocket1, pReceiveMsgs, 1, nn::socket::MsgFlag::Msg_None, &timeout) ;
    ERROR_IF( rc != 1, "RecvMMsg() expected to receive 1 message. Actual: %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    NN_LOG( "\tMessages received: %d\n", rc );
    mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
    nn::socket::InetNtop(
        nn::socket::Family::Af_Inet,
        &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
        addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]));
    NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    ERROR_IF_AND_COUNT( strncmp(mmsgReceiveBufs[rc - 1], msgs[3], strlen(msgs[3])) != 0, "Expected last message: %s\n", msgs[3]) ;

    // UDP Servers - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;
    nn::socket::Close( udpServerSocket1 );
    udpServerSocket1 = -1;

    // UDP Client - Close
    nn::socket::Close( udpSocket );
    udpSocket = -1;


    ////////////
    // [UDP] - Test Broadcast Address with no SetSockOpt - nn::socket::Errno::EAcces
    ///////////

    NN_LOG( "Testing: [UDP] - Broadcast Address with no SetSockOpt - nn::socket::Errno::EAcces\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Client for broadcast testing - Create
    broadcastSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 9005 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    NN_LOG( "\tMy IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    rc = nn::socket::Bind( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof(localAddr) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeded!\n" );

    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port     = nn::socket::InetHtons( 9006 ); // Somewhere harmeless
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
    NN_LOG( "\tDirected Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

    rc = nn::socket::Connect( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket! Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg( broadcastSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] SendMMsg() succeded but should of failed!" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] SendMMsg expceted nn::socket::Errno::EAcces. Errno:%d\n", myError );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( broadcastSocket );
    broadcastSocket = -1;


WRAP_FAILING_TEST( "SIGLO-58473", "RecvFrom() does not recieve broadcasted packets" )
{
    ////////////
    // [UDP] - Test Broadcast Address - VALID
    ///////////

    NN_LOG( "Testing: [UDP] - Brodcast Address - SockOpt - Valid\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port   = nn::socket::InetHtons( 0 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;
    serverAddr.sin_addr.S_addr = nn::socket::InetHtonl( nn::socket::InAddr_Any );

    // UDP Server - Don't wait forever..
    myTime.tv_sec   = 9;
    myTime.tv_usec  = 15000;
    optlen          = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc != 0, "Bind failed for local socket! Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Create
    broadcastSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );


    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 0 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    NN_LOG( "\tMy IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = nn::socket::InetHtonl( nn::socket::InAddr_Any );

    // UDP Clinet - Local Bind
    rc = nn::socket::Bind( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof(localAddr) );
    ERROR_IF( rc != 0, "Bind failed for local socket. Errno:%d\n", nn::socket::GetLastError() );

    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port     = nn::socket::InetHtons( 9058 ); // Point to UDP Server
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    // UDP - Target IP: Subnet broadcast address
    targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
    NN_LOG( "\tDirected Broadcast Address; %08x\n", targetAddr.sin_addr.S_addr );

    // UDP Client - May send Broadcast Packets
    rc = nn::socket::SetSockOpt( broadcastSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &enableFlag, sizeof(enableFlag) );
    ERROR_IF( rc < 0, "SetSockOpt for nn::socket::Option::So_Broadcast failed. Errno:%d\n", nn::socket::GetLastError() );

    // Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket! Errno:%d\n", nn::socket::GetLastError() );

    // Send Directed data from NX
    rc = nn::socket::SendMMsg( broadcastSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "[UDP] SendMMsg() Broadcast Address failed! Errno:%d\n", nn::socket::GetLastError() );

    NN_LOG( "\tSend - Sent %d messages \n", rc );

    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( idx = 0; idx < MaxMessages; idx++)
    {
        pReceiveIov[idx].iov_base              = mmsgReceiveBufs[idx];
        pReceiveIov[idx].iov_len               = 1400;
        pReceiveMsgs[idx].msg_hdr.msg_iov     = &pReceiveIov[idx];
        pReceiveMsgs[idx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[idx].msg_hdr.msg_name    = &pReceivedNames[idx];
        pReceiveMsgs[idx].msg_hdr.msg_namelen = sizeof(pReceivedNames[idx]);
    }

    isSuccess = true;

    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    if ( rc == -1 )
    {
        isSuccess = false;
        NN_LOG( "\tRecvMMsg() returned %d. Errno:%d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "\tPackets received: %d\n", rc );
        mmsgReceiveBufs[rc - 1][pReceiveMsgs[rc - 1].msg_len] = '\0';
        nn::socket::InetNtop(
            nn::socket::Family::Af_Inet,
            &(static_cast<nn::socket::SockAddrIn *>(pReceiveMsgs[rc - 1].msg_hdr.msg_name)->sin_addr),
            addrbuf, sizeof(addrbuf) / sizeof(addrbuf[0]) );
        NN_LOG( "\tLast message from %s: %s\n", addrbuf, mmsgReceiveBufs[rc - 1] );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "RecvMMsg() did not recieve any messages\n" );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( broadcastSocket );
    broadcastSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;

}


    ////////////
    // [UDP] - Test Broadcast Address with no SetSockOpt - nn::socket::Errno::EAcces
    ///////////

    NN_LOG( "Testing: [UDP] - Limited Broadcast Address with no SetSockOpt - nn::socket::Errno::EAcces\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Client for broadcast testing - Create
    broadcastSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 9005 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    NN_LOG( "\tMy IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    rc = nn::socket::Bind( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof(localAddr) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeded!\n" );

    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port     = nn::socket::InetHtons( 9006 ); // Somewhere harmeless
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    targetAddr.sin_addr.S_addr = myIp.S_addr |= ~myMask.S_addr;
    NN_LOG( "\tLimited Broadcast Address: %08x\n", targetAddr.sin_addr.S_addr );

    // This converts a directed broadcast to a limited broadcast.
    // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
    // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
    nn::socket::SetSockOpt( broadcastSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag));

    rc = nn::socket::Connect( broadcastSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket! Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::SendMMsg( broadcastSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] SendMMsg() succeded but should of failed!" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAcces, "[UDP] SendMMsg expceted nn::socket::Errno::EAcces. Errno:%d\n", myError );

    // UDP Client for broadcast testing - Close
    nn::socket::Close( broadcastSocket );
    broadcastSocket = -1;

    // UDP Server - Close
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;


    ////////////
    // [UDP] - Testing Disconnecting UDP Server nn::socket::Errno::EConnRefused
    ///////////

    NN_LOG( "Testing: [UDP] - Disconneted UDP Server - nn::socket::Errno::EConnRefused\n" );

    // UDP Server Sock
    udpServerSocket = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9050, "127.0.0.1", 9060 );
    ERROR_IF_AND_COUNT( udpServerSocket < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Close
    nn::socket::Close( udpSocket );

    // UDP Client - Connect to Server
    udpSocket = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9060, "127.0.0.1", 9050 );
    ERROR_IF_AND_COUNT( udpSocket < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // Send Message from Client to Server
    nn::socket::SendMMsg( udpSocket, pSendMsgs, 1, nn::socket::MsgFlag::Msg_None );

    // Close UDP Server
    nn::socket::Close( udpServerSocket );
    udpServerSocket = -1;

    // Client trys to keep writing
    isSuccess = false;
    for ( idx = 0; idx < 10; idx++ )
    {
        rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, 1, nn::socket::MsgFlag::Msg_None );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EConnRefused )
            {
                isSuccess = true;
                break;
            }
        }

        NN_LOG( "\t(Warning #%d) SendMMsg() Succeded. Expected to Fail.\n", idx );
    }

    ERROR_IF_AND_COUNT( !isSuccess, "SendMMsg() succeeded 10 times. Should have failed with nn::socket::Errno::EConnRefused\n" );

    // Close UDP Client
    nn::socket::Close( udpSocket );
    udpSocket = -1;


    ////////////
    // [UDP]
    //
    // nn::socket::MsgFlag::Msg_DontRoute - If an interface is not directly connected to the NX don't follow
    // the default route.
    ////////////

    NN_LOG( "Testing: [UDP] - nn::socket::MsgFlag::Msg_DontRoute - (Errno: nn::socket::Errno::EHostDown, nn::socket::Errno::ENetDown, etc)\n" );

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // UDP Client - Local Address
    memset( &localAddr, 0, sizeof(localAddr));
    localAddr.sin_port     = nn::socket::InetHtons( 9004 );
    localAddr.sin_family   = nn::socket::Family::Af_Inet;

    // UDP - Client - IP
    NN_LOG( "\tMy IP Address: %08x\n", myIp.S_addr );
    localAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Client - Local Bind
    rc = nn::socket::Bind( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&localAddr), sizeof( localAddr ) );
    ERROR_IF( rc != 0, "Bind failed for local socket, but should have succeeded!\n" );

    // UDP - Target Address
    memset( &targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port     = nn::socket::InetHtons( 53 );   // DNS == domain == 53
    targetAddr.sin_family   = nn::socket::Family::Af_Inet;

    // UDP - Target IP: (E-ROOT)
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof( targetAddr ) );
    ERROR_IF( rc != 0, "[UDP] Connect failed for UDP Peer socket!  Errno: %d\n", nn::socket::GetLastError() );

    // SendMMsg test data which is not (directly) routable from NX
    rc = nn::socket::SendMMsg( udpSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_DontRoute );
    ERROR_IF_AND_COUNT( rc > 0, "[UDP] SendMMsg() succeeded but should have failed!\n" );

    isSuccess = true;
    myError = nn::socket::GetLastError();
    switch( myError )
    {
    case nn::socket::Errno::EHostUnreach:
    case nn::socket::Errno::EHostDown:
    case nn::socket::Errno::ENetDown:
    case nn::socket::Errno::EConnRefused:
    case nn::socket::Errno::ENetUnreach:

        break;   // All is well

    default:
        isSuccess = false;
        break;
    }

    ERROR_IF_AND_COUNT( !isSuccess, "Unexpected Errno:%d, expected one of the 4x legal network ones\n", myError );

    // Close UDP Client
    nn::socket::Close( udpSocket );
    udpSocket = -1;


out:

    if ( tcpSocket > -1 )
    {
        nn::socket::Close( tcpSocket );
        tcpSocket = -1;
    }

    if ( udpSocket > -1 )
    {
        nn::socket::Close( udpSocket );
        udpSocket = -1;
    }

    if ( udpServerSocket > -1 )
    {
        nn::socket::Close( udpServerSocket );
        udpServerSocket = -1;
    }

    if ( pBigIov != NULL )
    {
        delete[] pBigIov;
        pBigIov = NULL;
    }

    if ( pBigIovs != NULL )
    {
        delete[] pBigIovs;
        pBigIovs = NULL;
    }

    if ( pReceiveIov != NULL )
    {
        delete[] pReceiveIov;
        pReceiveIov = NULL;
    }

    if ( pIov2 != NULL )
    {
        delete[] pIov2;
        pIov2 = NULL;
    }

    if ( pIov1 != NULL )
    {
        delete[] pIov1;
        pIov1 = NULL;
    }

    if ( pReceiveMsgs != NULL )
    {
        delete[] pReceiveMsgs;
        pReceiveMsgs = NULL;
    }

    if ( pBigMsgs != NULL )
    {
        delete[] pBigMsgs;
        pBigMsgs = NULL;
    }

    if ( pUniqueMsgs != NULL )
    {
        delete[] pUniqueMsgs;
        pUniqueMsgs = NULL;
    }

    if ( pNullIovecMsg != NULL )
    {
        delete[] pNullIovecMsg;
        pNullIovecMsg = NULL;
    }

    if ( pSetNameMsg != NULL )
    {
        delete[] pSetNameMsg;
        pSetNameMsg = NULL;
    }

    if ( pSendMsgs != NULL )
    {
        delete[] pSendMsgs;
        pSendMsgs = NULL;
    }

    if ( pReceivedNames != NULL )
    {
        delete[] pReceivedNames;
        pReceivedNames = NULL;
    }


} // NOLINT(impl/function_size)


TEST(ApiUnit, SendMMsg_Blocking_Test)
{
    nn::socket::TimeVal     myTime = { 0 };
    int                     clientSocket = -1, listenSocket = -1, acceptedSocket = -1, rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    bool                    isSuccess = true, returnBool = false;
    nn::socket::SockLenT    optlen  = 0;
    uint8_t                 noiseValue = 1;
    unsigned char *         inputBuf = NULL;
    nn::socket::MMsgHdr                 *pSendMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::Iovec                   *pIov1 = new nn::socket::Iovec[1]();

    int                 idx = 0;

    ////////////
    // Create TCP connection
    ///////////

    returnBool = NATF::API::COMMON::MakeListenSocket( listenSocket, 9060, "127.0.0.1" );
    ERROR_IF( returnBool == false, "Failed to create Listen Socket. Errno:%d\n", nn::socket::GetLastError() );

    clientSocket = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( clientSocket < 0, "Failed to create Client Socket. Errno:%d\n", nn::socket::GetLastError() );

    acceptedSocket = nn::socket::Accept( listenSocket, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSocket < 0, "Failed to Accept on Listen Socket. Errno:%d\n", nn::socket::GetLastError() );

    rc = NATF::API::COMMON::MakeRandomNoise( &inputBuf, 32767, noiseValue );
    ERROR_IF( rc < 0, "Failed calling MakeRandomNoise() for size 32767\n" );
    pIov1[0].iov_base = static_cast<void *>(inputBuf);
    pIov1[0].iov_len =  strlen( static_cast<const char *>(pIov1[0].iov_base) );

    for ( idx = 0; idx < MaxMessages; idx++ )
    {
        pSendMsgs[idx].msg_hdr.msg_iov = pIov1;
        pSendMsgs[idx].msg_hdr.msg_iovlen = 1;
    }

    ////////////
    // Establish Send Timeout
    ///////////

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rc = nn::socket::SetSockOpt( clientSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndTimeo, static_cast<void *>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed - Errno: %d\n", nn::socket::GetLastError() );

    isSuccess = false;
    while( 1 )
    {
        rc = nn::socket::SendMMsg( clientSocket, pSendMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                isSuccess = true;
                break;
            }
            NN_LOG( "SendMMsg() failed with Errno:%d\n", myError );
        }
    }

    ERROR_IF_AND_COUNT( !isSuccess, "SendMMsg() - SNDTIMEO failed to unlock after 5 seconds\n" );


out:

    if ( inputBuf != NULL )
    {
        free( inputBuf );
        inputBuf = NULL;
    }

    if ( clientSocket > -1 )
    {
        nn::socket::Close( clientSocket );
        clientSocket = -1;
    }

    if ( acceptedSocket > -1 )
    {
        nn::socket::Close( acceptedSocket );
        acceptedSocket = -1;
    }

    if ( listenSocket > -1 )
    {
        nn::socket::Close( listenSocket );
        listenSocket = -1;
    }

    if ( pIov1 != NULL )
    {
        delete[] pIov1;
        pIov1 = NULL;
    }

    if ( pSendMsgs != NULL )
    {
        delete[] pSendMsgs;
        pSendMsgs = NULL;
    }

} // NOLINT(impl/function_size)

TEST( ApiUnit,SendMMsg_SO_OOB)
{
    int                     clientSocket = -1, listenSocket = -1, acceptedSocket = -1;
    int                     rc = -1, idx = 0;
    bool                    isSuccess = true, isOOB = false;
    int                     disableFlag = 0, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( disableFlag );
    char                    receiveBuf[1024] = { 0 };
    nn::socket::MMsgHdr                 *pSendMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::Iovec                   *pIov1 = new nn::socket::Iovec[4]();

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    for ( idx = 0; idx < 4; idx++ )
    {
        pIov1[idx].iov_base = const_cast<char *>(msgs[idx]);
        pIov1[idx].iov_len =  strlen( static_cast<const char *>(pIov1[idx].iov_base) );
        pSendMsgs[idx].msg_hdr.msg_iov = &pIov1[idx];
        pSendMsgs[idx].msg_hdr.msg_iovlen = 1;
    }

    // Create Server
    rc = NATF::API::COMMON::MakeListenSocket( listenSocket, 9001, g_pEchoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSocket < 0, "Failed to create Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );

    // Create Client
    clientSocket = NATF::API::COMMON::MakeTCPConnection( g_pEchoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( clientSocket < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );

    // Server Accepts New Client
    acceptedSocket = nn::socket::Accept( listenSocket, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSocket < 0, "Failed calling Accept on Listen Socket - Errno:%d\n", nn::socket::GetLastError() );

    // Server Turns off OOBINLINE
    rc = nn::socket::SetSockOpt( acceptedSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, static_cast<void *>(&disableFlag), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno:%d\n", nn::socket::GetLastError() );

    // Server turns on NON-BLOCKING
    rc = nn::socket::Fcntl( acceptedSocket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno:%d\n", nn::socket::GetLastError() );

    // Client Writes (regular) and (OOB) data to server..
    NN_LOG( "SendMMsg(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[0], 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() failed with %d. Client failed to write; %s\n", rc, msgs[0] );

    NN_LOG( "SendMMsg() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[1], 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc < 0, "Client failed to SendMMsg (OOB); %s\n", msgs[1] );

    NN_LOG( "SendMMsg(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[2], 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "Client failed to write; %s\n", msgs[2] );


    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSocket );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::Recv( acceptedSocket, &receiveBuf[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::Recv( acceptedSocket, &receiveBuf[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", receiveBuf[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", receiveBuf[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, receiveBuf );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], receiveBuf, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, receiveBuf, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( clientSocket > -1 )
    {
        nn::socket::Close( clientSocket );
        clientSocket = -1;
    }

    if ( acceptedSocket > -1 )
    {
        nn::socket::Close( acceptedSocket );
        acceptedSocket = -1;
    }

    if ( listenSocket > -1 )
    {
        nn::socket::Close( listenSocket );
        listenSocket = -1;
    }

    if ( pIov1 != NULL )
    {
        delete[] pIov1;
        pIov1 = NULL;
    }

    if ( pSendMsgs != NULL )
    {
        delete[] pSendMsgs;
        pSendMsgs = NULL;
    }

} // NOLINT(impl/function_size)

TEST( ApiUnit,SendMMsg_SO_OOBINLINE)
{
    int                     clientSocket = -1, listenSocket = -1, acceptedSocket = -1;
    int                     rc = -1, idx = 0;
    bool                    isSuccess = true, isOOB = false;
    int                     enableFlag = 1, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( enableFlag );
    char                    receiveBuf[1024] = { 0 };
    nn::socket::MMsgHdr                 *pSendMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::Iovec                   *pIov1 = new nn::socket::Iovec[4]();

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    for ( idx = 0; idx < 4; idx++ )
    {
        pIov1[idx].iov_base = const_cast<char *>(msgs[idx]);
        pIov1[idx].iov_len =  strlen( static_cast<const char *>(pIov1[idx].iov_base) );
        pSendMsgs[idx].msg_hdr.msg_iov = &pIov1[idx];
        pSendMsgs[idx].msg_hdr.msg_iovlen = 1;
    }

    // Create Server
    rc = NATF::API::COMMON::MakeListenSocket( listenSocket, 9001, g_pEchoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSocket < 0, "Failed to create Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );

    // Create Client
    clientSocket = NATF::API::COMMON::MakeTCPConnection( g_pEchoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( clientSocket < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );

    // Server Accepts New Client
    acceptedSocket = nn::socket::Accept( listenSocket, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSocket < 0, "Failed calling Accept on Listen Socket - Errno:%d\n", nn::socket::GetLastError() );

    // Server Turns off OOBINLINE
    rc = nn::socket::SetSockOpt( acceptedSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, static_cast<void *>(&enableFlag), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno:%d\n", nn::socket::GetLastError() );

    // Server turns on NON-BLOCKING
    rc = nn::socket::Fcntl( acceptedSocket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno:%d\n", nn::socket::GetLastError() );

    // Client Writes (regular) and (OOB) data to server..
    NN_LOG( "SendMMsg(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[0], 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "SendMMsg() failed with %d. Client failed to write; %s\n", rc, msgs[0] );

    NN_LOG( "SendMMsg() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[1], 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc < 0, "Client failed to SendMMsg (OOB); %s\n", msgs[1] );

    NN_LOG( "SendMMsg(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::SendMMsg( clientSocket, &pSendMsgs[2], 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc < 0, "Client failed to write; %s\n", msgs[2] );


    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSocket );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::Recv( acceptedSocket, &receiveBuf[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::Recv( acceptedSocket, &receiveBuf[count], 1, nn::socket::MsgFlag::Msg_None );
                ERROR_IF( rc < 0, "Recv() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", receiveBuf[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "Recv() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", receiveBuf[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, receiveBuf );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], receiveBuf, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, receiveBuf, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( clientSocket > -1 )
    {
        nn::socket::Close( clientSocket );
        clientSocket = -1;
    }

    if ( acceptedSocket > -1 )
    {
        nn::socket::Close( acceptedSocket );
        acceptedSocket = -1;
    }

    if ( listenSocket > -1 )
    {
        nn::socket::Close( listenSocket );
        listenSocket = -1;
    }

    if ( pIov1 != NULL )
    {
        delete[] pIov1;
        pIov1 = NULL;
    }

    if ( pSendMsgs != NULL )
    {
        delete[] pSendMsgs;
        pSendMsgs = NULL;
    }

} // NOLINT(impl/function_size)


TEST(ApiUnit,SendMMsg_ControlData)
{
    int             udpSocket = -1, udpServerSocket = -1, rc = -1;
    nn::socket::SockAddrIn     targetAddr = { 0 }, serverAddr = { 0 };
    nn::socket::Iovec           pIov1[1] = { {0} };
    int             fd = 3, msgIdx = 0;

    nn::socket::SockLenT       optlen;
    nn::socket::MMsgHdr         *pReceiveMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::Iovec           *pReceiveIov = new nn::socket::Iovec[MaxMessages]();
    char            controlBuf[MaxMessages][CMSG_SPACE( sizeof(fd) )] = { { 0 } };
    static char     receiveBufs[MaxMessages][14000] = { { 0 } };
    nn::socket::SockAddrIn     *pReceivedNames = new nn::socket::SockAddrIn[MaxMessages]();

    bool            isSuccess = true;
    nn::socket::InAddr         myIp = { 0 }, myMask = { 0 }, myGw = { 0 }, myDns1 = { 0 }, myDns2 = { 0 };
    nn::socket::TimeVal         myTime = { 0 };
    nn::TimeSpan    timeout = nn::TimeSpan::FromSeconds(5);

    nn::socket::MMsgHdr         *pControlMsgs = new nn::socket::MMsgHdr[MaxMessages]();
    nn::socket::CMsgHdr         *cmsg = nullptr, *cmsgReceived = nullptr;
    int             error = 0;
    char            buf[CMSG_SPACE(sizeof(fd))] = { 0 };

WRAP_FAILING_TEST( "SIGLO-59012", "Control data not being received by RecvMMsg()" )
{

    /*
    * Initialize the I/O vector to send
    * the value in "er" (which is zero).
    * This is done because data must be
    * transmitted to send the fd.
    */
    pIov1[0].iov_base = &error;
    pIov1[0].iov_len = strlen(static_cast<const char *>(pIov1[0].iov_base));

    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++ )
    {
        pControlMsgs[msgIdx].msg_hdr.msg_iov = pIov1;
        pControlMsgs[msgIdx].msg_hdr.msg_iovlen = 1;

        pControlMsgs[msgIdx].msg_hdr.msg_control = buf;
        pControlMsgs[msgIdx].msg_hdr.msg_controllen = sizeof(buf);

        cmsg = reinterpret_cast<nn::socket::CMsgHdr *>(CMSG_FIRSTHDR(reinterpret_cast<msghdr *>(&pControlMsgs[msgIdx].msg_hdr) ));
        cmsg->cmsg_level = nn::socket::Level::Sol_Socket;
        cmsg->cmsg_type = SCM_RIGHTS;
        cmsg->cmsg_len = CMSG_LEN( sizeof(fd) );

        // Initialize payload
        *(reinterpret_cast<int *>( CMSG_DATA(cmsg) )) = fd;

        // Sum of the length of all control messages in buffer
        pControlMsgs[msgIdx].msg_hdr.msg_controllen = cmsg->cmsg_len;
    }

    // Ask NIFM for our current (primary interface) IP Address
    nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

    // UDP Server Socket - Create
    udpServerSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServerSocket < 0, "Failed to create UDP Server socket. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Address
    memset( &serverAddr, 0, sizeof(serverAddr) );
    serverAddr.sin_port = nn::socket::InetHtons( 9156 );
    serverAddr.sin_family = nn::socket::Family::Af_Inet;

    NN_LOG( "\tServer IP Address: %08x\n", myIp.S_addr );
    serverAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Server Socket - Receive Timeout
    myTime.tv_sec = 9;
    myTime.tv_usec = 15000;
    optlen = sizeof( myTime );

    rc = nn::socket::SetSockOpt( udpServerSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, static_cast<void*>(&myTime), optlen );
    ERROR_IF( rc < 0, "SetSockOpt() for nn::socket::Option::So_RcvTimeo failed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Server - Local Bind
    rc = nn::socket::Bind( udpServerSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr) );
    ERROR_IF( rc < 0, "Bind() failed for local socket. Expected to succeed! Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Create
    udpSocket = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSocket < 0, "Failed to created UDP client socket. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client: Target Address
    memset(&targetAddr, 0, sizeof(targetAddr));
    targetAddr.sin_port        = nn::socket::InetHtons( 9156 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;
    NN_LOG( "\tMy IP Address: %s\n", nn::socket::InetNtoa(myIp) );
    targetAddr.sin_addr.S_addr = myIp.S_addr;

    // UDP Client - Call Connect to Bind Servers IP as Peer
    rc = nn::socket::Connect( udpSocket, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    ERROR_IF( rc < 0, "Connect() failed. Expected to succeed. Errno:%d\n", nn::socket::GetLastError() );

    // UDP Client - Send 32 messages with same control data
    rc = nn::socket::SendMMsg( udpSocket, pControlMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != MaxMessages, "SendMMsg() failed to send %d messages. Actual: %d messages \n", MaxMessages, rc );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // Initialize receive messages
    memset(pReceiveMsgs, 0, sizeof(*pReceiveMsgs) * MaxMessages);
    for ( msgIdx = 0; msgIdx < MaxMessages; msgIdx++)
    {
        pReceiveIov[msgIdx].iov_base              = receiveBufs[msgIdx];
        pReceiveIov[msgIdx].iov_len               = sizeof(receiveBufs[msgIdx]);
        pReceiveMsgs[msgIdx].msg_hdr.msg_iov     = &pReceiveIov[msgIdx];
        pReceiveMsgs[msgIdx].msg_hdr.msg_iovlen  = 1;
        pReceiveMsgs[msgIdx].msg_hdr.msg_name    = &pReceivedNames[msgIdx];
        pReceiveMsgs[msgIdx].msg_hdr.msg_namelen = sizeof(pReceivedNames[msgIdx]);
        pReceiveMsgs[msgIdx].msg_hdr.msg_control = controlBuf[msgIdx];
        pReceiveMsgs[msgIdx].msg_hdr.msg_controllen = sizeof(controlBuf[msgIdx]);
    }

    // UDP Server Receive messages
    rc = nn::socket::RecvMMsg( udpServerSocket, pReceiveMsgs, MaxMessages, nn::socket::MsgFlag::Msg_None, &timeout );
    ERROR_IF( rc != MaxMessages, "RecvMMsg() failed to receive %d Messages. Actual: %d messages\n", MaxMessages, rc );

    cmsgReceived = reinterpret_cast<nn::socket::CMsgHdr *>(CMSG_FIRSTHDR( reinterpret_cast<msghdr *>(&pReceiveMsgs[0].msg_hdr) ));
    ERROR_IF_AND_COUNT( cmsgReceived->cmsg_level != nn::socket::Level::Sol_Socket && cmsgReceived->cmsg_type != SCM_RIGHTS,
        "Control data was not sent correctly. cmsg_levl %d, Acutal: %d. cmsg_type %d, Actual %d\n",
        cmsgReceived->cmsg_level, cmsg->cmsg_level, cmsgReceived->cmsg_type, cmsg->cmsg_type );
    NN_LOG( "Control Data: %d\n", *(reinterpret_cast<int *>( CMSG_DATA(cmsgReceived) )) );


} // End of failing test

out:

    if ( udpSocket > -1 )
    {
        nn::socket::Close( udpSocket );
        udpSocket = -1;
    }

    if ( udpServerSocket > -1 )
    {
        nn::socket::Close( udpServerSocket );
        udpServerSocket = -1;
    }

    if ( pControlMsgs != NULL )
    {
        delete[] pControlMsgs;
        pControlMsgs = NULL;
    }

    if ( pReceivedNames != NULL )
    {
        delete[] pReceivedNames;
        pReceivedNames = NULL;
    }

    if ( pReceiveIov != NULL )
    {
        delete[] pReceiveIov;
        pReceiveIov = NULL;
    }

    if ( pReceiveMsgs != NULL )
    {
        delete[] pReceiveMsgs;
        pReceiveMsgs = NULL;
    }

} // NOLINT(impl/function_size)


TEST(ApiUnit,SendMMsg_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,SendMMsg_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

} // Namespace: API
} // Namespace: NATF
