﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"
#include "Unit/testNet_CommonFunctions.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server

#include <nn/socket/socket_ApiPrivate.h>

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static EchoServer *         g_echoServer = NULL;

static bool                 g_workBool;
static bool                 g_shouldWrite;
static bool                 g_shouldRead;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}



static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if (g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    delete g_echoServer;
    g_echoServer = NULL;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


static void
SelectUnblockThreadTest( void * inArg )
{
    int * sockfds = (int *) inArg;

    NN_LOG( "Blocking Thread - Start\n" );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    nn::socket::Write( *sockfds, "1", 1 );

    g_workBool = true;

    NN_LOG( "Blocking Thread - Exit\n" );
}

////////////////
//
// B E G I N   T E S T I N G
//
////////////////


TEST(ApiUnit,Select_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Select_Blocking_Test)
{
    nn::socket::TimeVal      myTime;
    int                 sockfds = -1;
    int                 rc, thisThread;
    nn::socket::FdSet              readSet;

    bool                isSuccess = true, rval;
    nn::socket::SockLenT           optlen;

    ////////////
    // Create UDP connection
    ///////////

    myTime.tv_sec  = 1;
    myTime.tv_usec = 0;
    optlen         = sizeof( myTime );

    rval = NATF::API::COMMON::SetSocketOptAndConnect( sockfds, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, &myTime, optlen,
                                                      9000, g_echoServer->kInterfaceIPv4Addr,
                                                      8053, g_echoServer->kInterfaceIPv4Addr, true, 0 );   // isUDP = true

    ERROR_IF_AND_COUNT( rval != true, "SetSockOpt() returned false, but expected true!" );

    ////////////
    // Create a thread to run the "Unblock" routine
    ////////////

    g_workBool = false;
    thisThread = g_pThreadedTest->CreateThread( &SelectUnblockThreadTest, (void *) &sockfds, kThreadPriority );
    if ( thisThread < 0 )
    {
        NN_LOG( "Failed to create and start Unblock (thread)!\n" );
        goto out;
    }

    ////////////
    // Test Ublocking Select
    ////////////

    nn::socket::FdSetZero( &readSet );
    nn::socket::FdSetSet( sockfds, &readSet );

    PRINT_AND_CALL(rc = nn::socket::Select( sockfds + 1, &readSet, nullptr, nullptr, nullptr ) );
    if (  g_workBool == false )
    {
        NN_LOG( "===> Select unblocked - but Select wasn't unblokced by the control thread!!  This was unexpected\n" );
        nTestsFailing++;
        goto out;
    }
    else
    {
        NN_LOG( "================\n");
        NN_LOG( "== Pass - Select was unblocked by Write\n" );
        NN_LOG( "================\n");
        nTestsPassing++;
    }

    //////////////
    // Waiting for Thread
    /////////////

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( thisThread );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( thisThread );


out:
     if ( sockfds > -1 )
     {
         nn::socket::Close( sockfds );
         sockfds = -1;
     }


    return;
}


TEST(ApiUnit,Select_Descriptor_Tests)
{
    nn::socket::FdSet   fdset = { };
    nn::socket::TimeVal myTime = { 0 };
    int                 sockfds = -1, rc = -1;

    bool                isSuccess = true;

    myTime.tv_sec   = 0;
    myTime.tv_usec  = 0;

    ////////////
    // Negative File Descriptor
    ///////////

    PRINT_AND_CALL(rc = nn::socket::Select( -1, nullptr, nullptr, nullptr, &myTime ) );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EInval,
                       "Select() returned (true) when (numfds) is -1 !" );

    ////////////
    // Zero File Descriptor
    ///////////

    PRINT_AND_CALL(rc = nn::socket::Select( 0, nullptr, nullptr, nullptr, &myTime ) );
    if ( rc > 0 )
    {
        NN_LOG("Select() returned (true) when (numfds) is 0 !" );
        goto out;
    }

    ////////////
    // Good File Descriptor
    ///////////

    sockfds = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 8053 );

    PRINT_AND_CALL( rc = nn::socket::Select( sockfds + 1, nullptr, nullptr, nullptr, &myTime ) );
    ERROR_IF_AND_COUNT( rc > 0, "Select() returned (true) when (numfds) is (legal = %d) !", sockfds + 1 );

    nn::socket::Close( sockfds );
    sockfds = -1;

    ////////////
    // Maximum (Minimum) - File Descriptor
    ///////////

    PRINT_AND_CALL(rc = nn::socket::Select( ( INT_MAX * -1), nullptr, nullptr, nullptr, &myTime ) );
    ERROR_IF_AND_COUNT( rc > -1, "Select() returned: %d (true) when (numfds) is: %lld !", rc, ( INT_MAX * -1) );

    ////////////
    // Maximum - File Descriptor
    ///////////

    PRINT_AND_CALL( rc = nn::socket::Select( INT_MAX, nullptr, nullptr, nullptr, &myTime ); );
    ERROR_IF_AND_COUNT( rc != 0, "Select() returned: %d when all file descriptors are null. Should return 0.", rc );

    PRINT_AND_CALL( rc = nn::socket::Select( INT_MAX, &fdset, nullptr, nullptr, &myTime ); );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EBadf, "Select() returned: %d (true) when (numfds) is INTMAX !", rc );


out:

    if ( sockfds > 0 )
    {
        nn::socket::Close( sockfds );
        sockfds = -1;
    }
}


static void
SelectTimeout(nn::socket::TimeVal * pTimeout, nn::socket::Type socktype )
{
    int       rc = -1, sockfd = -1;
    nn::socket::FdSet    readfds, writefds, exceptionfds;

    uint64_t  start_usec, stop_usec, delta_usec, timeout_usec;
    uint64_t  smoothedTimeout;
    bool      isSuccess = true;

    if ( socktype == nn::socket::Type::Sock_Stream )
    {
        NN_LOG( "Testing with: [TCP][Tv_Sec = %ld][Tv_Usec = %d]\n", pTimeout->tv_sec, pTimeout->tv_usec );
        sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        ERROR_IF_AND_COUNT( sockfd < 0, "Unable to create nn::socket::Type::Sock_Stream, rc: %d, errno: %d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "Testing with: [UDP][Tv_Sec = %ld][Tv_Usec = %ld]\n", pTimeout->tv_sec, pTimeout->tv_usec );
        sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ERROR_IF_AND_COUNT(  sockfd < 0, "Unable to create nn::socket::Type::Sock_Dgram, rc: %d, errno: %d\n", rc, nn::socket::GetLastError() );
    }

    // Top level Test
    for( int idx = 0; idx < 3; idx++ )
    {
        nn::socket::FdSetZero( &readfds );
        nn::socket::FdSetZero( &writefds );
        nn::socket::FdSetZero( &exceptionfds );

        // Not used
        nn::socket::FdSetSet(sockfd, &readfds );
        nn::socket::FdSetSet(sockfd, &exceptionfds );

        // Start time - MicroSeconds
        start_usec = NATF::API::COMMON::GetMicroSeconds();

        // Select
        rc = nn::socket::Select(sockfd + 1, &readfds, &writefds, &exceptionfds, pTimeout );
        ERROR_IF_AND_COUNT( rc != 0 , "FAILED: select() return cocde is <%ld>, but I expected it to be (ZERO == 0)!\n", rc );

        // End time - MicroSecodns
        stop_usec = NATF::API::COMMON::GetMicroSeconds();

        // Delta time
        delta_usec = stop_usec - start_usec;

        // Convert timeout MS
        timeout_usec =  pTimeout->tv_sec * 1000000;
        timeout_usec += pTimeout->tv_usec;

        // NN_LOG( "pTimeout->sec %d, pTimeout->tv_usec %d us!\n",pTimeout->tv_sec,  pTimeout->tv_usec);

        // Display result
        NN_LOG( "Start_usec <%llu>, Stop_usec <%llu>, Elasped_usec <%llu>, Timeout_usec <%llu>\n",
                    start_usec, stop_usec, delta_usec, timeout_usec );
        // RESULTS:

        // We don't allow select() to timeout QUICKER/FASTER than the timeout
        ERROR_IF_AND_COUNT( delta_usec < timeout_usec,
                           "FAILED: select() timed out after waiting <%llu>usec, but timeout is set to <%llu>usec!\n",
                           delta_usec, timeout_usec );

        smoothedTimeout = timeout_usec + 3000000;

        // system call should unblock before 3s delta
        ERROR_IF_AND_COUNT( delta_usec > smoothedTimeout,
        "FAILED: select() timed out after waiting <%llu>usecs, but expected\n", delta_usec, smoothedTimeout );
    }

out:

    if ( sockfd > -1 )
    {
        nn::socket::Close(sockfd);
        sockfd = -1;
    }

};



TEST(ApiUnit,Select_notimeout)
{
    nn::socket::TimeVal  testtime = { 0, 0};

    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}


TEST(ApiUnit,Select_no1usec)
{
    nn::socket::TimeVal  testtime = { 0, 1 };

    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t5usecs)
{
    nn::socket::TimeVal  testtime = { 0, 5 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t10usecs)
{
    nn::socket::TimeVal  testtime = { 0, 10 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );

}

TEST(ApiUnit,Select_t100usecs)
{
    nn::socket::TimeVal  testtime = { 0, 100 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t1ms)
{
    nn::socket::TimeVal  testtime = { 0, (1 * 1000 ) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t5ms)
{
    nn::socket::TimeVal  testtime = { 0, (5 * 1000 ) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t10ms)
{
    nn::socket::TimeVal  testtime = { 0, (10 * 1000 ) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t100ms)
{
    nn::socket::TimeVal  testtime = { 0, (100 * 1000 ) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t1s)
{
    nn::socket::TimeVal  testtime = { 1, 0 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t11s)
{
    nn::socket::TimeVal  testtime = { 1, 1 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t15s)
{
    nn::socket::TimeVal  testtime = { 1, ( 500 * 1000 ) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t1999999s)
{
    nn::socket::TimeVal  testtime = { 1, 999999 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t5s)
{
    nn::socket::TimeVal  testtime = { 5, 0 };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Select_t51s)
{
    nn::socket::TimeVal  testtime = { 5, (1 * 1000) };
    SelectTimeout( &testtime, nn::socket::Type::Sock_Stream );
    SelectTimeout( &testtime, nn::socket::Type::Sock_Dgram );
}


struct SelectPayload
{
    uint8_t   type;
    int       threadId;
    nn::Bit64 desiredCore;
    int       sockfds;
    size_t    maxBufSize;
    uint64_t  bytesSent;
    uint64_t  writeWouldBlock;
    uint64_t  writeNotFull;
    int       writerDelay;
};


static void
SelectWriter( void * in_payload )
{
    nn::socket::TimeVal          mytime;
    size_t                  rc;
    nn::socket::FdSet                  writeSet;
    struct SelectPayload *  payload = (SelectPayload *) in_payload;
    unsigned char *         workPtr;
    uint8_t                 startPos = 0;
    bool                    isSuccess = true;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    NN_LOG( "Writer Thread: %d, Core: %lld, Socket: %d - Starting!\n",
            payload->threadId, payload->desiredCore, payload->sockfds );

    // Make Random Noise
    startPos = payload->sockfds;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, payload->maxBufSize, startPos );
    ERROR_IF( rc < 0, "MakeRandomNoise failed!  Errno: %d\n", nn::socket::GetLastError() );

    // Wait for start..
    while( g_workBool == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 5 ));
    }

    // While writer should run
    while( g_shouldWrite == true )
    {
        // Select for Writing
        mytime.tv_sec  = 0;
        mytime.tv_usec = 100;

        nn::socket::FdSetZero( &writeSet );
        nn::socket::FdSetSet( payload->sockfds, &writeSet );

        rc = nn::socket::Select( payload->sockfds + 1, nullptr, &writeSet, nullptr, &mytime );

        if (rc == 0 )
        {
             nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
             continue;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( payload->writerDelay ));

        if ( nn::socket::FdSetIsSet( payload->sockfds, &writeSet ) )
        {
            rc = NATF::API::COMMON::WriteData( payload->sockfds, workPtr, payload->maxBufSize, payload->writeWouldBlock, payload->writeNotFull );
            ERROR_IF( rc < 0, "Write failed!  Errno: %d\n", nn::socket::GetLastError() );

            payload->bytesSent = payload->bytesSent + rc;
         }
    }

out:

    NN_LOG( "Writer Thread: %d, Core: %lld, Socket: %d - Exiting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Clear Random Result
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;
}


static void
SelectReader( void * in_payload )
{
    nn::socket::TimeVal          mytime;
    size_t                  rc;
    nn::socket::FdSet                  readSet;
    struct SelectPayload *  payload = (SelectPayload *) in_payload;
    bool                    isSuccess = true;
    unsigned char *         workPtr;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    NN_LOG( "Reader Thread: %d, Core: %lld, Socket: %d - Starting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Allocate Buffer Ptr;
    workPtr = (unsigned char *) malloc( payload->maxBufSize );
    ERROR_IF( workPtr == NULL, "Work Pointer failed to allocate 4096 bytes!" );

    // Wait for start..
    while( g_workBool == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 5 ));
    }

    // While reader sahould run
    while( g_shouldRead == true )
    {
        mytime.tv_sec  = 0;
        mytime.tv_usec = 100;

        nn::socket::FdSetZero( &readSet );
        nn::socket::FdSetSet( payload->sockfds, &readSet );

        rc = nn::socket::Select( payload->sockfds + 1, &readSet, nullptr, nullptr, &mytime );

        if ( rc == 0 )
        {
             nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
             continue;
        }

        if ( nn::socket::FdSetIsSet( payload->sockfds, &readSet ) )
        {
            rc = NATF::API::COMMON::ReadData( payload->sockfds, workPtr, payload->maxBufSize, payload->writeWouldBlock );
            ERROR_IF( rc < 0, "Read failed!  Errno: %d\n", nn::socket::GetLastError() );

            payload->bytesSent = payload->bytesSent + rc;
        }
    }

out:

    NN_LOG( "Reader Thread: %d, Core: %lld, Socket: %d - Exiting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Clear Random Result
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }
}


static int
Select_Read_And_Write_UDP( int numberOfPairs, int  in_delaytimems, const char *  in_interface )
{
    struct SelectPayload   payload[15];

    int               idx;
    bool              isSuccess = true;
    size_t            maxSize;
    int               core, result;

    memset( (char *) &payload, 0, sizeof( payload ) );

    NN_LOG( "====> Testing Number of sockets: %d\n", numberOfPairs );

    ////////////////
    // Start 2 pairs of UDP sockets
    ///////////////

    // Indicate that threads should run..
    g_shouldWrite = true;
    g_shouldRead  = true;
    g_workBool    = false;

    // Start Threads
    maxSize = 0;
    core = 1;
    for( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // Set Desired Size
        maxSize = maxSize + 1024;

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        /////////////////
        // Create Reader - Thread
        /////////////////

        payload[idx].type = 'R';
        payload[idx].desiredCore = core;
        payload[idx].maxBufSize = maxSize;
        payload[idx].sockfds = NATF::API::COMMON::MakeUDPConnection( in_interface, ( 9000 + idx),
                                                                     in_interface, ( 9001 + idx) );
        ERROR_IF( payload[idx].sockfds < 0, "MakeUDPConnection failed" );

        payload[idx].threadId = g_pThreadedTest->CreateThread( &SelectReader, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create a Reader (thread)!\n" );
            goto out;
        }

        /////////////////
        // Createa Writer - Thread
        /////////////////

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx + 1].type = 'W';
        payload[idx + 1].writerDelay = in_delaytimems;
        payload[idx + 1].desiredCore = core;
        payload[idx + 1].maxBufSize = maxSize;
        payload[idx + 1].sockfds = NATF::API::COMMON::MakeUDPConnection( in_interface, (9001 + idx ),
                                                                         in_interface, (9000 + idx )  );

        payload[idx + 1].threadId = g_pThreadedTest->CreateThread( &SelectWriter, (void *) &payload[idx + 1], kThreadPriority );
        if ( payload[idx + 1].threadId < 0 )
        {
            NN_LOG( "Failed to create a Writer (thread)!\n" );
            goto out;
        }
    }
    // Give threads time to start
    NN_LOG( "====> Give threads time to start...\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Test Time 60 seconds
    NN_LOG( "====> Test running for next 60 seconds..." );
    g_workBool    = true;

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
    NN_LOG( "Done!\n\n" );

    // Stop Writers
    NN_LOG( "Stop Writers!\n" );
    g_shouldWrite = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Stop Readers
    NN_LOG( "Stop Readers!\n" );
    g_shouldRead = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Wait for each thread death..
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        // Waitthreads to die
        g_pThreadedTest->WaitThread( payload[idx].threadId );

        // Destroy Thread
        g_pThreadedTest->DestroyThread( payload[idx].threadId );

        // Close the Socket
        nn::socket::Close( payload[idx].sockfds );

        // If Writer Thread
        if (  payload[idx].type == 'W' )
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Sent: %d, wouldBlock: %d, writeNotFull: %d, writeDelay: %dms\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull,
                  payload[idx].writerDelay );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
        else
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Read: %d, wouldBlock: %d, readNotFull: %d\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
    }

    NN_LOG( "All threads stopped!\n" );

    result = 0;
    for ( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // If Reader and Writer disagree..
        if ( payload[idx].bytesSent != payload[idx + 1].bytesSent )
        {
            result = -1;
        }
    }

    return( result );

out:

    return( -1 );

}


static int
Select_Read_And_Write_TCP( int  numberOfPairs, int in_delaytimems, const char * in_interface )
{
    struct SelectPayload   payload[15];

    int               idx, listenSock = -1;
    bool              isSuccess = true;
    size_t            maxSize;
    int               core, result;
    bool              rval;

    memset( (char *) &payload, 0, sizeof( payload ) );


    NN_LOG( "====> Testing Number of sockets: %d\n", numberOfPairs );

    ////////////////
    // Start 2 pairs of UDP sockets
    ///////////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9000, in_interface );
    ERROR_IF( rval == false, "MakeListenSocket failed!" );

    // Indicate that threads should run..
    g_shouldWrite = true;
    g_shouldRead  = true;
    g_workBool    = false;   // Threads don't run read..

    // Start Threads
    maxSize = 0;
    core = 1;
    for( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // Set Desired Size
        maxSize = maxSize + 1024;

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        /////////////////
        // Create Reader - Thread
        /////////////////

        payload[idx].type         = 'R';
        payload[idx].desiredCore  = core;
        payload[idx].maxBufSize   = maxSize;

        payload[idx].sockfds      = NATF::API::COMMON::MakeTCPConnection( in_interface, 9000 );

        payload[idx].threadId = g_pThreadedTest->CreateThread( &SelectReader, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create a Reader (thread)!\n" );
            goto out;
        }

        /////////////////
        // Createa Writer - Thread
        /////////////////

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx + 1].type = 'W';
        payload[idx + 1].writerDelay = in_delaytimems;
        payload[idx + 1].desiredCore = core;
        payload[idx + 1].maxBufSize  = maxSize;
        payload[idx + 1].sockfds     = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );

        payload[idx + 1].threadId = g_pThreadedTest->CreateThread( &SelectWriter, (void *) &payload[idx + 1], kThreadPriority );
        if ( payload[idx + 1].threadId < 0 )
        {
            NN_LOG( "Failed to create a Writer (thread)!\n" );
            goto out;
        }
    }
    NN_LOG( "====> Give threads time to start...\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Test Time 60 seconds
    NN_LOG( "====> Test running for next 60 seconds..." );
    g_workBool = true;

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
    NN_LOG( "Done!\n\n" );

    // Stop Writers
    NN_LOG( "Stop Writers!\n" );
    g_shouldWrite = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Stop Readers
    NN_LOG( "Stop Readers!\n" );
    g_shouldRead = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Wait for each thread death..
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        // Waitthreads to die
        g_pThreadedTest->WaitThread( payload[idx].threadId );

        // Destroy Thread
        g_pThreadedTest->DestroyThread( payload[idx].threadId );

        // Close the Socket
        nn::socket::Close( payload[idx].sockfds );

        // If Writer Thread
        if (  payload[idx].type == 'W' )
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Sent: %d, wouldBlock: %d, writeNotFull: %d, writerDelay: %dms\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock,
                  payload[idx].writeNotFull, payload[idx].writerDelay );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
        else
        {
             NN_LOG( "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Read: %d, wouldBlock: %d, readNotFull: %d\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
    }

    NN_LOG( "All threads stopped!\n" );

    result = 0;
    for ( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // If Reader and Writer disagree..
        if ( payload[idx].bytesSent != payload[idx + 1].bytesSent )
        {
            result = -1;
        }
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = 0;
    }

    return( result );

out:

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = 0;
    }

    return( 0 );

}  // NOLINT(impl/function_size)


TEST(ApiUnit,Select_Read_Write_UDP)
{
    Select_Read_And_Write_UDP( 2, 50, "127.0.0.1" );

    Select_Read_And_Write_UDP( 4, 50, "127.0.0.1" );

    Select_Read_And_Write_UDP( 6, 50, "127.0.0.1" );
}


TEST(ApiUnit,Select_Read_Write_TCP)
{
    Select_Read_And_Write_TCP( 2, 50, "127.0.0.1" );

    Select_Read_And_Write_TCP( 4, 50, "127.0.0.1" );

    Select_Read_And_Write_TCP( 6, 50, "127.0.0.1" );
}


TEST(ApiUnit,Select_OOB)
{

WRAP_FAILING_TEST( "SIGLO-58138", "[NX] Recv with OOB data fails with nn::socket::Errno::EInval == 22 after Select" )
{

    nn::socket::TimeVal myTime = {};
    int                 clisock = -1, svrsock = -1, accepted = -1, rc = -1;
    nn::socket::MsgFlag is_oob = nn::socket::MsgFlag::Msg_None;
    nn::socket::FdSet   readSet = {}, exceptSet = {};
    char                buf1[1024] = {};
    bool                rval = false, isSuccess = true;

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server and Client Socket
    ////////////////

    // Create Server Listen Socket with OOB..
    rval = NATF::API::COMMON::MakeListenSocket( svrsock, 9000, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Server/Listen socket\n" );

    // Create Client socket with OOB...
    clisock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9000 );
    ERROR_IF( clisock < 0, "Failed to create Client socket\n" );

    // Server Accepts Client connection..
    accepted = nn::socket::Accept( svrsock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( accepted < 0, "Failed to accept Client socket\n" );

    // Server - Accepted:  Fcntl - nn::socket::FcntlFlag::O_NonBlock
    rc = nn::socket::Fcntl( accepted, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////////
    // Create Server and Client Socket
    ////////////////


    // Client writes normal data
    rc = nn::socket::Send( clisock, msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc != strlen( msgs[0] ), "Failed to send NORMAL data - rc: %d, errno: %d \n", rc, nn::socket::GetLastError());

    // Client writes Out-Of-Band (OOB) data
    rc = nn::socket::Send( clisock, msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF( rc != strlen( msgs[1] ), "Failed to send OOB data - rc: %d, errno: %d \n", rc, nn::socket::GetLastError());

    // Client writes normal data
    rc = nn::socket::Send( clisock, msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc != strlen( msgs[2] ), "Failed to send NORMAL data - rc: %d, errno: %d \n", rc, nn::socket::GetLastError());

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    for( ; ; )
    {
        // Server Selects on OOB data..
        nn::socket::FdSetZero( &exceptSet );
        nn::socket::FdSetZero( &readSet );

        nn::socket::FdSetSet( accepted, &readSet );
        nn::socket::FdSetSet( accepted, &exceptSet );

        myTime.tv_sec = 0;
        myTime.tv_usec = 500;

        // Select looking for OOB and Normal Data
        NN_LOG( "Before Select\n");
        rc = nn::socket::Select( accepted + 1, &readSet, nullptr, &exceptSet, &myTime );
        NN_LOG( "After Select - rc: %d\n", rc);

        if ( rc == 0 ) break;

        // Server recieves NORMAL data
        if ( nn::socket::FdSetIsSet( accepted, &readSet ) )
        {
            NN_LOG( "Normal Data - Ready-to-Read\n" );
            rc = nn::socket::Recv( accepted, (unsigned char *) buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
            ERROR_IF( rc < 0, "Failed to read NORMAK data from Socket!  errno: %d\n", nn::socket::GetLastError() );

            NN_LOG( "Read NORMAL Data: [%.*s]\n", rc, buf1 );
        }

        // Server recieves OOB data
        if ( nn::socket::FdSetIsSet( accepted, &exceptSet ) )
        {
            NN_LOG( "OOB Data - Ready-to-Read\n" );

            is_oob = nn::socket::MsgFlag::Msg_None;
            if( nn::socket::SockAtMark(accepted) == 1 )
            {
                is_oob = nn::socket::MsgFlag::Msg_Oob;
            }

            NN_LOG( "Recv with %s\n", ( is_oob == nn::socket::MsgFlag::Msg_Oob ? "nn::socket::MsgFlag::Msg_Oob" : "NORMAL" ) );
            rc = nn::socket::Recv( accepted, (unsigned char *) buf1, sizeof( buf1 ), is_oob );
            if ( rc < 0 )
            {
                if ( nn::socket::GetLastError() == nn::socket::Errno::EInval )
                {
                    NN_LOG( "(Second Recv) - First call returned nn::socket::Errno::EInval - Instructed to call Recv() without nn::socket::MsgFlag::Msg_Oob\n" );
                    rc = nn::socket::Recv( accepted, (unsigned char *) buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
                    ERROR_IF( rc < 0, "(Second Recv) - Failed to Recv OOB data from Socket!  errno: %d\n", nn::socket::GetLastError() );
                }
            }

            NN_LOG( "Read Except/OOB Data: [%.*s]\n", rc, buf1 );
        }
    }

out:

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    if ( svrsock > -1 )
    {
        nn::socket::Close( svrsock );
        svrsock = -1;
    }

    if ( accepted > -1 )
    {
        nn::socket::Close( accepted );
        accepted = -1;
    }

}   // End Failing Test

}


TEST(ApiUnit,Select_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Select_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
