﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nnt/nntest.h>

#include "Complex/testNet_SelectUnitNetworkCommon.h"

namespace NATF {
namespace API {

NN_ALIGNAS(4096) uint8_t g_PollTimeoutSimpleSocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

uint64_t getTicksNowPoll()
{
    nn::os::Tick now_in_ticks = nn::os::GetSystemTick();
    return( nn::os::ConvertToTimeSpan( now_in_ticks ).GetMicroSeconds() );
}

void testTCPPollTimeout(int timeout_ms, int in_flags )
{
    nn::Result result;
    nn::socket::PollFd     poll_struct;

    int rc = -1;
    int sockfd = -1;
    int flags = -1;
    uint64_t start_usecs, stop_usecs, delta_usecs, timeout_usecs;
    uint64_t kTimeoutValue = 0;

    // retain result to finalize the socket layer below
    result = nn::socket::Initialize(g_PollTimeoutSimpleSocketMemoryPoolBuffer,
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);

    if( result.IsFailure() )
    {
        NN_LOG("Error: nn::socket::Initialize() failed. Err Desc: %d\n", result.GetDescription());
        ADD_FAILURE();
        goto bail;
    }
    // create the socket
    else if (-1 == (sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)))
    {
        NN_LOG("Unable to get socket, rc: %d, errno: %d\n", rc, sockfd, nn::socket::GetLastError());
        ADD_FAILURE();
        goto bail;
    }

    if ( SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING == in_flags )
    {
        if ( -1 == (rc = nn::socket::Fcntl(sockfd, nn::socket::FcntlCommand::F_SetFl, 0)))
        {
            NN_LOG("Error: Fcntl(%d, nn::socket::FcntlCommand::F_SetFl, 0) returned -1\n", sockfd);
            ADD_FAILURE();
            goto bail;
        }
        else if (-1 == (rc = nn::socket::Fcntl(sockfd, nn::socket::FcntlCommand::F_GetFl, flags)))
        {
            NN_LOG("Error: Fcntl(%d, nn::socket::FcntlCommand::F_GetFl, 0) returned -1\n", sockfd);
            ADD_FAILURE();
            goto bail;
        }
        else if( nn::socket::FcntlFlag::None != (nn::socket::FcntlFlag::O_NonBlock & static_cast<nn::socket::FcntlFlag>(rc)) )
        {
            NN_LOG("Error:  flags(%x) contains nn::socket::FcntlFlag::O_NonBlock, but should not, because we set the initial flags value.\n", rc);
            ADD_FAILURE();
            goto bail;
        }
    }

    // Poll structure
    poll_struct.fd      = sockfd;
    poll_struct.events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollPri;
    poll_struct.revents = nn::socket::PollEvent::PollNone;

    // Start time
    start_usecs = getTicksNowPoll();

    // Poll
    rc = nn::socket::Poll( &poll_struct, 1, timeout_ms );
    if ( 0 != rc )
    {
        NN_LOG( "FAILED: Poll() return code is <%d>, but should ONLY BE (ZERO == 0)!\n", rc );
        ADD_FAILURE();
        goto bail;
    }

    // End time
    stop_usecs = getTicksNowPoll();

    // Delta time
    delta_usecs = stop_usecs - start_usecs;

    // Translate Timeout MILLISECONDS to Timeout MicroSeconds
    timeout_usecs = timeout_ms * 1000;

    NN_LOG( "timeout_ms %ld, timeout_usecs %ld\n", timeout_ms, timeout_usecs );

    // Display result
    NN_LOG( "Start_usec <%llu>, Stop_usec <%llu>, Elasped_usec <%llu>, Timeout_usec <%llu>\n",
                    start_usecs, stop_usecs, delta_usecs, timeout_usecs );
    // RESULTS:

    // We don't allow Poll() to timeout QUICKER/FASTER than the timeout
    if ( delta_usecs < timeout_usecs ) {
        NN_LOG( "FAILED: Poll() timed out after waiting <%llu>ms, but timeout is set to <%llu>ms!\n",
                            delta_usecs, timeout_usecs );
        ADD_FAILURE();
        goto bail;
    }

    kTimeoutValue = timeout_usecs + 3000000;

    // system call should unblock before 3s delta
    if ( delta_usecs > kTimeoutValue )
    {
        NN_LOG( "FAILED: Poll() timed out after waiting <%llu>usec's\n", delta_usecs );
        NN_LOG( "but expected timeout to UNBLOCK within 3 seconds, or <%llu>usec's\n", kTimeoutValue  );
        ADD_FAILURE();
        goto bail;
    }

bail:
    if ( -1 != sockfd )
    {
        nn::socket::Close(sockfd);
        sockfd = -1;
    };

    if (result.IsSuccess())
    {
        result = nn::socket::Finalize();
        if( result.IsFailure() )
        {
            NN_LOG("Error: nn::socket::Finalize() failed. Err Desc: %d\n", result.GetDescription());
        };
    };
};

TEST(PollTimeoutSimple, notime)
{
    testTCPPollTimeout( 0, 0 );
};

TEST(PollTimeoutSimple, 1ms)
{
    testTCPPollTimeout( 1, 0 );
};

TEST(PollTimeoutSimple, 5ms)
{
    testTCPPollTimeout( 5, 0 );
};

TEST(PollTimeoutSimple, 10ms)
{
    testTCPPollTimeout( 10,  0 );
};

TEST(PollTimeoutSimple, 100ms)
{
    testTCPPollTimeout( 100, 0 );
};

TEST(PollTimeoutSimple, 1s)
{
    testTCPPollTimeout( (1 * 1000 ), 0 );
};

TEST(PollTimeoutSimple, 1_1s)
{
    testTCPPollTimeout( ( 1 * 1000 ) + 1, 0 );
};

TEST(PollTimeoutSimple, 1_5s)
{
    testTCPPollTimeout( ( 1 * 1000 ) + 500, 0 );
};

TEST(PollTimeoutSimple, 1_999s)
{
    testTCPPollTimeout(  ( 1 * 1000 ) + 999, 0 );
};

TEST(PollTimeoutSimple, 5s)
{
    testTCPPollTimeout( ( 5 *  1000 ), 0);
};

TEST(PollTimeoutSimple, 5_1s)
{
    testTCPPollTimeout( (5 * 1000 ) + 1, 0 );
};

TEST(PollTimeoutSimple, 5_999s)
{
    testTCPPollTimeout(  ( 5 * 1000 ) + 999, 0 );
};


} // namespace NATF
} // namespace API
