﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>

#include "natf/Utils/CommandLineParser.h" // NATF::Utils::ParserGroup
#include "natf/natf.h" // NATF::BaseTest

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_CommonFunctions.h"       // Common Functions


namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

// Anonymous
namespace {

ThreadedController            *g_pThreadedTest = NULL;
nn::nifm::NetworkConnection   *g_pNifmConnection = nullptr;
bool                          g_NIFMIsUp = false;
nn::util::Uuid                g_netProfile;

const int                     kMaxNumberOfThreads = 32;
const int                     kThreadPriority = 10;

NN_ALIGNAS(4096) uint8_t g_NIFMShutdownSocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];


bool ParseOptions(char* testName, nn::util::Uuid* pNetProfile) NN_NOEXCEPT
{
    bool isSuccess = true;
    nn::Result rval;

    NATF::Utils::ParserGroup parser;
    nn::util::Uuid netProfileLocal;
    char testNameLocal[NATF::BaseTest::NameBufferLen];

    int argC;
    const char * const * argV;

    ERROR_IF(nullptr == testName, "TestSetup: testName nullptr");
    ERROR_IF(nullptr == pNetProfile, "TestSetup: pNetProfile nullptr");

    NETTEST_GET_ARGS(argC, argV);

    NN_LOG("ParseOptions: argC=%d\n", argC);
    for( int i = 0; i < argC; ++i )
    {
        NN_LOG("ParseOptions: argV[%d]=%s\n", i, argV[i]);
    }

    parser.AddParser(NATF::Utils::StringParser  ("--Name", nullptr, testNameLocal, sizeof(testNameLocal)));
    parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfileLocal));

    ERROR_IF( !parser.Parse(argC, argV), "Failed to parse command line arguments");

    strcpy(testName, testNameLocal);
    *pNetProfile = netProfileLocal;

out:
        return isSuccess;
}


bool
StartNIFM()
{
    nn::Result       rval;

    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    if ( g_NIFMIsUp == true )
    {
        return true;
    }

    PRINT_AND_CALL(rval = nn::nifm::Initialize());
    if (rval.IsFailure() )
    {
        NN_LOG( "nn::nifm::Initialize failed" );
        return( false );
    }

    PRINT_AND_CALL(rval = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId()));
    if ( rval.IsFailure() )
    {
        NN_LOG( "nn::nifm::SetExclusiveClient failed" );
        return( false );
    }

    g_pNifmConnection = new nn::nifm::NetworkConnection;
    if ( nullptr == g_pNifmConnection )
    {
        NN_LOG(  "nn::nifm::NetworkConnection constructor failed" );
        return false;
    }

    nn::nifm::RequestHandle requestHandle = g_pNifmConnection->GetRequestHandle();
    nn::nifm::SetRequestConnectionConfirmationOption(requestHandle,nn::nifm::ConnectionConfirmationOption_Prohibited);
    nn::nifm::SetRequestPersistent(requestHandle, true);

    if( nn::util::InvalidUuid != g_netProfile )
    {
        char uuidBuffer[nn::util::Uuid::StringSize];
        NN_LOG("Setting Uuid.\n");
        PRINT_AND_CALL(rval = nn::nifm::SetRequestNetworkProfileId(requestHandle, g_netProfile));
        if ( 0 > sprintf(uuidBuffer, "Failed to set network profile id! Uuid: %s", g_netProfile.ToString(uuidBuffer, sizeof(uuidBuffer))))
        {
            NN_LOG( "TestSetup: sprint failed");
            return( false );
        }

        if ( rval.IsFailure() )
        {
            NN_LOG( uuidBuffer);
        }
    }

    // Submit asynchronous request to NIFM
    g_pNifmConnection->SubmitRequest();

    // Wait while NIFM brings up the interface
    while(!g_pNifmConnection->IsAvailable())
    {
        NN_LOG("Waiting for network interface...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }


    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    g_NIFMIsUp = true;

    return( true );
}


bool
StopNIFM()
{

    ////////////////////
    ////  Stop Testing
    ////////////////////


    if ( nullptr != g_pNifmConnection )
    {
        g_pNifmConnection->CancelRequest();
        delete g_pNifmConnection;
        g_pNifmConnection = nullptr;
    }

    // NIFM: Network Interface is Offline
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  D O W N     <===\n" );
    NN_LOG( "====================================\n" );

    g_NIFMIsUp = false;

    return( true );
}

}


bool
InitializeTesting()
{
    bool            isSuccess = true;
    nn::Result      result;
    char            testName[NATF::BaseTest::NameBufferLen];

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// Parse Options
    ///////////////////////////

    ERROR_IF(!ParseOptions(testName, &g_netProfile), "TestSetup: ParseOptions failed");


    ///////////////////////////
    //// Socket Library: Initialize
    ///////////////////////////

    result = nn::socket::Initialize( g_NIFMShutdownSocketMemoryPoolBuffer,
                                     nn::socket::DefaultSocketMemoryPoolSize,
                                     nn::socket::MinSocketAllocatorSize,
                                     nn::socket::DefaultConcurrencyLimit);

    if ( result.IsFailure() )
    {
        NN_LOG( "Failed calling nn::socket::Initialize\n" );
        goto out;
    }



    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumberOfThreads );


out:

    return( false );
}


static bool
TeardownTesting()
{
    bool       isSuccess = true;
    nn::Result result;

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    ////////////////////
    //// STOP NIFM
    ////////////////////

    if ( g_NIFMIsUp == true )
    {
        isSuccess  = StopNIFM();
    }

    ////////////////////
    //// STOP Socket API
    ////////////////////

    result = nn::socket::Finalize();
    if ( result.IsFailure() )
    {
        NN_LOG( "Failed calling: nn::socket::Finalize()\n" );
        goto out;
    }

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


void
ToggleNIFM( void * inArg )
{
    NN_LOG( "Sleeping\n" );
    nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );

    NN_LOG( "Stopping NIFM\n" );
    StopNIFM();
}


TEST(ApiUnit,NIFMShutdown_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,NIFMShutdown_Recv)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                recvBuff[128];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: Recv()" );
    rc = nn::socket::Recv( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: Recv()\n" );
    rc = nn::socket::Recv( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );


    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: Recv()\n" );
    rc = nn::socket::Recv( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: Recv()\n" );
    rc = nn::socket::Recv( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( clientSock );

out:

    return;

}   // NOLINT(impl/function_size)


TEST(ApiUnit,NIFMShutdown_RecvFrom)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                recvBuff[128];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: RecvFrom()" );
    rc = nn::socket::RecvFrom( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None, nullptr, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: RecvFrom()\n" );
    rc = nn::socket::RecvFrom( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None, nullptr, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: RecvFrom()\n" );
    rc = nn::socket::RecvFrom( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None, nullptr, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: RecvFrom()\n" );
    rc = nn::socket::RecvFrom( clientSock, recvBuff, sizeof( recvBuff ), nn::socket::MsgFlag::Msg_None, nullptr, nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( clientSock );


out:

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,NIFMShutdown_Send)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                sendBuff[10];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: Send()" );
        rc = nn::socket::Send( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None );
        if ( rc > 0 )
        {
            NN_LOG( "TCP Send Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc == 0)
        {
            NN_LOG( "TCP Send Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
           NN_LOG( "TCP Recv() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }

    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: Send()\n" );
    rc = nn::socket::Send( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );


    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7001 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Connect() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: Send()" );
        rc = nn::socket::Send( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None );
        if ( rc > 0 )
        {
            NN_LOG( "UDP Send Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc ==  0)
        {
            NN_LOG( "UDP Send Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
            NN_LOG( "UDP Recv() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }

    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: Send()\n" );
    rc = nn::socket::Send( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );


out:

    return;
}   // NOLINT(impl/function_size)


TEST(ApiUnit,NIFMShutdown_SendTo)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                sendBuff[10];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: SendTo()" );
        rc = nn::socket::SendTo( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None, nullptr, 0 );
        if ( rc > 0 )
        {
            NN_LOG( "TCP SendTo Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc == 0 )
        {
            NN_LOG( "TCP SendTo Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
           NN_LOG( "TCP SendTo() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }

    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: SendTo()\n" );
    rc = nn::socket::SendTo( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7001 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Connect() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: SendTo()" );
        rc = nn::socket::SendTo( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None, nullptr, 0 );
        if ( rc > 0 )
        {
            NN_LOG( "UDP SendTo Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc ==  0)
        {
            NN_LOG( "UDP SendTo Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
            NN_LOG( "UDP SendTo() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }

    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: SendTo()\n" );
    rc = nn::socket::SendTo( clientSock, sendBuff, 1, nn::socket::MsgFlag::Msg_None, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EPipe\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );


out:

    return;

}   // NOLINT(impl/function_size)

TEST(ApiUnit,NIFMShutdown_Write)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                sendBuff[10];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: Write()" );
        rc = nn::socket::Write( clientSock, sendBuff, 1 );
        if ( rc > 0 )
        {
            NN_LOG( "TCP Write Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc == 0)
        {
            NN_LOG( "TCP Write Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
           NN_LOG( "TCP Write() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "Passed - Receieved nn::socket::Errno::EPipe\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: Write()\n" );
    rc = nn::socket::Write( clientSock, sendBuff, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "Passed - Receieved nn::socket::Errno::EPipe\n" );


    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7001 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Connect() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    rc = 1;
    while( rc > 0 )
    {
        NN_LOG( "Before: Write()" );
        rc = nn::socket::Write( clientSock, sendBuff, 1 );
        if ( rc > 0 )
        {
            NN_LOG( "UDP Write Call wrote: %d bytes\n", rc );
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            continue;
        }
        else if ( rc ==  0)
        {
            NN_LOG( "UDP Write Call (ZERO): Peer performed orderly shutdown!  rc: %d\n", rc );
            break;
        }
        else
        {
            NN_LOG( "UDP Write() - GetLastError=<%d>\n", nn::socket::GetLastError() );
           break;
        }
    }

    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "Passed - Receieved nn::socket::Errno::EPipe\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: Write()\n" );
    rc = nn::socket::Write( clientSock, sendBuff, 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EPipe, "Error - Exepcted Errno = nn::socket::Errno::EPipe\n" );
    NN_LOG( "Passed - Receieved nn::socket::Errno::EPipe\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );


out:

    return;
}  // NOLINT(impl/function_size)


TEST(ApiUnit,NIFMShutdown_Read)
{
    nn::socket::SockAddrIn  clientAddr = { 0 }, serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    char                recvBuff[128];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: Read()" );
    rc = nn::socket::Read( clientSock, recvBuff, sizeof( recvBuff ) );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: Read()\n" );
    rc = nn::socket::Read( clientSock, recvBuff, sizeof( recvBuff ) );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( serverSock );
    nn::socket::Close( clientSock );

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // UDP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: Read()\n" );
    rc = nn::socket::Read( clientSock, recvBuff, sizeof( recvBuff ) );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "After: Read\n" );
    rc = nn::socket::Read( clientSock, recvBuff, sizeof( recvBuff ) );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received any data!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::ENetDown, "Error - Exepcted Errno = nn::socket::Errno::ENetDown\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::ENetDown\n" );

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Sockets
    //////////

    nn::socket::Close( clientSock );

out:

    return;
}  // NOLINT(impl/function_size)

TEST(ApiUnit,NIFMShutdown_Accept)
{
    nn::socket::SockAddrIn  serverAddr = { 0 };
    bool                isSuccess = true;
    int                 serverSock = 0;
    int                 rc = 0;
    int                 threadId = 0;

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Server
    //////////

    serverSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( serverSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( serverSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( serverSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    NN_LOG( "Before: Accept()" );
    rc = nn::socket::Accept( serverSock, static_cast<nn::socket::SockAddr*>(nullptr), nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received a new socket!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnAborted, "Error - Exepcted Errno = nn::socket::Errno::EConnAborted\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EConnAborted\n" );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // After NIFM Down
    //////////

    NN_LOG( "After: Accept()\n" );
    rc = nn::socket::Accept( serverSock, static_cast<nn::socket::SockAddr*>(nullptr), nullptr );
    ERROR_IF_AND_COUNT( rc > -1, "Should not have received a new socket!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnAborted, "Error - Exepcted Errno = nn::socket::Errno::EConnAborted\n" );
    NN_LOG( "PASSED - Received nn::socket::Errno::EConnAborted\n" );


    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Socket
    //////////

    nn::socket::Close( serverSock );

out:

    return;
}


TEST(ApiUnit,NIFMShutdown_Select)
{
    nn::socket::SockAddrIn  serverAddr = { 0 };
    bool                isSuccess = true;
    int                 tcpSock = 0;
    int                 udpSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    int                 idx = 0;
    int                 bytes = 0;
    nn::socket::FdSet   readSet, writeSet, exceptSet;
    unsigned char       recvBuff[128];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Sock
    //////////

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( tcpSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // UDP Sock
    //////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    nn::socket::FdSetZero( &readSet );
    nn::socket::FdSetZero( &writeSet );

    nn::socket::FdSetSet( tcpSock, &readSet );
    nn::socket::FdSetSet( udpSock, &readSet );

    nn::socket::FdSetSet( tcpSock, &writeSet );
    nn::socket::FdSetSet( udpSock, &writeSet );

    nn::socket::FdSetSet( tcpSock, &exceptSet );
    nn::socket::FdSetSet( udpSock, &exceptSet );

    NN_LOG( "Before: Select()" );
    rc = nn::socket::Select( udpSock + 1, &readSet, &writeSet, &exceptSet, nullptr );
    if ( rc == 0 )
    {
        NN_LOG( "No Descriptors are ready for I/O\n" );

        if ( nn::socket::FdSetIsSet( tcpSock, &readSet ) )
        {
            NN_LOG( "TCP Socket ready for read I/O\n" );
        }
        if ( nn::socket::FdSetIsSet( udpSock, &readSet ) )
        {
            NN_LOG( "UDP Socket ready for write I/O\n" );
        }
        if ( nn::socket::FdSetIsSet( tcpSock, &writeSet ) )
        {
            NN_LOG( "TCP Socket ready for write I/O\n" );
        }
        if ( nn::socket::FdSetIsSet( udpSock, &writeSet ) )
        {
            NN_LOG( "UDP Socket ready for read I/O\n" );
        }
        if ( nn::socket::FdSetIsSet( tcpSock, &exceptSet ) )
        {
            NN_LOG( "TCP Socket ready for Except I/O\n" );
        }
        if ( nn::socket::FdSetIsSet( udpSock, &exceptSet ) )
        {
            NN_LOG( "UDP Socket ready for Except I/O\n" );
        }
    }
    if ( rc > -1 )
    {
        NN_LOG( "There are %d descriptors ready for I/O\n", rc );
    }
    else
    {
        NN_LOG( "TCP Select() - GetLastError=<%d>\n", nn::socket::GetLastError() );
    }

    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    for( idx = 0; idx < 10; idx++ )
    {

        //////////
        // After NIFM Down
        //////////

        nn::socket::FdSetZero( &readSet );
        nn::socket::FdSetZero( &writeSet );

        nn::socket::FdSetSet( tcpSock, &readSet );
        nn::socket::FdSetSet( udpSock, &readSet );

        nn::socket::FdSetSet( tcpSock, &writeSet );
        nn::socket::FdSetSet( udpSock, &writeSet );

        nn::socket::FdSetSet( tcpSock, &exceptSet );
        nn::socket::FdSetSet( udpSock, &exceptSet );

        NN_LOG( "After: Select()\n" );
        rc = nn::socket::Select( udpSock + 1, &readSet, &writeSet, &exceptSet, nullptr );
        if ( rc == 0 )
        {
            NN_LOG( "No Descriptors are ready for I/O\n" );
        }
        if ( rc > -1 )
        {
            NN_LOG( "There are %d descriptors ready for I/O\n", rc );

            if ( nn::socket::FdSetIsSet( tcpSock, &readSet ) )
            {
                NN_LOG( "TCP Socket ready for read I/O\n" );
                bytes = nn::socket::Read( tcpSock, recvBuff, sizeof( recvBuff ) );
                NN_LOG( "TCP Read() Return Code: %d\n", bytes );
            }
            if ( nn::socket::FdSetIsSet( udpSock, &readSet ) )
            {
                NN_LOG( "UDP Socket ready for write I/O\n" );
                bytes = nn::socket::Read( udpSock, recvBuff, sizeof( recvBuff ) );
                NN_LOG( "UDP Read() Return Code: %d\n", bytes );
            }
            if ( nn::socket::FdSetIsSet( tcpSock, &writeSet ) )
            {
                NN_LOG( "TCP Socket ready for write I/O\n" );
                bytes = nn::socket::Write( tcpSock, recvBuff, 1 );
                if ( bytes < 0 ) NN_LOG( "Write Failed with Errno: %d\n", nn::socket::GetLastError() );
            }
            if ( nn::socket::FdSetIsSet( udpSock, &writeSet ) )
            {
                NN_LOG( "UDP Socket ready for read I/O\n" );
                bytes = nn::socket::Write( udpSock, recvBuff, 1 );
                if ( bytes < 0 ) NN_LOG( "Write Failed with Errno: %d\n", nn::socket::GetLastError() );
            }
            if ( nn::socket::FdSetIsSet( tcpSock, &exceptSet ) )
            {
                NN_LOG( "TCP Socket ready for Except I/O\n" );
            }
            if ( nn::socket::FdSetIsSet( udpSock, &exceptSet ) )
            {
                NN_LOG( "UDP Socket ready for Except I/O\n" );
            }
        }
        else
        {
            NN_LOG( "TCP Select() - GetLastError=<%d>\n", nn::socket::GetLastError() );
        }

    }

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Socket
    //////////

    nn::socket::Close( tcpSock );
    nn::socket::Close( udpSock );

out:

    return;

}  // NOLINT(impl/function_size)

TEST(ApiUnit,NIFMShutdown_Poll)
{
    nn::socket::PollFd  myPollFds[2];
    nn::socket::SockAddrIn  serverAddr = { 0 };
    bool                isSuccess = true;
    int                 tcpSock = 0;
    int                 udpSock = 0;
    int                 rc = 0;
    int                 threadId = 0;
    int                 bytes = 0;
    unsigned char       recvBuff[128];

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // TCP Sock
    //////////

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Target Address
    memset( &serverAddr, 0, sizeof( serverAddr ) );
    serverAddr.sin_port        = nn::socket::InetHtons( 7000 );
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &serverAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( tcpSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );

    //////////
    // UDP Sock
    //////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &serverAddr, sizeof( serverAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // NIFM transition from (Up to Down)
    //////////

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    myPollFds[1].fd      = udpSock;
    myPollFds[1].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[1].revents = nn::socket::PollEvent::PollNone;

    rc = nn::socket::Poll( (nn::socket::PollFd *)  &myPollFds, 2, 999999999 );
    if ( rc < 0 )
    {
       NN_LOG( "Poll Failed - Errno=<%d>\n", nn::socket::GetLastError() );
    }
    else
    {
      NN_LOG( "Number of POLL Fds Ready=%d\n", rc );
    }

    if ( ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "TCP Socket can READ\n" );
    }
    if ( ( myPollFds[0].revents & nn::socket::PollEvent::PollOut ) != nn::socket::PollEvent::PollNone  )
    {
        NN_LOG( "TCP Socket can WRITE\n" );
    }
    if ( ( myPollFds[1].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "UDP Socket Can READ\n");
    }
    if ( ( myPollFds[1].revents & nn::socket::PollEvent::PollOut ) != nn::socket::PollEvent::PollNone  )
    {
        NN_LOG( "UDP Socket can WRITE\n" );
    }


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    //////////
    // NIFM Is Down
    //////////

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    myPollFds[1].fd      = udpSock;
    myPollFds[1].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[1].revents = nn::socket::PollEvent::PollNone;

    rc = nn::socket::Poll( (nn::socket::PollFd *)  &myPollFds, 2, 999999999 );
    if ( rc < 0 )
    {
       NN_LOG( "Poll Failed - Errno=<%d>\n", nn::socket::GetLastError() );
    }
    else
    {
      NN_LOG( "Number of POLL Fds Ready=%d\n", rc );
    }

    if ( ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "TCP Socket can READ\n" );
        bytes = nn::socket::Read( tcpSock, recvBuff, sizeof( recvBuff ) );
        NN_LOG( "TCP Read() Return Code: %d\n", bytes );
    }
    if ( ( myPollFds[0].revents & nn::socket::PollEvent::PollOut ) != nn::socket::PollEvent::PollNone  )
    {
        NN_LOG( "TCP Socket can WRITE\n" );
    }
    if ( ( myPollFds[1].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "UDP Socket Can READ\n");
        bytes = nn::socket::Read( udpSock, recvBuff, sizeof( recvBuff ) );
        NN_LOG( "TCP Read() Return Code: %d\n", bytes );
    }
    if ( ( myPollFds[1].revents & nn::socket::PollEvent::PollOut ) != nn::socket::PollEvent::PollNone  )
    {
        NN_LOG( "UDP Socket can WRITE\n" );
    }

    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Socket
    //////////

    nn::socket::Close( tcpSock );
    nn::socket::Close( udpSock );

out:

    return;

}  // NOLINT(impl/function_size)


TEST(ApiUnit,NIFMShutdown_Connect)
{
    nn::socket::SockAddrIn  clientAddr = { 0 };
    bool                isSuccess = true;
    int                 clientSock = 0;
    int                 rc = 0;
    int                 threadId = 0;

WRAP_FAILING_TEST( "SIGLO-62919", "Socket API: [NX] Connect will not unblock when NIFM is offline" )
{

    //////////
    // Bring up NIFM
    //////////

    if ( g_NIFMIsUp == false )
    {
        StartNIFM();
    }

    //////////
    // Trigger NIFM
    //////////

    threadId = g_pThreadedTest->CreateThread( &ToggleNIFM, reinterpret_cast<void *>(ToggleNIFM), kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and NIFM (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started NIFM thread\n" );


    //////////
    // TCP Client
    //////////

    clientSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clientSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &clientAddr, 0, sizeof( clientAddr ) );
    clientAddr.sin_port        = nn::socket::InetHtons( 7000 );
    clientAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "1.2.3.4", &clientAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Connect( clientSock, (nn::socket::SockAddr *) &clientAddr, sizeof( clientAddr ) );
    ERROR_IF( rc < 0, "Connect() failed for TCP socket - Errno: %dn", nn::socket::GetLastError() );


    //////////
    // NIFM Down
    //////////

    while( g_NIFMIsUp == true )
    {
        NN_LOG( "Waiting for NIFM to finish stopping\n" );
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }


    //////////
    // Cleanup Thread
    //////////

    g_pThreadedTest->WaitThread( threadId );
    g_pThreadedTest->DestroyThread( threadId );

    //////////
    // Close Socket
    //////////

    nn::socket::Close( clientSock );

}

out:

    return;
}

TEST(ApiUnit,NIFMShutdown_Teardown)
{
    TeardownTesting();
}


#else // Windows

TEST(ApiUnit,NIFMShutdown_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
