﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/resolver/resolver_Client.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#endif

#include "Unit/testNet_LoopbackDnsServer.h"

namespace NATF {
namespace API {

///////////////////
// Constructor
///////////////////

LoopbackDnsServer::LoopbackDnsServer()
{
     memset( udpHost, 0, sizeof( udpHost ) );
     memcpy( udpHost, "127.0.0.1", strlen( "127.0.0.1" ) );
     udpPort = kDnsServerUDPPort;

     memset( tcpHost, 0, sizeof( tcpHost ) );
     memcpy( tcpHost, "127.0.0.1", strlen( "127.0.0.1" ) );
     tcpPort = kDnsServerTCPPort;

     dnsServerShouldRun = false;
     dnsServerRunning = false;
     UDPResponse = NULL;
     UDPResponseLen = 0;
     TCPResponse = NULL;
     TCPResponseLen = 0;
     serverUDP = -1;
     serverTCP = -1;
     acceptedTCP = -1;
     SetDNSRequestCB(  MakeDnsRequestCB );
     SetDNSResponseCB( MakeDnsResponseCB );
}

LoopbackDnsServer::LoopbackDnsServer( const char * in_udpHost, uint16_t in_udpPort,
                                      const char * in_tcpHost, uint16_t in_tcpPort )
{
     memset( udpHost, 0, sizeof( udpHost ) );
     memcpy( udpHost, in_udpHost, strlen( in_udpHost ) );
     udpPort = in_udpPort;

     memset( tcpHost, 0, sizeof( tcpHost ) );
     memcpy( tcpHost, in_tcpHost, strlen( in_tcpHost ) );
     tcpPort = in_tcpPort;

     dnsServerShouldRun = false;
     dnsServerRunning = false;
     UDPResponse = NULL;
     UDPResponseLen = 0;
     TCPResponse = NULL;
     TCPResponseLen = 0;
     serverUDP = -1;
     serverTCP = -1;
     acceptedTCP = -1;
     SetDNSRequestCB(  MakeDnsRequestCB );
     SetDNSResponseCB( MakeDnsResponseCB );
}



///////////////
// Destructor
///////////////

LoopbackDnsServer::~LoopbackDnsServer()
{
    if ( dnsServerShouldRun == true )
    {
        NN_LOG( "(Destructor): DNS Server sill running - stopping it\n" );
        Stop();
    }
}


///////////////////
// NX DNS traffic can be re-routed to use (loopback device): Resolver
//
///////////////////


bool
LoopbackDnsServer::SetDnsResolver( const char * dnsHost, uint16_t dnsPort )
{
    bool                         isSuccess = true;

#ifndef NN_BUILD_CONFIG_OS_WIN32

    nn::socket::SockAddrIn                     localAddr;
    int                                        rc;
    nn::socket::ResolverOption                 myOption;
    nn::socket::PrivateDnsAddressArrayArgument arg;

    // Tell Log what we are doing!
    NN_LOG( "=========================================\n" );
    NN_LOG( "==  Setting DNS Server to:  Host: %s, Port: %d\n", dnsHost, dnsPort );
    NN_LOG( "=========================================\n" );

    // Target DNS: Address
    memset(&localAddr, 0, sizeof(localAddr));

    // [DNS1] Fill in Port and Family
    localAddr.sin_port        = nn::socket::InetHtons( dnsPort );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // [DNS1] Turn IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, dnsHost, &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Tell NIFM to use this new DNS Server Address
    arg.count = 1;
    arg.addresses[0] = localAddr;

    myOption.key = static_cast<nn::socket::ResolverOptionKey>(nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer);
    myOption.type = nn::socket::ResolverOptionType::Pointer;
    myOption.size = sizeof( arg );
    myOption.data.pointerValue = (char *) &arg;

    rc = nn::socket::resolver::ResolverSetOption(myOption);
    ERROR_IF( rc == -1, "ResolverOptionPrivateKey::SetDnsServerAddressesPointer - Failed but Success was expected!  rc = %d\n", rc );

    return( true );

#endif

    isSuccess = false;
    ERROR_IF( isSuccess == false, "SetDnsAddressesPrivate() is only avaiable for BX/BIONIC\n" );

out:

    return( false );
}


int
LoopbackDnsServer::MakeDnsRequestCB( void *               thisObj,
                                     unsigned char *      dnsRqst,
                                     size_t &             dnsRqstLen,
                                     bool *               isUDP )
{
    return( 0 );
}


int
LoopbackDnsServer::MakeDnsResponseCB( void *              thisObj,
                                      unsigned char *     dnsRqst,
                                      size_t &            dnsRqstLen,
                                      unsigned char *     dnsResp,
                                      size_t &            dnsRespLen,
                                      bool *              isUDP,
                                      bool *              shouldContinue )
{
    struct DNS_HEADER_REC * pDnsHeader;
    uint16_t                saveId = 0x00;

    // restore this
    LoopbackDnsServer * thisPtr = (LoopbackDnsServer *) thisObj;

    // Save DNS Request Transaction Id
    pDnsHeader = (struct DNS_HEADER_REC *) dnsRqst;
    saveId = nn::socket::InetNtohs( pDnsHeader->Id );

    // Fill in the right Response
    memset( dnsResp, 0, dnsRespLen );

    if ( *isUDP == true )
    {
        memcpy( dnsResp, thisPtr->UDPResponse, thisPtr->UDPResponseLen );
        dnsRespLen = thisPtr->UDPResponseLen;
    }
    else
    {
        memcpy( dnsResp, thisPtr->TCPResponse, thisPtr->TCPResponseLen );
        dnsRespLen = thisPtr->TCPResponseLen;
    }

    // Fix DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;
    pDnsHeader->Id    = nn::socket::InetHtons( saveId );

    return( 0 );
}

void
LoopbackDnsServer::PrintHex( unsigned char *    buff,
                             size_t &           buffLen,
                             bool *             isReq,
                             bool *             isUDP )
{
    int       idx, count = 0;   // Counts where we are in the buffer
    char      hexbuff[1024];
    uint16_t  dnsLen;


    if ( *isReq == true )
    {
        NN_LOG( "[DNS Request: - Begin]\n" );
    }
    else
    {
        NN_LOG( "[DNS Response: - Begin]\n" );
    }

    memset( hexbuff, 0, sizeof( hexbuff ) );

    // If TCP - The Length bytes go first
    count = 0;
    if ( *isUDP == false )
    {
        dnsLen = (uint16_t) buffLen;
        sprintf( &hexbuff[count], " [0x%02x] [0x%02x]",
                 (uint8_t) (dnsLen >> 8 ),
                 (uint8_t) (dnsLen) );
        count = 14;
    }

    // UDP and TCP - Dump the formatted HEX
    for( idx = 0; idx < (int) buffLen; idx++ )
    {
        sprintf( &hexbuff[count], " 0x%02x,", buff[idx] );
        count = count + 6;
        if ( count >= 72 )
        {
            hexbuff[count++] = '\n';
            hexbuff[count]   = 0;
            NN_LOG( hexbuff );
            memset( hexbuff, 0, sizeof( hexbuff ) );
            count = 0;
        }
    }

    if (count > 0 )
    {
        hexbuff[count++] = '\n';
        hexbuff[count]   = 0;
        NN_LOG( hexbuff );
        memset( hexbuff, 0, sizeof( hexbuff ) );
        count = 0;
    }

    if ( *isReq == true )
    {
        NN_LOG( "[DNS Request: - End]\n" );
    }
    else
    {
        NN_LOG( "[DNS Response: - End]\n" );
    }
}


int
LoopbackDnsServer::ProcessDns( unsigned char *        dnsReq,
                               size_t &               dnsReqLen,
                               unsigned char *        dnsResp,
                               size_t &               dnsRespLen,
                               bool *                 isUDP,
                               bool *                 shouldContinue )
{
    int                rc;
    bool               isReq;

    //Invoke Make Dns Request Function
    rc = DNSRequestCB( this, dnsReq, dnsReqLen, isUDP );
    if ( rc < 0 )
    {
        NN_LOG( "Make DNS Request: Returned failure - Stopping\n" );
        return( -1 );
    }

    // Print Request out in Hex
    isReq = true;
    PrintHex( dnsReq, dnsReqLen, &isReq, isUDP );

    // Not large enough to hold DNS header?
    if ( dnsReqLen < 12 )
    {
        NN_LOG( "Received DNS Header is less than [12 bytes] - it is [%d bytes]!  Won't display DNS Header Info\n", dnsReqLen );
    }
    else
    {
        // Display DNS Request
        pDnsHeader = (struct DNS_HEADER_REC *) dnsReq;

        NN_LOG(
        "Rqst Dns: Id: %d, OPCODE: %d, QR: %d, AA: %d, TC: %d, RD: %d, RA: %d, AD: %d, CD: %d, rcode: %d, totQ: %d, totA %d, totAut: %d, totAdd: %d\n",
        nn::socket::InetNtohs( pDnsHeader->Id ), pDnsHeader->opcode, pDnsHeader->QR, pDnsHeader->AA, pDnsHeader->TC,
        pDnsHeader->RD, pDnsHeader->RA,  pDnsHeader->AD, pDnsHeader->CD,  pDnsHeader->rcode,
        nn::socket::InetNtohs( pDnsHeader->qdcount ),  nn::socket::InetNtohs( pDnsHeader->ancount ),
        nn::socket::InetNtohs( pDnsHeader->nscount ),  nn::socket::InetNtohs( pDnsHeader->arcount ) );
    }

    // Invoke Make Dns Response Function
    rc = DNSResponseCB( this, dnsReq, dnsReqLen, dnsResp, dnsRespLen, isUDP, shouldContinue );
    if ( rc < 0 )
    {
        NN_LOG( "Make DNS Response: Returned failure - Stopping\n" );
        return( -1 );
    }

    // If (DON'T Continue) - Stop
    if ( *shouldContinue == false )
    {
         return( 0 );
    }

    // Point at DNS Response
    pDnsHeader = (struct DNS_HEADER_REC *) dnsResp;

    NN_LOG(
    "Resp Dns: Id: %d, OPCODE: %d, QR: %d, AA: %d, TC: %d, RD: %d, RA: %d, AD: %d, CD: %d, rcode: %d, totQ: %d, totA %d, totAut: %d, totAdd: %d\n",
    nn::socket::InetNtohs( pDnsHeader->Id ), pDnsHeader->opcode, pDnsHeader->QR, pDnsHeader->AA, pDnsHeader->TC,
    pDnsHeader->RD, pDnsHeader->RA,  pDnsHeader->AD, pDnsHeader->CD,  pDnsHeader->rcode,
    nn::socket::InetNtohs( pDnsHeader->qdcount ),  nn::socket::InetNtohs( pDnsHeader->ancount ),
    nn::socket::InetNtohs( pDnsHeader->nscount ),  nn::socket::InetNtohs( pDnsHeader->arcount ) );

    // Print Response out in Hex
    isReq = false;
    PrintHex( dnsResp, dnsRespLen, &isReq, isUDP );

    return( 0 );

} // NOLINT(impl/function_size)



void
LoopbackDnsServer::Start( void *    inArg )
{
    nn::socket::TimeVal         mytime;
    nn::socket::FdSet                 readSet;

    int                    rc, optval = 1;
    int                    bytesSent, bytesRead;
    bool                   isSuccess = true, haveTCPLen, isUDP, shouldContinue;
    unsigned char          recvBuff[8092];
    ssize_t                rc1;
    size_t                 recvBuffSize, bytesLeft;
    uint16_t               dnsLen;

    /////////////////////////
    // D N S   S E R V E R    S T A R T I N G
    /////////////////////////

    if ( dnsServerRunning == true )
    {
        NN_LOG( "DNS Server is already running!\n" );
        return;
    }
    else
    {
        dnsServerRunning = true;
        NN_LOG( "==> DNS Server Starting up!\n" );
        NN_LOG( "==> Thread: %d - Starting on CPU: %lld\n", 1, kDnsServerCore );
    }

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue,
                                kDnsServerCore );

    /////////////////////////
    // SERVER:  TCP
    /////////////////////////

    // Server TCP: Socket
    serverTCP = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( serverTCP < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server TCP: Address
    memset( &tcpAddr, 0, sizeof( tcpAddr ) );
    tcpAddr.sin_family      = nn::socket::Family::Af_Inet;
    tcpAddr.sin_port        = nn::socket::InetHtons( tcpPort  );

    // Server TCP: Loopback
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, tcpHost, &tcpAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server TCP: SetSockOpt()
    rc = nn::socket::SetSockOpt( serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "SetSockOpt() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server TCP: Bind
    rc = nn::socket::Bind( serverTCP, reinterpret_cast<nn::socket::SockAddr *>( &tcpAddr ), sizeof( tcpAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server: Listen
    rc = nn::socket::Listen( serverTCP, 5 );
    ERROR_IF( rc < 0, "Listen() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    NN_LOG( "DNS TCP: Listen - Addr: %s, Port: %d\n", tcpHost, tcpPort );


    /////////////////////////
    // SERVER:  UDP
    /////////////////////////

    // Server UDP: Socket
    serverUDP = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( serverUDP < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server UDP: Address
    memset( &udpAddr, 0, sizeof( udpAddr ) );
    udpAddr.sin_family      = nn::socket::Family::Af_Inet;
    udpAddr.sin_port        = nn::socket::InetHtons( udpPort );

    // Server UDP: Loopback
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, udpHost, &udpAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server UDP: SetSockOpt()
    rc = nn::socket::SetSockOpt( serverUDP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "SetSockOpt() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server UDP: Bind
    rc = nn::socket::Bind( serverUDP, reinterpret_cast<nn::socket::SockAddr *>( &udpAddr ), sizeof( udpAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    NN_LOG( "DNS UDP: Addr: %s, Port: %d\n", udpHost, udpPort );


    /////////////////////////
    // Main Loop
    /////////////////////////

    dnsServerShouldRun = true;
    while( dnsServerShouldRun == true )
    {
        nn::socket::FdSetZero(&readSet);

        nn::socket::FdSetSet(serverTCP, &readSet);
        nn::socket::FdSetSet(serverUDP, &readSet);

        mytime.tv_sec  = 1;
        mytime.tv_usec = 0;

        // Select
        rc = nn::socket::Select( serverUDP + 1, &readSet, nullptr, nullptr, &mytime);

        NN_LOG( "Select rc = %d\n", rc );

        if ( rc < 0 )
        {
            NN_LOG( "Select Failed - Errno: %^d\n", nn::socket::GetLastError() );
            goto out;
        }

        if ( rc == 0 )
        {
            continue;    // Select timed out
        }

        // UDP Ready to read?
        if ( nn::socket::FdSetIsSet( serverUDP, &readSet ) )
        {
            NN_LOG( "Processing: [UDP DNS Request]\n" );

            //  DNS Request (From) Client
            memset( &recvAddr, 0, sizeof( nn::socket::SockAddrIn ) );
            recvAddrSize = sizeof( nn::socket::SockAddrIn );
            recvBuffSize = sizeof( recvBuff );
            rc1 = nn::socket::RecvFrom( serverUDP, recvBuff, recvBuffSize, nn::socket::MsgFlag::Msg_None,
                     reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), &recvAddrSize );

            NN_LOG( "UDP: RecvFrom rc = %lld\n", rc1 );

            if ( rc1 < 0 )
            {
                NN_LOG( "DNS UDP: RecvFrom(): Failed - errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            // Process DNS Request / Response
            recvBuffSize = rc1;
            isUDP = true;
            shouldContinue = true;
            if ( ProcessDns( recvBuff, recvBuffSize, recvBuff, recvBuffSize, &isUDP, &shouldContinue ) != 0 )
            {
                NN_LOG( "ProcessDNS()!\n" );
                goto out;
            }

            // If (DON'T Continue)..
            if ( shouldContinue == false )
            {
                continue;
            }

            // DNS Response (From) this Server
            recvAddrSize = sizeof( nn::socket::SockAddrIn );
            rc1 = nn::socket::SendTo( serverUDP, recvBuff, recvBuffSize, nn::socket::MsgFlag::Msg_None,
                    reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), recvAddrSize );

            NN_LOG( "UDP: SendTo rc = %lld\n", rc1 );

            if ( rc1 < 0 )
            {
                NN_LOG( "DNS UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            // Jump to Top
            continue;
        }

        // TCP Ready to read?
        if ( nn::socket::FdSetIsSet( serverTCP, &readSet ) )
        {
            NN_LOG( "Processing: [TCP DNS Request]\n" );

            // Accept Client socket
            recvAddrSize = sizeof( recvAddr );
            acceptedTCP = nn::socket::Accept( serverTCP, reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), &recvAddrSize);
            if ( acceptedTCP < 0 )
            {
                NN_LOG( "serverTCP: Accept failed!  Errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            NN_LOG( "TCP DNS Request received from: [%s:%d]\n",
                                nn::socket::InetNtoa( recvAddr.sin_addr ),
                                nn::socket::InetNtohs( recvAddr.sin_port ) );

            // Keep Reading.. byte stream..
            bytesRead  = 0;
            bytesLeft  = 2;           // Want 2x bytes to know how long the DNS Request is
            haveTCPLen = false;        // Looking for TCP Length bytes
            while( bytesLeft > 0 )
            {
                 rc = nn::socket::Read( acceptedTCP, &recvBuff[bytesRead], bytesLeft );
                 if ( rc < 0 )
                 {
                     if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                     {
                         break;
                     }

                     NN_LOG( "acceptedTCP: nn::socket::Read() Failed - Errno: %d\n", nn::socket::GetLastError() );
                     goto out;
                 }

                 bytesRead = bytesRead + rc;
                 bytesLeft = bytesLeft - rc;

                 // Determine TCP DNS Request len if needed
                 if (   ( haveTCPLen == false )  &&
                        ( bytesRead > 1 )       )
                 {
                     dnsLen = (8 << recvBuff[0]) | recvBuff[1];  // Yes, this is endian safe
                     NN_LOG( "[TCP] DNS Request Len: %d\n", dnsLen );
                     bytesLeft  = dnsLen;
                     bytesRead  = 0;
                     haveTCPLen = true;
                 }
            }

            // Process DNS Request / Response
            recvBuffSize = bytesRead;
            isUDP = false;
            shouldContinue = true;
            if ( ProcessDns( recvBuff, recvBuffSize, recvBuff, recvBuffSize, &isUDP, &shouldContinue ) != 0 )
            {
                NN_LOG( "Process DNS Request failed!\n" );
                goto out;
            }

            // If (DON'T Continue)..
            if ( shouldContinue == false )
            {
                continue;
            }

            // DNS Response (From) this Server
            // Write Length Bytes
            NN_LOG( "[TCP DNS]: Writing DNS Response Length Bytes (Value: %d)\n", recvBuffSize );
            dnsLen = nn::socket::InetHtons( (uint16_t) recvBuffSize );
            rc = nn::socket::Write( acceptedTCP, (char *) &dnsLen, 2 );
            ERROR_IF( rc != 2, "TCP DNS: Tried to Write 2x length bytes failed!" );

            // Write DNS Response Message
            bytesSent = 0;
            bytesLeft = recvBuffSize;
            while( bytesLeft > 0 )
            {
                 rc = nn::socket::Write( acceptedTCP, &recvBuff[bytesSent], bytesLeft );
                 if ( rc < 0 )
                 {
                     NN_LOG( "acceptedTCP: nn::socket::Write() Failed - Errno: %d\n", nn::socket::GetLastError() );
                     goto out;
                 }

                 bytesSent = bytesSent + rc;
                 bytesLeft = bytesLeft - rc;
            }

            // BLOCKING READ:  Wait for Client to close
            NN_LOG( "[DNS TCP]: Waiting for client to close\n" );
            rc = nn::socket::Read( acceptedTCP, &recvBuff, sizeof( recvBuff ) );
            NN_LOG( "[DNS TCP]: DNS Client has closed\n" );

            // Close Accepted Socket
            nn::socket::Close( acceptedTCP );

            // Jump to TOP
            continue;
        }
    }

out:

    NN_LOG( "==> DNS Server -- told to Exit\n" );


    // Close UDP Socket
    if ( serverUDP > -1 )
    {
        nn::socket::Close( serverUDP );
        serverUDP = -1;
    }

    // Close TCP Socket
    if ( serverTCP > -1 )
    {
        nn::socket::Close( serverTCP );
        serverTCP = -1;
    }

    // DNS Server - Stopped Running
    dnsServerRunning   = false;
    dnsServerShouldRun = false;

    return;

}  // NOLINT(impl/function_size)


int
LoopbackDnsServer::Stop()
{
    if ( dnsServerShouldRun == false )
    {
        NN_LOG( "[DNS Server] is not running - nothing to do\n" );
        return( 0 );
    }

    NN_LOG( "[DNS Server] is runnimg - stopping it\n" );
    dnsServerShouldRun = false;

    while( dnsServerRunning == true )
    {
        NN_LOG( "Waiting for DNS Server to shutdown...\n" );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG( "[DNS Server] - STOPPED!\n" );

    return( 0 );
}



}}; // namespace:NAFT / API
