﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Listen_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Listen_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( outAddr );
    int                     tcpSock = -1, udpSock = -1, tcpServer = -1, acceptedSock = -1, rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    char *                  asciiAddr;
    int                     sockets[32];
    int                     numSockCreated = 0, queueSize = 2;
    double                  queueLimit = queueSize * 1.5;
    bool                    isSuccess = true;

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////
    NN_LOG( "Testing descriptor:" );

    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );
        if ( idx == g_echoServer->serverUDP || idx == g_echoServer->serverTCP )
        {
            continue;
        }

        rc = nn::socket::Listen( idx, 5 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf)
            {
                NN_LOG( "\nListen() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno: %d\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nListen() succeeded but fds: %d. Expected to fail.\n", idx );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf failed\n" );
    isSuccess = true;

    ////////////
    // Test nn::socket::Errno::EOpNotSupp
    ///////////
    NN_LOG( "Testing nn::socket::Errno::EOpNotSupp\n" );

    rc = nn::socket::Listen( g_echoServer->serverUDP, 5 );
    ERROR_IF_AND_COUNT( rc != -1, "Listen() succeeded but should of failed\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EOpNotSupp, "Listen() should of failed with nn::socket::Errno::EOpNotSupp. Errno=<%d>\n", myError );

    ////////////
    // Test nn::socket::Errno::EIsConn
    ///////////

    NN_LOG( "Testing nn::socket::Errno::EIsConn\n" );

    // Make TCP Client Connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpSock, 5 );
    ERROR_IF_AND_COUNT( rc != -1, "Listen() succeded but should of failed\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EIsConn, "Listen() should of failed with nn::socket::Errno::EIsConn. Errno=<%d>\n", myError );

    nn::socket::Close( tcpSock );
    tcpSock = -1;

    ////////////
    // Test negative backlog
    ///////////
    NN_LOG( "Testing negative backlog \n" );
    rc = nn::socket::Listen( g_echoServer->serverTCP, -2 );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( rc < 0, "Listen() failed but should have succeded. Errno=<%d>\n", myError );

    ////////////
    // Test non-negative backlog
    ///////////
    NN_LOG( "Testing non-negative backlog:" );

    for ( idx = 0; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );
        rc = nn::socket::Listen( g_echoServer->serverTCP, idx );
        if ( rc < 0 )
        {
            NN_LOG("\nListen() failed but expect to succeded. Errno=<%d>\n", nn::socket::GetLastError() );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing non-negative backlog parameter failed!\n" );
    isSuccess = true;

    rc = nn::socket::Listen( g_echoServer->serverTCP, 32 );
    ERROR_IF( rc < 0, "Listen() failed setting TCP echo server back to 32. Errno=<%d>\n" );

    ////////////
    // Exceeding limit queue
    ////////////
    NN_LOG( "Testing limit queue\n" );

    // Create TCP Server
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Socket() failed! Errno=<%d>\n", nn::socket::GetLastError() );

    // Server: Fcntl - nn::socket::FcntlFlag::O_NonBlock
    rc = nn::socket::Fcntl( tcpServer, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 8051 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( tcpServer, reinterpret_cast<nn::socket::SockAddr *>( &localAddr ), sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, queueSize );
    ERROR_IF( rc < 0, "Listen() failed! errno=<%d>\n\n", nn::socket::GetLastError() );


    idx = 0;
    while( isSuccess && numSockCreated < 32 )
    {
        // Make TCP Client Connection
        tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
        ERROR_IF( tcpSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );
        sockets[idx] = tcpSock;
        numSockCreated++;

        // Client: Set Non-Blocking
        rc = NATF::API::COMMON::SetNonBlocking( tcpSock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        // Client: Connect
        rc = nn::socket::Connect( tcpSock, reinterpret_cast<nn::socket::SockAddr *>( &localAddr ), sizeof (localAddr ) );
        if (rc < 0 )
        {
            if ( idx == queueLimit + 1 )
            {
                NN_LOG( "Connect() Failed as expected. Errno=<%d>\n", nn::socket::GetLastError() );
                break;
            }

            isSuccess = false;
        }

        idx++;
    }

    // Windows has higher limits
    // ERROR_IF_AND_COUNT( !isSuccess, "Limit() did not fail when reaching max queue size\n");

    // Close all opened socketes
    for ( idx = 0; idx < numSockCreated; idx++ )
    {
        nn::socket::Close( sockets[idx] );
    }

    tcpSock = -1;

    nn::socket::Close( tcpServer );
    tcpServer = -1;


    ////////////
    // Test nn::socket::Errno::EDestAddrReq
    ///////////
    NN_LOG( "Testing nn::socket::Errno::EDestAddrReq\n" );

    // Create TCP socket
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Socket() failed! Errno=<%d>\n", nn::socket::GetLastError() );

    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9021 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    rc = nn::socket::Bind( tcpServer, reinterpret_cast<nn::socket::SockAddr *>( &localAddr ), sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, 5 );
    ERROR_IF_AND_COUNT( rc < 0, "Listen() should have succeeded but failed. Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::GetSockName( tcpServer, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc < 0, "GetSockName() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    asciiAddr = nn::socket::InetNtoa( outAddr.sin_addr );
    NN_LOG( "===> Returned Ascii Address: %s\n", asciiAddr );
    NN_LOG( "===> Returned Port Number: %d\n",   nn::socket::InetNtohs( outAddr.sin_port ) );

    nn::socket::Close( tcpServer );
    tcpServer = -1;


out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( tcpServer > -1 )
    {
        nn::socket::Close( tcpServer );
        tcpServer = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }
}// NOLINT(impl/function_size)


TEST(ApiUnit,Win_Listen_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
