﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_GetPeerName_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_GetPeerName_Various_Tests)
{
    nn::socket::SockAddrIn  outAddr = { 0 }, localAddr = { 0 };
    nn::socket::SockLenT    outAddrLen = sizeof( nn::socket::SockAddr );
    int                     tcpSock = -1, udpSock = -1, peerSock = -1, acceptedSock = -1, rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    char *                  asciiAddr;
    bool                    isSuccess = true;

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing File Descriptor for nn::socket::Errno::EBadf:\n" );

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );

        outAddrLen = sizeof( nn::socket::SockAddr );
        rc = nn::socket::GetPeerName( idx, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
        if ( rc == 0 )
        {
            NN_LOG( "\nGetPeerAddr() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }

        myError = nn::socket::GetLastError();
        if ( myError != nn::socket::Errno::EBadf )
        {
            // Echo Server [TCP] and [UDP] sockets should return nn::socket::Errno::ENotConn
            if (  ( idx == g_echoServer->serverUDP ) ||
                  ( idx == g_echoServer->serverTCP )  )
            {
                if ( myError == nn::socket::Errno::ENotConn )
                {
                    continue;
                }

                // Fall Thru
            }

            NN_LOG( "\nGetPeerAddr() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Test NULL Receive address
    ///////////

    NN_LOG( "Testing: [TCP] - NULL Response buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( tcpSock, static_cast<nn::socket::SockAddr*>(nullptr), &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeeded but should have failed [TCP]\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval , "GetPeerAddr() succeeded but should have failed [TCP]\n" );


    ////////////
    // Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (No Address Len) - nn::socket::Errno::EInval\n" );

    outAddrLen = 0;
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeeded but should have failed [TCP]\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval , "GetPeerAddr() succeeded but should have failed [TCP]\n" );

    ////////////
    // Test Address Length to short
    //
    // This passes on FreeBSD with the correct result 'truncted' and stored in a too short buffer.
    // This results in a wrong answer returned with no indication that the result was truncated.
    ///////////

    NN_LOG( "Testing: [TCP] - (Address Len too short == 3)\n" );

    outAddrLen = 3;
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeeded but should have failed [TCP]\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EInval , "GetPeerAddr() succeeded but should have failed [TCP]\n" );


    ////////////
    // Test TCP Peer
    ///////////

    NN_LOG( "Testing: [TCP] Peer (Echo Server)\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetPeerAddr() failed, but should have succeded [TCP]\n" );

    asciiAddr = nn::socket::InetNtoa( outAddr.sin_addr );
    NN_LOG( "===> Returned Ascii Address: %s\n", asciiAddr );

    ERROR_IF_AND_COUNT( strlen( asciiAddr ) != strlen( "127.0.0.1" ), "Returned Address is not the right length!\n" );
    ERROR_IF_AND_COUNT( memcmp( asciiAddr, "127.0.0.1", strlen( "127.0.0.1" ) ) != 0,
                        "Returned Address does not equal [127.0.0.1] - Actual returned address: %s\n", asciiAddr );


    ////////////
    // Test UDP Peer
    ///////////

    NN_LOG( "Testing: [UDP] Peer (Echo Server)\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( udpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc != 0, "GetPeerAddr() failed, but should have succeded [UDP]\n" );

    asciiAddr = nn::socket::InetNtoa( outAddr.sin_addr );
    NN_LOG( "===> Returned Ascii Address: %s\n", asciiAddr );

    ERROR_IF_AND_COUNT( strlen( asciiAddr ) != strlen( "127.0.0.1" ), "Returned Address is not the right length!\n" );
    ERROR_IF_AND_COUNT( memcmp( asciiAddr, "127.0.0.1", strlen( "127.0.0.1" ) ) != 0,
                        "Returned Address does not equal [127.0.0.1] - Actual returned address: %s\n", asciiAddr );


    ////////////
    // Test UDP (NO Peer) - nn::socket::Errno::ENotConn
    ///////////

    NN_LOG( "Testing: [UDP] No Peer - nn::socket::Errno::ENotConn\n" );

    // Close UDP Socket
    nn::socket::Close( udpSock );
    udpSock = -1;

    // Create UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Target Address
    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: Bind (self)
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() Failed but Success was expected!" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::GetPeerName( udpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeded but should have failed [UDP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENotConn, "Expected errno: nn::socket::Errno::ENotConn, but actually got errno: %d [UDP]\n", myError );

    // Close UDP Socket
    nn::socket::Close( udpSock );
    udpSock = -1;


    ////////////
    // Test TCP (NO Peer) - nn::socket::Errno::ENotConn
    ///////////

    NN_LOG( "Testing: [TCP] No Peer - nn::socket::Errno::ENotConn\n" );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    // Create TCP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Call GetPeerName
    rc = nn::socket::GetPeerName( tcpSock, (nn::socket::SockAddr *) &outAddr, &outAddrLen );
    ERROR_IF_AND_COUNT( rc == 0, "GetPeerAddr() succeded but should have failed [UDP]\n" );

    // Check Result
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENotConn, "Expected errno: nn::socket::Errno::ENotConn, but actually got errno: %d [UDP]\n", myError );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( peerSock > -1 )
    {
        nn::socket::Close( peerSock );
        peerSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

}    // NOLINT(impl/function_size)


TEST(ApiUnit,Win_GetPeerName_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
