﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: DISABLED
//
// #define ENABLE_FAILING_TESTS


#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/os/os_Thread.h>

#include <cstdio>   // sprintf
#include <cctype>   // isprint

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_LoopbackDnsServer.h"     // Loopback DNS

#include <nn/socket/socket_ApiPrivate.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/resolver/resolver_Client.h>
#endif

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

// Anonynous Name Space

namespace {

typedef struct {

    const char *        desc;
    nn::socket::Family  family;
    const char *        dottedIP;
    uint32_t            dottedIPLen;
    bool                result;
    const char *        regex;
    int                 syserrno;

} GHBAArgumentResult;

static GHBAArgumentResult    g_myArgumentResults[] =
{

    /* description                                                                               */
    /* family    IP Address (ASCII)          Length of IP   result           regex    syserrno  */

    { "Call GHBA for: [B-ROOT USC] IPv4",
      nn::socket::Family::Af_Inet,   "192.228.79.201",                4,        true,          ".root-servers.",   0 },

    { "Call GHBA for: [B-ROOT USC] with an IPv6 aAddress",
      nn::socket::Family::Af_Inet6,   "2001:500:84::b",               6,        false,          ".root-servers.",  0 },

    { "Call GHBA for: [B-ROOT USC] IPv4 address - but IPv6 length",
      nn::socket::Family::Af_Inet,   "192.228.79.201",                6,        true,          ".root-servers.",   0 },

    { "Call GHBA for: [B-ROOT USC] IPv4 address - but IPv6 family",
      nn::socket::Family::Af_Inet6,   "192.228.79.201",               4,        false,         ".root-servers.",   0 },

    { "Call GHBA for: [B-ROOT USC] IPv4 address - but IPv6 length and family",
      nn::socket::Family::Af_Inet6,   "192.228.79.201",               6,        false,         ".root-servers.",   0 },

    { "Call GHBA for: [B-ROOT USC] IPv4 address - but length (max 32)",
      nn::socket::Family::Af_Inet,   "192.228.79.201",            0x7FFFFFFF,    true,        ".root-servers.",   0 },

    { "Call GHBA for: [B-ROOT USC] IPv4 address - but length HIGH values",
      nn::socket::Family::Af_Inet,   "192.228.79.201",            0xFFFFFFFF,    true,         ".root-servers.",  0 },

    { "Call GHNA for [E-ROOT] - NASA AMES Research Center]: 192.203.230.10",
      nn::socket::Family::Af_Inet,   "192.203.230.10",                4,         true,          ".root-servers.net", 0 }
};


} // Anonymous namespace


static bool
InitializeTesting()
{
    nn::Result                result;
    bool                      isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    return( true );

out:

    return( false );
}


static bool
TeardownTesting()
{
    bool    isSuccess = true;

    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}


static bool
CallGetHostByAddr( const char *  dottedIP, nn::socket::SockLenT addrLen, nn::socket::Family family, nn::socket::HostEnt ** rtrnHosts )
{
    nn::socket::InAddr      ipv4Addr = { 0 };
    nn::socket::HostEnt *   respHosts = nullptr;
    nn::socket::InAddr **   addr_list;
    int                     idx = 0, rc = 0;

    // Initialize Return Value
    *rtrnHosts = NULL;

    // Turn English IP Address into octets
    if ( addrLen > 3 )
    {
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, dottedIP, &ipv4Addr );
        if ( rc != 1 )
        {
            NN_LOG( "InetPton() failed - rc: %d errno: %d\n", rc, nn::socket::GetLastError() );
            goto out;
        }
    }

    NN_LOG( "Loopup IP Address: %d.%d.%d.%d, Len: %d, family: %d\n",
            (uint8_t) ( ipv4Addr.S_addr ), (uint8_t) ( ipv4Addr.S_addr >> 8 ),
            (uint8_t) ( ipv4Addr.S_addr >> 16 ), (uint8_t) ( ipv4Addr.S_addr >> 24 ),
                        addrLen, family );

    respHosts = nn::socket::GetHostEntByAddr( &ipv4Addr, addrLen, family );
    if ( respHosts == nullptr )
    {
        NN_LOG( "nn::socket::GetHostByAddr() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
                 nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {

           NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {

           NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    // Save Return Pointer
    *rtrnHosts = respHosts;

    return( true );

out:

    return( false );
}

TEST(ApiUnit,Win_GetHostByAddr_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_GetHostByAddr_Argument_Results)
{
    nn::socket::HostEnt * respHosts;
    unsigned int        idx;
    bool                rc, isSuccess = false, keepgoing = true;

    size_t myResultSize = sizeof( g_myArgumentResults ) / sizeof( GHBAArgumentResult  );

    for( idx = 0; idx < myResultSize; idx++ )
    {
        NN_LOG( "Test Description:  %s\n", g_myArgumentResults[idx].desc );
        NN_LOG( "Inputs: Address Family: %d, Dotted IP Address: %s, Dotted IP Length: %d, Result: %s, regex: %s, syserrno: %d\n",
                g_myArgumentResults[idx].family, g_myArgumentResults[idx].dottedIP,
                g_myArgumentResults[idx].dottedIPLen, ( g_myArgumentResults[idx].result == true ? "true" : "false" ),
                g_myArgumentResults[idx].regex,  g_myArgumentResults[idx].syserrno );

        // Call: GetHostEntByAddr()
       rc = CallGetHostByAddr( g_myArgumentResults[idx].dottedIP, g_myArgumentResults[idx].dottedIPLen,
                               g_myArgumentResults[idx].family, &respHosts );

       if ( g_myArgumentResults[idx].result == true )
       {
           if ( rc != true ) {
               NN_LOG( "Return Code Failed, but should have Succeeded.  Errno=<%d>\n", nn::socket::GetLastError() );
               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }
       }
       else
       {
           if ( rc == true )
           {
               ERROR_IF_AND_COUNT( keepgoing == true, "Return Code is Success, but should have Failed! idx: %d\n", idx );
               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
            }
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }

out:

    return;
}



#if 0

TEST(ApiUnit,Win_GetHostByAddr_AddressFamily)
{
    nn::socket::HostEnt *    respHosts;
    int                 idx;
    bool                rc, isSuccess = false, keepgoing = true;

    // Normal DNS Rqst/Resp Parsers
    g_loopbackDnsServer->SetDNSRequestCB(  g_loopbackDnsServer->MakeDnsRequestCB );
    g_loopbackDnsServer->SetDNSResponseCB( GetAddressFamilyCB );

    ///////////////////
    // Simple Google
    ///////////////////

    g_loopbackDnsServer->UDPResponse = (unsigned char *) Amazon_com_simple;
    g_loopbackDnsServer->UDPResponseLen = sizeof( Amazon_com_simple );

    // Address Family Tests
    for( idx = -5; idx < 255; idx++ )
    {
        // Skip these
        if ( idx == nn::socket::Family::Af_Unspec ) continue;
        if ( idx == nn::socket::Family::Af_Inet )   continue;
#ifdef NN_BUILD_CONFIG_OS_WIN32
        if ( idx == nn::socket::Family::Af_Inet6 )  continue;
#endif

        // Initalize Flag: If flag tripped then GHBA connected to a nameserver.
        g_workVarBool = false;

        // Call GetHostByAddress()
        rc = CallGetHostByAddr( "192.203.230.10", 4, idx, &respHosts );

        // Print Result:
        NN_LOG( "Return code: %d, Var Bool: %d\n", rc, g_workVarBool );

        if ( rc == true || g_workVarBool == true )
        {
            NN_LOG( "GHNA: (Address Family) = %d Succeeded - but should have FAILEDi!  Work Var: %d\n",
                        idx, g_workVarBool );
            ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }

out:

    return;
}

#endif

TEST(ApiUnit,Win_GetHostByAddr_Teardown)
{
    TeardownTesting();
}


}}  // Namespace: NATF::API
