﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: DISABLED
//
// #define ENABLE_FAILING_TESTS

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/os/os_Thread.h>

#include <cstdio>   // sprintf
#include <cctype>   // isprint

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************


static bool
CallGetAddrInfo( const char *in_domainname, const char *in_service,
                 nn::socket::AddrInfo *in_hints, nn::socket::AddrInfo *out_resp, nn::socket::AiErrno * out_error )
{
    nn::socket::SockAddrIn  *ipv4 = nullptr;
    nn::socket::AddrInfo    *respIdx = nullptr;

    *out_error = nn::socket::GetAddrInfo( in_domainname, in_service, in_hints, &out_resp );
    if ( nn::socket::AiErrno::EAi_Success != *out_error )
    {
       goto out;
    }

    for ( respIdx = out_resp; respIdx; respIdx = respIdx->ai_next)
    {
        ipv4 = (nn::socket::SockAddrIn *) (respIdx->ai_addr);

        NN_LOG( "Family: %d, Socktype: %d, Protocol: %d, Flags: 0x%0x, IPAddr: %s, Port: %d, Name: %s\n",
                respIdx->ai_family, respIdx->ai_socktype, respIdx->ai_protocol,
                respIdx->ai_flags, nn::socket::InetNtoa( ipv4->sin_addr ),
                nn::socket::InetNtohs( ipv4->sin_port ), respIdx->ai_canonname );
    }

    return( true );

out:

    return( false );
}

TEST(ApiUnit,Win_GetAddrInfo_Initialize)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    INITIALIZE_TEST_COUNTS;

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

out:
    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

#ifndef EAI_MAX
static const int EAI_MAX = 15;
#endif


TEST(ApiUnit,Win_GetAddrInfo_GAIStrError)
{
    int       idx;
    char *    errorMsg;
    bool      isSuccess = false;

    for( idx = -5; idx < static_cast<int>(nn::socket::AiErrno::EAi_Max) + 5; idx++ )
    {
        errorMsg = (char *) nn::socket::GAIStrError( static_cast<nn::socket::AiErrno>(idx) );
        ERROR_IF_AND_COUNT( errorMsg == NULL, "GAIStrError() returned NULL for Index: %d\n", idx );
        NN_LOG( "GAIStrError(): Idx: %d, Message: %s\n", idx, errorMsg );
    }

out:

    return;
}

TEST(ApiUnit,Win_GetAddrInfo_FreeAddrInfo_NULL)
{
    NN_LOG( "FreeAddrInfo - Null Pointer\n" );
    nn::socket::FreeAddrInfo( static_cast<nn::socket::AddrInfo*>(nullptr) );
}


typedef struct {

    const char *                desc;
    nn::socket::Family          family;
    nn::socket::Type            socktype;
    nn::socket::AddrInfoFlag    flags;
    const char *                domainname;
    const char *                service;
    bool                        result;
    nn::socket::AiErrno         error;
    nn::socket::Errno           syserrno;

} GAIArgumentResult;


namespace {


static GAIArgumentResult    g_myArgumentResults[] =
{

    /* description                                                          */
    /* family   socktype    flags               domainname      service     result  error       syserrno*/

         // 'Service' Tests
         { "Call GAI with a NULL service name: Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream,    nn::socket::AddrInfoFlag::Ai_CanonName,           "example.com",      0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI for records that support UDP (nn::socket::Type::Sock_Dgram) for a TCP-only service (telnet) - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::AddrInfoFlag::Ai_CanonName,           "example.com",      "telnet",   false,  nn::socket::AiErrno::EAi_Service,    nn::socket::Errno::ESuccess },

         { "Call GAI for records that support TCP (nn::socket::Type::Sock_Stream) for a UDP-only service (tftp) - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream,    nn::socket::AddrInfoFlag::Ai_CanonName,           "example.com",      "tftp",     false,  nn::socket::AiErrno::EAi_Service,    nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric service port: (tftp: port 69) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericServ,  "example.com",      "69",       true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with an (unknown) port: idonotexist - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName,           "example.com",      "idonotexist",  false,  nn::socket::AiErrno::EAi_Service,    nn::socket::Errno::ESuccess },

         // Domain Name + flag:nn::socket::AddrInfoFlag::Ai_NumericHost
         { "Call GAI with a numeric IPv4 Address: (IANA: F-ROOT DNS 192.5.5.241) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "192.5.5.241",      0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv6 Address: (IANA: F-ROOT DNS 2001:500:2f::f) - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "2001:500:2f::f",   0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 Broadcast Address - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "255.255.255.255",  0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 Loopback Address (127.0.0.1) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "127.0.0.1",        0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv6 Loopback Address (::1) - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "::1",          0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 (nn::socket::InAddr_Any) Address - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "0.0.0.0",      0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 (300.300.300.300) Address - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "300.300.300.300",  0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 (192.168.0.A) Address - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "192.168.0.A",      0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a numeric IPv4 (A.B.C.D) Address - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "A.B.C.D",      0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a (domainame) but (numeric address) expected: Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  "example.com",      0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

         { "Call GAI with a (NULL domainame) and (NULL service) - Expected Result: FAIL",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericHost,  0,          0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

#ifdef COMMENTED_OUT

         { "Call GAI, Pass a (NULL domainname) and (tftp) for port number - Expected Result: FAIL",
           nn::socket::Family::Af_Inet,   nn::socket::Type::Sock_Default,      nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_NumericServ,  0,          "tftp",     false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },
#endif


    /* description                                                          */
    /* family   socktype    flags               domainname      service     result  error       syserrno*/

     // Socket Type
         { "Call GAI with a (socktype == nn::socket::Type::Sock_Stream) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream,    nn::socket::AddrInfoFlag::Ai_CanonName,           "www.example.com",  0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a (socktype == nn::socket::Type::Sock_Dgram) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::AddrInfoFlag::Ai_CanonName,           "www.example.com",  0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

         { "Call GAI with a (socktype == nn::socket::Type::Sock_Raw) - Expected Result: PASS",
           nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Raw,   nn::socket::AddrInfoFlag::Ai_CanonName,           "www.example.com",  0,      true,   nn::socket::AiErrno::EAi_Success,      nn::socket::Errno::ESuccess },

     // NULL arguments
         { "Call GAI with (ALL 0) arguments",
           nn::socket::Family::Af_Unspec, nn::socket::Type::Sock_Default,  nn::socket::AddrInfoFlag::Ai_None,      0,                      0,      false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess }

};


} // Anonymous namespace


TEST(ApiUnit,Win_GetAddrInfo_Argument_Results)
{
    nn::socket::AddrInfo    hints, *resp = NULL;
    nn::socket::AiErrno     out_error;
    unsigned int            idx;
    bool                    rc, isSuccess = false, keepgoing = true;


    WRAP_FAILING_TEST("SIGLO-57714", "[WIN] - When (domain name) is NULL, GAI returns WSAEINVAL.  Expected getservicebyname() functionality" )
    {

        // { "Call GAI, Pass a (NULL domainname) and (tftp) for port number - Expected Result: PASS",
        //      nn::socket::Family::Af_Inet,   nn::socket::Type::Sock_Default,      AI_CANONNAME | AI_NUMERICSERV,  0,          "tftp",     false,  nn::socket::AiErrno::EAi_NoName, nn::socket::Errno::ESuccess },

    }

    WRAP_FAILING_TEST("SIGLO-57714", "[WIN] - Appears to be successfully resolving AF_INET6 requests" )
    {

        // {  "Call GAI with (family == nn::socket::Family::Af_Inet6, flags == nn::socket::AddrInfoFlag::Ai_V4Mapped | nn::socket::AddrInfoFlag::Ai_All) - Expected Result: FAIL",
        //       nn::socket::Family::Af_Inet6,  nn::socket::Type::Sock_Default,  nn::socket::AddrInfoFlag::Ai_CanonName | nn::socket::AddrInfoFlag::Ai_V4Mapped | nn::socket::AddrInfoFlag::Ai_All,    "www.google.com",   0,      false,  nn::socket::AiErrno::EAi_BadFlags,   nn::socket::Errno::ESuccess },

    }

    size_t myResultSize = sizeof( g_myArgumentResults ) / sizeof( GAIArgumentResult  );

    for( idx = 0; idx < myResultSize; idx++ )
    {
        memset( (char *) &hints, 0, sizeof( hints ) );

        hints.ai_family   = g_myArgumentResults[idx].family;
        hints.ai_socktype = g_myArgumentResults[idx].socktype;
        hints.ai_flags    = g_myArgumentResults[idx].flags;

        NN_LOG( "Test Description: %s\n", g_myArgumentResults[idx].desc );

#ifdef COMMENTED_OUT
        NN_LOG( "Inputs: family: %d, socktype: %d, flags: 0x%d, domainname: %s, service: %s, result: %d, error: %s, syserrno: %d\n",
                g_myArgumentResults[idx].family, g_myArgumentResults[idx].socktype, g_myArgumentResults[idx].flags,
                g_myArgumentResults[idx].domainname, g_myArgumentResults[idx].service, (int) g_myArgumentResults[idx].result,
                g_myArgumentResults[idx].error, g_myArgumentResults[idx].syserrno );
#endif

        // Call: GetAddrInfo()
        resp = NULL;

        rc = CallGetAddrInfo( g_myArgumentResults[idx].domainname,
                              g_myArgumentResults[idx].service,
                              &hints, resp, &out_error );

       if ( g_myArgumentResults[idx].result == true )
       {
           if ( rc != true ) {
               NN_LOG( "Return Code Failed, but should have Succeeded.  Error=<%d>\n", out_error );
               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }

           // Free returned response
           nn::socket::FreeAddrInfo( resp );
       }
       else
       {
           if ( rc == true )
           {
               // Free returned response
               nn::socket::FreeAddrInfo( resp );

               ERROR_IF_AND_COUNT( keepgoing == true, "Return Code is Success, but should have Failed!\n" );
           }

           if ( out_error != g_myArgumentResults[idx].error )
           {
               NN_LOG( "GAI Error is not <%d>, but it should be!  Error=<%d> / GAI Error Str=<%s>\n",
                                        g_myArgumentResults[idx].error, out_error,
                                nn::socket::GAIStrError( out_error ) );

               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }

           if ( ( out_error == nn::socket::AiErrno::EAi_System ) && ( g_myArgumentResults[idx].syserrno != nn::socket::GetLastError() ) )
           {
               NN_LOG( "GAI Error is nn::socket::AiErrno::EAi_System, but errno is not <%d>, but it should be!  Errno=<%d>\n",
                                          g_myArgumentResults[idx].syserrno, nn::socket::GetLastError() );

               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }

out:

    return;
}


TEST(ApiUnit,Win_GetAddrInfo_HintValueRanges)
{
    nn::socket::AddrInfo    hints, *resp = NULL;
    nn::socket::AiErrno     out_error;
    int                     idx;
    bool                    rc, isSuccess = false, keepgoing = true;


    // Clear hints
    memset( (char *) &hints, 0, sizeof( hints ) );


    // Address Family Tests
    for( idx = -5; idx < 25; idx++ )
    {

        // Skip these
        if ( idx == static_cast<int>(nn::socket::Family::Af_Unspec) ) continue;
        if ( idx == static_cast<int>(nn::socket::Family::Af_Inet) )   continue;
#ifdef NN_BUILD_CONFIG_OS_WIN32
        if ( idx == static_cast<int>(nn::socket::Family::Af_Inet6) )  continue;
#endif

        hints.ai_family   = static_cast<nn::socket::Family>(idx);


        rc = CallGetAddrInfo( "example.com",
                              "http",
                              &hints, resp, &out_error );
        if ( rc == true )
        {
            NN_LOG( "HINT: (Address Family) = %d Succeeded - but should have FAILEDi!\n", idx );
            ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
        }
        else
        {
            if ( out_error != nn::socket::AiErrno::EAi_Family )
            {
                NN_LOG( "HINT: (Address Family) = %d returned error: %d, but was expecting: nn::socket::AiErrno::EAi_Family\n",
                        idx, out_error );
                ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
        }
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }


    // Clear hints
    memset( (char *) &hints, 0, sizeof( hints ) );


    // Sock Type Tests
    for( idx = -5; idx < 25; idx++ )
    {

        // Skip these
        if ( idx == 0 )         continue;
        if ( idx == static_cast<int>(nn::socket::Type::Sock_Dgram) )    continue;
        if ( idx == static_cast<int>(nn::socket::Type::Sock_Stream) )   continue;
        if ( idx == static_cast<int>(nn::socket::Type::Sock_Raw) )      continue;

        hints.ai_socktype   = static_cast<nn::socket::Type>(idx);

        rc = CallGetAddrInfo( "example.com",
                              "http",
                              &hints, resp, &out_error );
        if ( rc == true )
        {
            NN_LOG( "HINT: (Socket Type) = %d Succeeded - but should have FAILEDi!\n", idx );
            ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
        }
        else
        {
            if ( out_error != nn::socket::AiErrno::EAi_SockType )
            {
                NN_LOG( "HINT: (Socket Type) = %d returned error: %d, but was expecting: EAI_SOCKTYPE\n",
                        idx, out_error );

                ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
        }
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }


out:

    return;
}


typedef struct {

    const char *        domainname;
    const char *        service;
    bool                result;
    nn::socket::AiErrno error;

} GAIDomainNameTestResult;


TEST(ApiUnit,Win_GetAddrInfo_Domain_Lookups)
{
    nn::socket::AddrInfo    hints = { }, *resp = NULL;
    nn::socket::AiErrno     out_error;
    unsigned int            idx;
    bool                    rc, isSuccess = false, keepgoing = true;

    WRAP_FAILING_TEST("SIGLO-57714", "[WIN] - Returns WSANO_RECOVERY - should return EAI_NODATA" )
    {

        // {"!@#$%^&*()`\";:?/\\<>+=[]{}|-_.com",
        //              "http",     false, EAI_NODATA },    // Special (non-escaped) chars

    }

    GAIDomainNameTestResult myResult[] =
    {
        { "google.com",             "http",     true,      nn::socket::AiErrno::EAi_Success },   // Google TCP
        { "google.com",             "tftp",     true,      nn::socket::AiErrno::EAi_Success },   // Google UDP
        { "google",                 "http",     false,     nn::socket::AiErrno::EAi_NoData },    // Google (one label) TCP

#ifdef NN_BUILD_CONFIG_OS_WIN32
        { "google",                 "tftp",     false,     nn::socket::AiErrno::EAi_NoData },    // Google (one label) UDP - Win
#else
        { "google",                 "tftp",     false,     nn::socket::AiErrno::EAi_Fail },      // Google (one label) UDP - NX
#endif

        { "com",                    "http",     false,     nn::socket::AiErrno::EAi_NoData },    // TLD com (one label) UDP
        { ".com",                   "http",     false,     nn::socket::AiErrno::EAi_NoData },    // TLD .com (one label) UDP
        { "amazon.gr",              "http",     true,      nn::socket::AiErrno::EAi_Success },   // Amazon - Greece
        { "amazon.fr",              "http",     true,      nn::socket::AiErrno::EAi_Success },   // Amazon - France
        { "amazon.jp",              "http",     true,      nn::socket::AiErrno::EAi_Success },   // Amazon - Japan
        { "windows.microsoft",      "http",     true,      nn::socket::AiErrno::EAi_Success },   // "Windows" .microsoft

#ifdef NN_BUILD_CONFIG_OS_WIN32
        { "",                       "http",     true,      nn::socket::AiErrno::EAi_Success },   // Blank Label (empty) - Windows: getservicebyname()
#else
        { "",                       "http",     false,     nn::socket::AiErrno::EAi_NoData },    // Blank Label (empty) - NX: No data
#endif

        { "microsoft.net",          "http",     true,      nn::socket::AiErrno::EAi_Success },   // .net lookup
        { "amazon.jobs",            "http",     true,      nn::socket::AiErrno::EAi_Success },   // .jobs
        { "idonotexist.localhost",  "http",     false,     nn::socket::AiErrno::EAi_NoData },    // .localhost (TLD should always be disabled)
        { "xn--bcher-kva.ch",       "http",     true,      nn::socket::AiErrno::EAi_Success },   // IDN German: books, ch = Switzerland
        { "xn--v8jxj3d1dzdz08w.com","http",     true,      nn::socket::AiErrno::EAi_Success },   // IDN Japanese: 名がドメイン.com (Name Domain)
        { "a.b.c.d.e.f",            "http",     false,     nn::socket::AiErrno::EAi_NoData },    // 6 labels
        { "a.b.c.d.e.f.g.h.i.j.k.l" ,
                                    "http",     false,     nn::socket::AiErrno::EAi_NoData },    // 12 labels
        { "a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x",
                                    "http",     false,     nn::socket::AiErrno::EAi_NoData },    // 24 labels
        { "%AB.%AB.%AB.com",        "http",     false,     nn::socket::AiErrno::EAi_NoData },    // Precent sign in multiple labels

{"a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // Many labels - to 253 chars

{"a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o.p.q.r.s.t.u.v.w.x.y.z.",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // Many labels - too far

{"google.........................................com",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // Google with (too many) labels

{ "....",       "http",     false, nn::socket::AiErrno::EAi_NoData },    // Domain name of just periods

{ "216.58.193.78",  "http",     true, nn::socket::AiErrno::EAi_Success },  // Already Rsolved domain name

#ifndef NN_BUILD_CONFIG_OS_WIN32

{"!@#$%^&*()`\";:?/\\<>+=[]{}|-_.com",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // Special (non-escaped) chars

#endif


{ "012345678901234567890123456789012345678901234567890123456789012.comcomcom",     "http",     false, nn::socket::AiErrno::EAi_NoData },  // 63 char label

{ "012345678901234567890123456789012345678901234567890123456789012.012345678901234567890123456789012345678901234567890123456789012.comcommcom",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // 2x - 63 char label

{ "012345678901234567890123456789012345678901234567890123456789012.012345678901234567890123456789012345678901234567890123456789012.012345678901234567890123456789012345678901234567890123456789012.comcomcom",
                "http",     false, nn::socket::AiErrno::EAi_NoData },    // 3x - 63 char label

{ "012345678901234567890123456789012345678901234567890123456789012.012345678901234567890123456789012345678901234567890123456789012.012345678901234567890123456789012345678901234567890123456789012.01234567890123456789012345678901234567890123456789012345678901",
                "http",       false, nn::socket::AiErrno::EAi_NoData },      // 3x - 63 char label, 1x - 62 char label


{ "012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890",
                "http",       false, nn::socket::AiErrno::EAi_NoData },      // 256 char label - 1 byte too large

{ "                                                                                                                                                                                                                                                               ",
                "http",       false, nn::socket::AiErrno::EAi_NoData },      // 255 space characters

{ "012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234567890012345678901234567890123456789012345678901234",
                "http",       false, nn::socket::AiErrno::EAi_NoData }       // 300 char label - 45 bytes too large

    };

    size_t myResultSize = sizeof( myResult ) / sizeof( GAIDomainNameTestResult );


    hints.ai_family = nn::socket::Family::Af_Inet;
    hints.ai_flags = nn::socket::AddrInfoFlag::Ai_CanonName;

    for( idx = 0; idx < myResultSize; idx++ )
    {
       NN_LOG( "Testing Hostname [%s], Service=[%s]\n",
               myResult[idx].domainname, myResult[idx].service );


       // Call: GetAddrInfo()
       resp = NULL;

       rc = CallGetAddrInfo( myResult[idx].domainname,
                             myResult[idx].service,
                             &hints, resp, &out_error );

       if ( myResult[idx].result == true )
       {
           // Free returned response
           nn::socket::FreeAddrInfo( resp );

           if ( rc != true ) {
               NN_LOG( "Return Code Failed, but should have Succeeded.  Error=<%d>\n", out_error );
               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }
       }
       else
       {
           if ( rc == true )
           {
               NN_LOG( "Return Code is Success, but should have Failed!\n" );
               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }

           if ( out_error != myResult[idx].error )
           {
               NN_LOG( "GAI Error is not <%d>, but it should be!  GAI Error=<%d> / GAI Error Str=<%s>\n",
                                          myResult[idx].error, out_error,
                      nn::socket::GAIStrError( out_error ) );

               ERROR_IF_AND_COUNT( keepgoing == true, "Stopping Test" );
           }
        }

        ERROR_IF_AND_COUNT( keepgoing == false, "No Op" );
    }

out:

    return;
}


TEST(ApiUnit,Win_GetAddrInfo_Binary_Domainame)
{
    nn::socket::AddrInfo    hints, *resp = NULL;
    nn::socket::AiErrno     error = nn::socket::AiErrno::EAi_Success;
    int                     idx, idx2;
    bool                    isSuccess = false;
    unsigned char           domainname[256], hexdomain[525];
    uint8_t                 value = 1;

    // Build a binary blob..
    memset( domainname, 0, sizeof( domainname ) );

    for( idx = 0; idx < 255; idx++ )
    {
        if ( ( idx == 63 )  ||          // Set Intermediate labels
             ( idx == 127 ) ||
             ( idx == 192 )  )
        {
            domainname[idx] = '.';
            continue;
        }

        if ( idx == 231 )               // Append .com
        {
            domainname[idx++] = '.';
            domainname[idx++] = 'c';
            domainname[idx++] = 'o';
            domainname[idx++] = 'm';

            break;
        }

        while( isprint( value ) )
        {
            value++;
        }

        domainname[idx] = value;

        value++;                        // Next Value
        if ( value == 0 ) value++;      // Avoid NULL
    }

    // Display domainname in Hex
    idx2 = 0;

    for( idx = 0; idx < 255; idx++ )
    {
        if ( domainname[idx] == 0 ) break;

        sprintf( (char *) &hexdomain[idx2], "%02x", domainname[idx] );
        idx2 = idx2 + 2;
    }

    NN_LOG( "Binary Domain Name: 0x%s", hexdomain );

    // Set hints
    memset( (char *) &hints, 0, sizeof( hints ) );

    hints.ai_family   = nn::socket::Family::Af_Inet;
    hints.ai_socktype = nn::socket::Type::Sock_Default;
    hints.ai_flags    = nn::socket::AddrInfoFlag::Ai_CanonName;

    // NULL Response pointer
    error = nn::socket::GetAddrInfo( (const char *) domainname, "http", &hints, &resp );
    ERROR_IF_AND_COUNT( error == nn::socket::AiErrno::EAi_Success, "GAI succeeded - but should have failed!" );
    if ( error != nn::socket::AiErrno::EAi_NoData )
    {
        NN_LOG( "GAI Binary domainname, expected error == nn::socket::AiErrno::EAi_NoData, but got return error %d\n", error );
        ERROR_IF_AND_COUNT( error != nn::socket::AiErrno::EAi_NoData, "Stop testing!" );
    }

out:

    return;
}



TEST(ApiUnit,Win_GetAddrInfo_NULL_Response_Pointer)
{
    nn::socket::AddrInfo    hints;
    nn::socket::AiErrno     error = nn::socket::AiErrno::EAi_Success;
    bool                    isSuccess = false;

    // Set hints
    memset( (char *) &hints, 0, sizeof( hints ) );

    hints.ai_family   = nn::socket::Family::Af_Inet;
    hints.ai_socktype = nn::socket::Type::Sock_Default;
    hints.ai_flags    = nn::socket::AddrInfoFlag::Ai_CanonName;

    // NULL Response pointer
    error = nn::socket::GetAddrInfo( "example.com", "http", &hints, NULL );
    ERROR_IF_AND_COUNT( error == nn::socket::AiErrno::EAi_Success, "GAI succeeded - but should have failed!" );

    if ( error != nn::socket::AiErrno::EAi_System )
    {
        NN_LOG( "GAI Error Code: nn::socket::AiErrno::EAi_System was expected.  Actually got error code %d\n",  error );
        ERROR_IF_AND_COUNT( error != nn::socket::AiErrno::EAi_NoData, "Stop testing!" );
    }
    else
    {
        if ( nn::socket::GetLastError() != nn::socket::Errno::EInval )
        {
            NN_LOG( "GAI Error Code: nn::socket::AiErrno::EAi_System received - but expected errno = nn::socket::Errno::EInval.  Actually got errno %d\n",
                                                                          nn::socket::GetLastError() );
            ERROR_IF_AND_COUNT( error != nn::socket::AiErrno::EAi_NoData, "Stop testing!" );
        }
    }

out:

    return;
}


TEST(ApiUnit,Win_GetAddrInfo_Teardown)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:
    PRINT_TEST_COUNTS;

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}


}} // namespace NATF API
