﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Fcntl_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Fcntl_Various_Tests)
{
    int                 tcpSock = -1, udpSock = -1, rc = -1;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;
    int                 idx, opts = -1;
    bool                isSuccess = true;
    char                buf1[1024];
    nn::socket::TimeVal myTime;

    ////////////////////
    ////  nn::socket::Errno::EBadf
    ////////////////////

    NN_LOG( "Testing descriptor: " );

    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );
        if ( idx == g_echoServer->serverUDP || idx == g_echoServer->serverTCP )
        {
            continue;
        }

        rc = nn::socket::Fcntl( idx, nn::socket::FcntlCommand::F_GetFl );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nFcntl() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno=<%d>\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nFcntl() succeeded but fds: %d. Expected to fail.\n" );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf Failed\n" );
    isSuccess = true;


    ////////////////////
    ////  nn::socket::Errno::EOpNotSupp
    ////////////////////

    NN_LOG( "Testing nn::socket::Errno::EOpNotSupp" );

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    for ( idx = 0; idx <= 16; idx++ )
    {
        NN_LOG( "%d, ", idx );
        if ( idx == static_cast<int>(nn::socket::FcntlCommand::F_GetFl) || idx == static_cast<int>(nn::socket::FcntlCommand::F_SetFl) )
        {
            NN_LOG( "\nSkipped fds: %d\n", idx );
            continue;
        }

        rc = nn::socket::Fcntl( tcpSock, static_cast<nn::socket::FcntlCommand>(idx) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EOpNotSupp )
            {
                NN_LOG( "\nFcntl() bad fds: %d should have failed with nn::socket::Errno::EOpNotSupp. Errno=<%d>\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nFcntl() succeded but fds: %d. Expected to fail.\n" );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EOpNotSupp Failed\n" );
    isSuccess = true;

    nn::socket::Close( tcpSock );
    tcpSock = -1;


    ////////////////////
    //// TCP - nn::socket::FcntlCommand::F_GetFl - Blocking
    ////////////////////

    NN_LOG( "[TCP]: Testing nn::socket::FcntlCommand::F_GetFl - Blocking\n" );

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = opts & ( static_cast<int>(~nn::socket::FcntlFlag::O_NonBlock) );
    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( ( opts & static_cast<int>(nn::socket::FcntlFlag::O_NonBlock) ) != 0, "Fcntl() Failed, Flag nn::socket::FcntlFlag::O_NonBlock is set! Flag:0x%x\n", opts );

    nn::socket::Close( tcpSock );
    tcpSock = -1;


    ////////////////////
    //// UDP - nn::socket::FcntlCommand::F_GetFl - Blocking
    ////////////////////

    NN_LOG( "[UDP]: Testing nn::socket::FcntlCommand::F_GetFl - Blocking\n" );

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = opts & ( static_cast<int>(~nn::socket::FcntlFlag::O_NonBlock) );
    rc = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( ( opts & static_cast<int>(nn::socket::FcntlFlag::O_NonBlock) ) != 0, "Fcntl() Failed, Flag nn::socket::FcntlFlag::O_NonBlock is set! Flag:0x%x\n", opts );

    nn::socket::Close( udpSock );
    tcpSock = -1;


    ////////////////////
    ////  TCP - nn::socket::FcntlCommand::F_GetFl - Non-Blocking
    ////////////////////

    NN_LOG( "[TCP] - Testing nn::socket::FcntlCommand::F_GetFl - Non-Blocking\n" );

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    opts |= static_cast<int>(nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Non-Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( ( opts & static_cast<int>(nn::socket::FcntlFlag::O_NonBlock) ) == 0, "Fcntl() Failed, Flag nn::socket::FcntlFlag::O_NonBlock not set!\n" );

    nn::socket::Close( tcpSock );
    tcpSock = -1;


    ////////////////////
    ////  UDP - nn::socket::FcntlCommand::F_GetFl - Non-Blocking
    ////////////////////

    NN_LOG( "[UDP] - Testing nn::socket::FcntlCommand::F_GetFl - Non-Blocking\n" );

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( udpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    opts |= static_cast<int>(nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Non-Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    opts = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_GetFl );
    ERROR_IF_AND_COUNT( opts < 0, "Fcntl() nn::socket::FcntlCommand::F_GetFl failed. Expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT( (opts & static_cast<int>(nn::socket::FcntlFlag::O_NonBlock)) == 0, "Fcntl() Failed, Flag nn::socket::FcntlFlag::O_NonBlock not set!\n" );

    nn::socket::Close( udpSock );
    udpSock = -1;


    ////////////////////
    ////  TCP - nn::socket::FcntlCommand::F_SetFl - Blocking
    ////////////////////

    NN_LOG( "[TCP] - Testing nn::socket::FcntlCommand::F_SetFl - Blocking\n" );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    opts = opts & ( static_cast<int>(~nn::socket::FcntlFlag::O_NonBlock) );
    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;

    // Testing if Recv() is a blocking call
    rc = nn::socket::SetSockOpt( tcpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_RcvTimeo - Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc > 0, "Recv succeeded for nn::socket::Option::So_RcvTimeo, but was supposed to FAIL!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
             "Recv() failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", myError );


    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!  Unblocked after [5] second timeout time\n" );
    NN_LOG( "=============\n" );

    nn::socket::Close( tcpSock );
    tcpSock = -1;


    ////////////////////
    ////  UDP - nn::socket::FcntlCommand::F_SetFl - Blocking
    ////////////////////

    NN_LOG( "[UDP] - Testing nn::socket::FcntlCommand::F_SetFl - Blocking\n" );

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    opts = opts & ( static_cast<int>(~nn::socket::FcntlFlag::O_NonBlock) );
    rc = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;

    // Testing if Recv() is a blocking call
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_RcvTimeo - Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Recv( udpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc > 0, "Recv succeeded for nn::socket::Option::So_RcvTimeo, but was supposed to FAIL!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
             "Recv() failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", myError );


    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!  Unblocked after [5] second timeout time\n" );
    NN_LOG( "=============\n" );

    nn::socket::Close( udpSock );
    udpSock = -1;


    ////////////////////
    ////  TCP - nn::socket::FcntlCommand::F_SetFl - Non-Blocking
    ////////////////////

    NN_LOG( "[TCP] - Testing nn::socket::FcntlCommand::F_SetFl - Non-Blocking\n" );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    opts = opts | static_cast<int>(nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Non-Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    // Testing if Recv() is a non blocking call
    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc > -1, "Recv() Succeeded but expected to Fail! Size: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
             "Recv() Expected Errno nn::socket::Errno::EAgain or nn::socket::Errno::EWouldBlock - Errno=<%d>\n", myError );

    nn::socket::Close( tcpSock );
    tcpSock = -1;


    ////////////////////
    ////  UDP - nn::socket::FcntlCommand::F_SetFl - Non-Blocking
    ////////////////////

    NN_LOG( "[UDP] - Testing nn::socket::FcntlCommand::F_SetFl - Non-Blocking\n" );

    udpSock = NATF::API::COMMON::MakeUDPConnection(  "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF( udpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    opts = opts | static_cast<int>(nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( udpSock, nn::socket::FcntlCommand::F_SetFl, opts );
    ERROR_IF_AND_COUNT( rc < 0, "Fcntl() Failed to set Non-Blocking. Errno=<%d>\n", nn::socket::GetLastError() );

    // Testing if Recv() is a non blocking call
    rc = nn::socket::Recv( udpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc > -1, "Recv() Succeeded but expected to Fail! Size: %d\n", rc );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
             "Recv() Expected Errno nn::socket::Errno::EAgain or nn::socket::Errno::EWouldBlock - Errno=<%d>\n", myError );

    nn::socket::Close( udpSock );
    udpSock = -1;


out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }


}   // NOLINT(impl/function_size)


TEST(ApiUnit,Fcntl_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Fcntl_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
