﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include <nnt/nntest.h>

#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;

    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF( !NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    return( true );

out:

    return( false );
}

static bool
TeardownTesting()
{
     bool    isSuccess = true;


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}


////////////////
// B E G I N   T E S T I N G
//////////////////

TEST(ApiUnit,Win_Errno_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Errno_GetLastErrno)
{
    int         retcd;

    // Make a system call to fnctl() that will fail
    retcd = nn::socket::Fcntl( -1, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    if ( 0 == retcd )
    {
        NN_LOG("Error: Fcntl returned (0) -- but requested socket descriptor is (-1)\n" );
        ADD_FAILURE();
        goto out;
    }

    NN_LOG( "Fcntl(): Sucessfullly (FAILED) like it was supposed to...\n" );

    if ( nn::socket::GetLastError() != nn::socket::Errno::EBadf )
    {
        NN_LOG("Error: nn::socket::GetLastError (IS NOT) nn::socket::Errno::EBadf, it is <%ld>\n", nn::socket::GetLastError() );
        ADD_FAILURE();
        goto out;
    }

    NN_LOG( " nn::socket::GetLastError() result is: nn::socket::Errno::EBadf.  This is ccorrect!\n" );

out:

    ; // Nothing to do
}


TEST(ApiUnit,Win_Errno_SetLastErrno)
{
    int                 retcd;
    nn::socket::Errno   error = nn::socket::Errno::ESuccess;
    bool                isSuccess = true;

    // I can set ERRNO to (ZERO == 0)
    NN_LOG("Forcing ERRNO to (ZERO == 0)\n" );
    nn::socket::SetLastError(nn::socket::Errno::ESuccess);      // void

    // Get internal API value of Errno
    error =  nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( nn::socket::Errno::ESuccess != error, "Error: nn::socket::SetLastError() was set to (nn::socket::Errno::ESuccess == 0), but actually returned (%ld)\n", error );

    NN_LOG("ERRNO is (ZERO == 0)\n" );

    // Make a system call to fnctl() that will fail - this populates (ERRNO) with nn::socket::Errno::EBadf
    NN_LOG("Issuing (BAD) Fcntl() function call to populate (ERRNO == nn::socket::Errno::EBadf)\n" );
    retcd = nn::socket::Fcntl( -1, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF_AND_COUNT( retcd == 0, "Got zero retcd: %d, but expected a non-zero result!\n", retcd );

    // I can set ERRNO to (ZERO == 0)
    NN_LOG("Forcing ERRNO to (ZERO == 0)\n" );
    nn::socket::SetLastError(nn::socket::Errno::ESuccess);      // void

    // Get internal API value of Errno
    error =  nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( nn::socket::Errno::ESuccess != error, "Error: nn::socket::SetLastError() was set to (nn::socket::Errno::ESuccess == 0), but actually returned (%ld)\n", error );

    NN_LOG("ERRNO is (ZERO == 0)\n" );

out:

    ;  // Nothing to do

}


TEST(ApiUnit,Win_Errno_Remap_EFAULT)
{
    nn::socket::Errno   error = nn::socket::Errno::ESuccess;
    bool                isSuccess = true;

    // Set nn::socket::Errno::EFault
    nn::socket::SetLastError( nn::socket::Errno::EFault );

    // Get Last Errno should remap to nn::socket::Errno::EInval
    error = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( error != nn::socket::Errno::EInval,"nn::socket::GetLastError() expected nn::socket::Errno::EInval but actually got: %d\n", error );

    NN_LOG( "[nn::socket::Errno::EFault] is correctly remapped to [nn::socket::Errno::EInval]\n" );

out:

    ;  // Do nothing
}



////////////////
// End  T E S T I N G
//////////////////

TEST(ApiUnit,Win_Errno_TearDown)
{
    TeardownTesting();
}


} }  // End namespace NATF API
