﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/resolver/resolver_Client.h>
#endif

#include "Unit/testNet_EchoServer.h"

namespace NATF {
namespace API {

/////////
// Static const
//

const char * EchoServer::kInterfaceIPv4Addr = "127.0.0.1";

///////////////////
// Default Constructor
///////////////////

EchoServer::EchoServer()
{
     int  idx;

     echoServerUDPPort   = kEchoServerUDPPort;
     echoServerTCPPort   = kEchoServerTCPPort;
     echoServerCore      = kEchoServerCore;
     falseEchoMode       = false;
     recvPtrLen          = kEchoServerMaxReceiveSize;
     recvPtr             = NULL;

     echoServerShouldRun = false;
     echoServerRunning   = false;
     serverUDP           = -1;
     serverTCP           = -1;
     acceptedTCP         = -1;

    ////////////////////////
    // Initalize Socket Array
    ////////////////////////

    for( idx = 0; idx < kSocketArrLen; idx++ )
    {
        socketArr[idx] = -1;
    }

}

///////////////////
// (Optional) Constructor
///////////////////

EchoServer::EchoServer( uint16_t    udpPort,
                        uint16_t    tcpPort,
                        nn::Bit64   core,
                        size_t      recvSize )
{
     int  idx;

     echoServerUDPPort   = udpPort;
     echoServerTCPPort   = tcpPort;
     echoServerCore      = core;
     falseEchoMode       = false;
     recvPtrLen          = recvSize;
     recvPtr             = NULL;

     echoServerShouldRun = false;
     echoServerRunning   = false;
     serverUDP           = -1;
     serverTCP           = -1;
     acceptedTCP         = -1;

    ////////////////////////
    // Initalize Socket Array
    ////////////////////////

    for( idx = 0; idx < kSocketArrLen; idx++ )
    {
        socketArr[idx] = -1;
    }

}



///////////////
// Destructor
///////////////

EchoServer::~EchoServer()
{
    if ( echoServerShouldRun == true )
    {
        NN_LOG( "(Destructor): Echo Server sill running - stopping it\n" );
        Stop();
    }

    NN_LOG( "(Destructor): Echo Server (IS) Stopped...\n" );
}


#ifdef COMMENTED_OUT

void
EchoServer::PrintHex( unsigned char *    buff,
                      size_t &           buffLen,
                      bool *             isReq )
{
    int       idx, count = 0;   // Counts where we are in the buffer
    char      hexbuff[1024];

    if ( *isReq == true )
    {
        NN_LOG( "[EchoServer] [Request: - Begin]\n" );
    }
    else
    {
        NN_LOG( "[EchoServer] [Response: - Begin]\n" );
    }

    memset( hexbuff, 0, sizeof( hexbuff ) );

    // Dump formatted HEX
    count = 0;
    for( idx = 0; idx < (int) buffLen; idx++ )
    {
        sprintf( &hexbuff[count], " 0x%02x,", buff[idx] );
        count = count + 6;
        if ( count >= 72 )
        {
            hexbuff[count++] = '\n';
            hexbuff[count]   = 0;
            NN_LOG( hexbuff );
            memset( hexbuff, 0, sizeof( hexbuff ) );
            count = 0;
        }
    }

    if (count > 0 )
    {
        hexbuff[count++] = '\n';
        hexbuff[count]   = 0;
        NN_LOG( hexbuff );
        memset( hexbuff, 0, sizeof( hexbuff ) );
        count = 0;
    }

    if ( *isReq == true )
    {
        NN_LOG( "[Request: - End]\n" );
    }
    else
    {
        NN_LOG( "[Response: - End]\n" );
    }
}

#endif

void
EchoServer::Start( void *    inArg )
{
    nn::socket::TimeVal    mytime;
    nn::socket::FdSet      readSet;

    int                    idx, bestFDS, rc, optval;
    int                    bytesSent, bytesRead, bytesLeft;
    bool                   isSuccess = true;
    ssize_t                rc1, sendLength;

    /////////////////////////
    // E C H O   S E R V E R    S T A R T I N G
    /////////////////////////

    if ( echoServerRunning == true )
    {
        NN_LOG( "Echo Server is already running!\n" );
        return;
    }
    else
    {
        echoServerRunning = true;
        NN_LOG( "==> Echo Server Starting up!\n" );
        NN_LOG( "==> Thread Starting on CPU: %lld\n", echoServerCore );
    }

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue,
                              echoServerCore );

    /////////////////////////
    // SERVER:  TCP
    /////////////////////////

    // Server TCP: Socket
    serverTCP = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( serverTCP < 0, "[EchoServer] Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server TCP: Address
    memset( &tcpAddr, 0, sizeof( tcpAddr ) );

    tcpAddr.sin_family  = nn::socket::Family::Af_Inet;
    tcpAddr.sin_port    = nn::socket::InetHtons( echoServerTCPPort );

    // Server TCP: Loopback
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, kInterfaceIPv4Addr, &tcpAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "[EchoServer] InetPton() Failed but Success was expected!" );

    // Server TCP: SetSockOpt()
    optval = 1;
    rc = nn::socket::SetSockOpt( serverTCP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "[EchoServer] SetSockOpt() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server TCP: Bind
    rc = nn::socket::Bind( serverTCP, reinterpret_cast<nn::socket::SockAddr *>( &tcpAddr ), sizeof( tcpAddr ) );
    ERROR_IF( rc < 0, "[EchoServer] Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server: Listen
    rc = nn::socket::Listen( serverTCP, kSocketArrLen );
    ERROR_IF( rc < 0, "[EchoServer] Listen() failed! errno=<%d>\n\n", nn::socket::GetLastError() );


    /////////////////////////
    // SERVER:  UDP
    /////////////////////////

    // Server UDP: Socket
    serverUDP = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( serverUDP < 0, "[EchoServer] Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server UDP: Address
    memset( &udpAddr, 0, sizeof( udpAddr ) );
    udpAddr.sin_family      = nn::socket::Family::Af_Inet;
    udpAddr.sin_port        = nn::socket::InetHtons( echoServerUDPPort );

    // Server UDP: Loopback
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, kInterfaceIPv4Addr, &udpAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "[EchoServer] InetPton() Failed but Success was expected!" );

    // Server UDP: SetSockOpt()
    optval = 1;
    rc = nn::socket::SetSockOpt( serverUDP, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "[EchoServer] SetSockOpt() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server UDP: Bind
    rc = nn::socket::Bind( serverUDP, reinterpret_cast<nn::socket::SockAddr *>( &udpAddr ), sizeof( udpAddr ) );
    ERROR_IF( rc < 0, "[EchoServer] Bind() failed! errno=<%d>\n\n", nn::socket::GetLastError() );

    /////////////////////////
    // Allocate Recieve Buffer
    /////////////////////////

    recvPtr = (unsigned char *) malloc( recvPtrLen );
    if ( recvPtr == NULL )
    {
        NN_LOG( "[EchoServer] 'malloc' failed creating a buffer of size: %d\n", recvPtrLen );
        goto out;
    }

    ////////////////////////
    // Initalize Socket Array
    ////////////////////////

    for( idx = 0; idx < kSocketArrLen; idx++ )
    {
        socketArr[idx] = -1;
    }

    /////////////////////////
    // Main Loop
    /////////////////////////

    echoServerShouldRun = true;
    while( echoServerShouldRun == true )
    {
        nn::socket::FdSetZero( &readSet );

        nn::socket::FdSetSet( serverTCP, &readSet );

        mytime.tv_sec  = 1;
        mytime.tv_usec = 0;

        // Only TCP for now!
        bestFDS = serverTCP;

        // If NOT in falseEchoMode..
        if ( falseEchoMode == false )
        {
            // Determine highest open FD
            if ( serverUDP > serverTCP )
            {
                bestFDS = serverUDP;
            }

            // Tricky - don't read UDP when we are not supposed to
            nn::socket::FdSetSet( serverUDP, &readSet );

            for( idx = 0; idx < kSocketArrLen; idx++ )
            {
                if ( socketArr[idx] > -1 )
                {
                    nn::socket::FdSetSet( socketArr[idx], &readSet );

                    if ( socketArr[idx] > bestFDS )
                    {
                        bestFDS = socketArr[idx];
                    }
                }
            }
        }

        // Select
        rc = nn::socket::Select( bestFDS + 1, &readSet, nullptr, nullptr, &mytime);
        // NN_LOG( "Select rc: %d, Bestfds: %d\n", rc, bestFDS );

        if ( rc < 0 )
        {
            NN_LOG( "[EchoServer] Select Failed - Errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }

        if ( rc == 0 )
        {
            // NN_LOG( "Select Timeout - Continuing!\n" );
            continue;    // Select timed out
        }

        // is UDP Ready to read?
        if ( nn::socket::FdSetIsSet( serverUDP, &readSet ) )
        {
            // NN_LOG( "Processing: [UDP Echo Request]\n" );

            //  Initialize Recv addr
            memset( &recvAddr, 0, sizeof( recvAddr ) );

            //  Request (From) Client
            recvAddrSize = sizeof( recvAddr );
            rc1 = nn::socket::RecvFrom( serverUDP, recvPtr, recvPtrLen, nn::socket::MsgFlag::Msg_None,
                                        (nn::socket::SockAddr *) &recvAddr, &recvAddrSize );

            //NN_LOG( "[EchoServer] socket: %d, UDP Request received from: [%s:%d] family %d\n",
            //         serverUDP, nn::socket::InetNtoa( recvAddr.sin_addr ), nn::socket::InetNtohs( recvAddr.sin_port ), recvAddr.sin_family );

            //NN_LOG( "[EchoServer] [UDP]: Echo %lld bytes\n", rc1 );
            //NN_LOG( "[EchoServer] UDP: RecvFrom rc:%lld\n", rc1 );

            if ( rc1 < 0 )
            {
                NN_LOG( "[EchoServer] UDP: RecvFrom(): Failed - errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            // Response (From) this Server
            recvAddrSize = sizeof( recvAddr );
            sendLength   =  rc1;
            rc1 = nn::socket::SendTo( serverUDP, recvPtr, sendLength, nn::socket::MsgFlag::Msg_None,
                                      (nn::socket::SockAddr *) &recvAddr, recvAddrSize );

            // NN_LOG( "UDP: SendTo rc = %lld\n", rc1 );

            if ( rc1 < 0 )
            {
                NN_LOG( "[EchoServer] UDP: SendTo(): Failed - errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            // Fall Thru
        }

        // Is the TCP Listen Socket Ready to read?
        if ( nn::socket::FdSetIsSet( serverTCP, &readSet ) )
        {
            // NN_LOG( "Processing: [TCP Echo Request]\n" );

            // Accept Client socket
            recvAddrSize = sizeof( recvAddr );
            acceptedTCP = nn::socket::Accept( serverTCP, (nn::socket::SockAddr *) &recvAddr, &recvAddrSize);
            if ( acceptedTCP < 0 )
            {
                NN_LOG( "[EchoServer] serverTCP: Accept failed!  Errno: %d\n", nn::socket::GetLastError() );
                goto out;
            }

            NN_LOG( "[EchoServer] socket: %d, TCP Request received from: [%s:%d]\n",
                                acceptedTCP,
                                nn::socket::InetNtoa( recvAddr.sin_addr ),
                                nn::socket::InetNtohs( recvAddr.sin_port ) );

            // Search for a free slot
            for ( idx = 0; idx < kSocketArrLen; idx++ )
            {
                if ( socketArr[idx] == -1 )
                {
                    socketArr[idx] = acceptedTCP;
                    break;
                }
            }

            // If no slots..
            if ( idx == kSocketArrLen )
            {
                NN_LOG( "[EchoServer] Out of Availale Slots - Can't accept new TCP socket fds: %d\n", acceptedTCP );
                nn::socket::Close( acceptedTCP );
                continue;
            }

            // Set Non-Blocking Read() on FDS
            rc = nn::socket::Fcntl( acceptedTCP, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
            ERROR_IF( rc < 0, "socket: %d, Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n",
                                                 acceptedTCP, nn::socket::GetLastError() );

#if 0 // November 3rd remarks: We were getting nn::socket::Errno::EBadf here, possibly have a problem
           // Disable (David) Nagels algorimthum
           optval = 1;
           rc = nn::socket::SetSockOpt( acceptedTCP, nn::socket::Protocol::IpProto_Tcp, nn::socket::Option::Tcp_NoDelay, (char *) &optval, sizeof( int) );
           ERROR_IF( rc < 0, "Failed to turn off Nagle algorithum!" );
#endif
            // Fall Thru
        }

        // Are any of the TCP Connected sockets ready-to-read?
        for( idx = 0; idx < kSocketArrLen; idx++ )
        {
            if (  ( socketArr[idx] > -1 )                  &&
                  ( nn::socket::FdSetIsSet( socketArr[idx], &readSet ) )  )
            {
                bytesRead  = 0;
                bytesLeft  = (int) recvPtrLen;
                while( bytesLeft > 0 )
                {
                    rc = nn::socket::Read( socketArr[idx], &recvPtr[bytesRead], bytesLeft );

                    if ( rc == 0 )
                    {
                        NN_LOG( "[EchoServer] socket: %d, End-Of-File - Closing\n", socketArr[idx] );
                        nn::socket::Close( socketArr[idx] );
                        socketArr[idx] = -1;
                        break;
                    }

                    if ( rc < 0 )
                    {
                        if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                        {
                              break;
                        }

                        NN_LOG( "[EchoServer] Server socket: %d, nn::socket::Read() Failed - Errno: %d\n",
                        socketArr[idx], nn::socket::GetLastError() );

                        nn::socket::Close( socketArr[idx] );
                        socketArr[idx] = -1;
                        break;
                     }

                     bytesRead = bytesRead + rc;
                     bytesLeft = bytesLeft - rc;

                     break;   // Something to write!
                 }

                 // Echo Message
                 // NN_LOG( "[TCP]: Echo %d bytes\n", bytesRead );
                 bytesSent = 0;
                 bytesLeft = bytesRead;
                 while( bytesLeft > 0 )
                 {
                     rc = nn::socket::Write( socketArr[idx], &recvPtr[bytesSent], bytesLeft );

                     if ( rc == 0 )
                     {
                         NN_LOG( "[EchoServer] socket: %d, Server End-Of-File - Closing\n", socketArr[idx] );
                         nn::socket::Close( socketArr[idx] );
                         socketArr[idx] = -1;
                         break;
                     }

                     if ( rc < 0 )
                     {
                         if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                         {
                             nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                             continue;   // Jump to Top
                         }

                         NN_LOG( "[EchoServer] Server socket: %d nn::socket::Write() Failed - Errno: %d\n",
                                        socketArr[idx], nn::socket::GetLastError() );
                         nn::socket::Close( socketArr[idx] );
                         socketArr[idx] = -1;
                         break;
                     }

                     bytesSent = bytesSent + rc;
                     bytesLeft = bytesLeft - rc;
                 }
            }
        }

        // Bottom of Loop
    }

out:

    NN_LOG( "==> Echo Server -- told to Exit\n" );

    // If Recv Pointer is allocated
    if ( recvPtr != NULL )
    {
        free( recvPtr );
        recvPtr = NULL;
    }

    // Close TCP Socket
    if ( serverTCP > -1 )
    {
        nn::socket::Close( serverTCP );
        serverTCP = -1;
    }

    // Close Connected TCP sockets
    for( idx = 0; idx < kSocketArrLen; idx++ )
    {
        nn::socket::Close( socketArr[idx] );
        socketArr[idx] = -1;
    }

    // Close UDP Socket
    if ( serverUDP > -1 )
    {
        nn::socket::Close( serverUDP );
        serverUDP = -1;
    }

    // Echo Server - Stopped Running
    echoServerRunning   = false;
    echoServerShouldRun = false;

    return;

}  // NOLINT(impl/function_size)


int
EchoServer::Stop()
{
    if ( echoServerShouldRun == false )
    {
        NN_LOG( "[Echo Server] is not running - nothing to do\n" );
        return( 0 );
    }

    NN_LOG( "[Echo Server] is running - stopping it\n" );
    echoServerShouldRun = false;

    while( echoServerRunning == true )
    {
        NN_LOG( "Waiting for Echo Server to shutdown...\n" );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG( "[Echo Server] - STOPPED!\n" );

    return( 0 );
}



}}; // namespace:NAFT / API
