﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

static bool                 g_startThread = false;
static int                  ServerPort = 8053;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the Echo Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server\n" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server (%p)\n", g_echoServer );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    NN_LOG( "Waiting Echo Thread\n" );
    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    NN_LOG( "Destroying Echo Thread\n" );
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}


////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Connect_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Connect_Various)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, remoteAddr = { 0 }, newAddr = { 0 };
    bool                    isSuccess = true;
    int                     rc = -1, idx = 0, testsock = -1, ttl = 0, optval = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;

    NN_LOG( "In\n\n" );

    // Set up Client (Local) Address
    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons( 9000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client local Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Set up Server (Remote) Address
    memset(&remoteAddr, 0, sizeof(remoteAddr));
    remoteAddr.sin_port        = nn::socket::InetHtons( 8053 );
    remoteAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client Server Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &remoteAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////////////////////////
    // Errno: (NX) nn::socket::Errno::EBadf
    // Errno: (WIN) nn::socket::Errno::EBadf
    ////////////////////////////////


    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  || ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::Connect( idx, (nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
        if ( rc > -1 )
        {
            NN_LOG( "\nConnect() succeeded but should have failed processing descriptor: %d\n", idx );
            goto out;
        }
        else
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nConnect() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }
    }

    NN_LOG( "\n" );


    ////////////////////////////////
    // Errno: nn::socket::Errno::EAddrNotAvail - TCP
    ////////////////////////////////

    NN_LOG( "TCP - Test Connect to illegal TCP port == 0, Errno = nn::socket::Errno::EAddrNotAvail\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "A: Socket failed!" );

    remoteAddr.sin_port = 0;   // Illegal TCP Port (ANY)
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );

    ERROR_IF_AND_COUNT( rc == 0, "Successful connect, but expected FAIL!" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EAddrNotAvail, "Expected errno to be nn::socket::Errno::EAddrNotAvail - but it is not!" );

    nn::socket::Close( testsock );
    testsock = -1;


    ////////////////////////////////
    // Errno: nn::socket::Errno::EAddrNotAvail - UDP
    ////////////////////////////////

    NN_LOG( "UDP - Test Connect to (ANY) UDP port == 0, (PASS)\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram,  nn::socket::Protocol::IpProto_Udp);
    ERROR_IF( testsock < 0, "B: Socket failed!" );

    rc = nn::socket::Bind( testsock, (const nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Initial Bind Failed!" );

    remoteAddr.sin_port = 0;   // Illegal UDP Port (ANY)
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    remoteAddr.sin_port = nn::socket::InetHtons( (uint16_t) ServerPort );

    ERROR_IF_AND_COUNT( rc < 0,"Connect failed but was expecting to succeed!  errno: %d\n", nn::socket::GetLastError() );

    nn::socket::Close( testsock );
    testsock = -1;


    ////////////////////////////////
    // Errno: nn::socket::Errno::EAfNoSupport - TCP
    ////////////////////////////////

    NN_LOG( "TCP - Connect to (Unsupported) Address Family: Novel IPX - Errno: nn::socket::Errno::EAfNoSupport\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "Socket failed!" );

    remoteAddr.sin_family = static_cast<nn::socket::Family>(PF_IPX);   // Illegal Address Family
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    remoteAddr.sin_family = nn::socket::Family::Af_Inet;

    ERROR_IF_AND_COUNT( rc == 0, "Successful connect, but expected FAIL!" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EAfNoSupport, "Expected errno to be nn::socket::Errno::EAfNoSupport - but it is not!" );

    nn::socket::Close( testsock );
    testsock = -1;


    ////////////////////////////////
    // Errno: nn::socket::Errno::EAfNoSupport - UDP
    ////////////////////////////////

    NN_LOG( "UDP - Connect to (Unsupported) Address Family: Novel IPX - Errno: nn::socket::Errno::EAfNoSupport\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram,  nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( testsock < 0, "Socket failed!" );

    rc = nn::socket::Bind( testsock, (const nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind Failed!" );

    remoteAddr.sin_family = static_cast<nn::socket::Family>(PF_IPX);
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    remoteAddr.sin_family = nn::socket::Family::Af_Inet;

    ERROR_IF_AND_COUNT( rc == 0, "Successful connect, but expected FAIL!" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EAfNoSupport, "Expected errno to be nn::socket::Errno::EAfNoSupport - but it is not!" );

    nn::socket::Close( testsock );
    testsock = -1;


    ////////////////////////////////
    // Errno: nn::socket::Errno::EIsConn - TCP
    ////////////////////////////////

    NN_LOG( "TCP - Call Connect Twice - Errno: nn::socket::Errno::EIsConn\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "Socket failed!" );

    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc == -1, "Connect() Failed but Success was expected!  errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc == 0, "Connect() Succeeded but Failure was expected!" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EIsConn, "Expected errno to be nn::socket::Errno::EIsConn - but it is not!" );

    nn::socket::Close( testsock );
    testsock = -1;


    ////////////////////////////////
    // UDP - Reset (Source) of sending datagrams
    //
    // UDP uses Connect() to define the (peer) for to recieve datagrams from (aka the sender).  Connect is used to
    // specify the senders IP Address and Port.  After this call only datagrams from the specified sender are
    // received.  Later the senders IP Address and Port can be cleared by calling Connect() and passing a nullptr
    // to the network address argument.
    ////////////////////////////////

    NN_LOG( "UDP - Call Connect Mutiple Times\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    ERROR_IF( testsock < 0, "Socket failed!" );

    // Set initial sender of datagrams (TO ME)
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc == -1, "Connect() Failed but Success was expected!" );

    // Turn off (inital) sender of datagrams (TO ME)
    rc = nn::socket::Connect( testsock, static_cast<nn::socket::SockAddr*>(nullptr), 0);
    ERROR_IF_AND_COUNT( rc > -1, "Connect() Succeed but was expected to fail!" );

    // Set initial sender of datagrams (TO ME)
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc == -1, "Connect() Failed but Success was expected!" );

    // Set (same) sender of datagrams (TO ME) - aka (NO CHANGE)
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc == -1, "Connect() Failed but Success was expected!" );

    nn::socket::Close( testsock );


    ////////////////////////////////
    // Errno: (WIN)  nn::socket::Errno::ENetUnreach / nn::socket::Errno::EAgain - TCP
    ////////////////////////////////

    NN_LOG( "TCP - Connect to an unreachabled network - Errno: ( nn::socket::Errno::ENetUnreach | nn::socket::Errno::ETimedOut )\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    NN_LOG( "Connect can block up to 2 minutes!  Be patient..\n" );
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "172.16.1.1", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );

    ERROR_IF_AND_COUNT( rc == 0, "Connect() Succeeded but Fail was expected!" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::ENetUnreach &&
                        myError != nn::socket::Errno::EAgain,
                        "Expected errno to be nn::socket::Errno::ENetUnreach/nn::socket::Errno::EAgain - but it is not!" );

    nn::socket::Close( testsock );

    ////////////////////////////////
    // Errno: (NX) nn::socket::Errno::ENetUnreach - UDP
    // Win - Successful Connect
    ////////////////////////////////

    NN_LOG( "UDP - Connect to an unreachabled network - (NOT AN ERROR)\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "172.16.1.1", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    ERROR_IF_AND_COUNT( rc < 0, "Connect() Failed, but success was expected!  Errno: %d", nn::socket::GetLastError() );

    nn::socket::Close( testsock );


    ////////////////////////////////
    // nn::socket::Errno::EConnRefused
    ////////////////////////////////

    NN_LOG( "TCP - Connection Refused -- errno: nn::socket::Errno::EConnRefused\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "127.0.0.1", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_port = nn::socket::InetHtons( 6666 );  // IRCD
    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, (const nn::socket::SockAddr *) &remoteAddr, sizeof( remoteAddr ) );
    remoteAddr.sin_port = nn::socket::InetHtons( (uint16_t) ServerPort);

    ERROR_IF_AND_COUNT( rc == 0, "Connect() Succeeded but Fail was expected!" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnRefused, "Expected errno to be nn::socket::Errno::EConnRefused - but it is not!" );

    nn::socket::Close( testsock );

    ////////////////////////////////
    // nn::socket::Errno::ENetUnreach - Because TTL is too low
    ////////////////////////////////

WRAP_FAILING_TEST( "SIGLO-58427", "Socket API: [WIN] TCP too low IP TTL results in (zero == 0) errno" )
{

    NN_LOG( "TCP - Unreachable because TTL is too low (TTL = 1, Dest=8.8.8.8) - nn::socket::Errno::EHostUnreach\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    // Set IP TTL == 0 (Make sure packet times out before transmit)
    ttl = 1;
    rc = nn::socket::SetSockOpt( testsock, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Ttl, &ttl, sizeof( ttl ) );
    ERROR_IF( rc < 0, "SetSockOpt failed setting nn::socket::Option::Ip_Ttl!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "8.8.8.8", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );

    ERROR_IF_AND_COUNT( rc == 0, "Connect Succeed but Failure was expected!" );

    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EHostUnreach &&
                        nn::socket::GetLastError() != nn::socket::Errno::ETimedOut,
                        "Expected errno to be nn::socket::Errno::EHostUnreach/nn::socket::Errno::ETimedOut - but it is actually errno: %d", nn::socket::GetLastError() );

    nn::socket::Close( testsock );
}


    ////////////////////////////////
    // Errno (WIN) nn::socket::Errno::EAddrNotAvail: TCP - Connect to IPv4 Broadcast Address
    ////////////////////////////////

WRAP_FAILING_TEST( "SIGLO-58357", "Socket API: [WIN] Connect TCP to Broadcast Address failed - but no errno is returned\n" )
{

    NN_LOG( "TCP - Connect to IPv4 Broadcast Address (255.255.255.255) - Errno: nn::socket::Errno::EConnRefused\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "255.255.255.255", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );

    ERROR_IF_AND_COUNT( rc == 0, "Connect Succeed but Failure was expected!" );

    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnRefused &&
                        nn::socket::GetLastError() != nn::socket::Errno::ETimedOut,
                        "Expected errno to be nn::socket::Errno::ENetUnreach/nn::socket::Errno::ETimedOut - but it is actually errno: %d!\n", nn::socket::GetLastError() );

    nn::socket::Close( testsock );

}

    ////////////////////////////////
    // Errno (NX): nn::socket::Errno::ENetUnreach: UDP - Connect to IPv4 Broadcast Address  (No Socket Option Set)
    // WIN: (BUG) -- Successful Connect
    ////////////////////////////////

    NN_LOG( "UDP - Connect to IPv4 Broadcast Address (255.255.255.255) - (NO SOCK OPT)\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "255.255.255.255", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );

    ERROR_IF_AND_COUNT( rc != 0, "Connect failed but success was expected, errno: %d!", nn::socket::GetLastError() );

    nn::socket::Close( testsock );


    ////////////////////////////////
    // nn::socket::Option::So_Broadcast:  UDP - Connect to IPv4 Broadcast Address   (Socket Option: SET)
    //
    // (This is correct because you can't receive from a UDP Broadcast)
    ////////////////////////////////

    NN_LOG( "UDP - Connect to IPv4 Broadcast Address (255.255.255.255) - (SOCK OPT SET!)\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    ERROR_IF( testsock < 0, "F: Socket failed!" );

    memset( (char *) &newAddr, 0, sizeof( nn::socket::SockAddrIn ) );

    rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "255.255.255.255", &newAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Set Socket Option: nn::socket::Option::So_Broadcast
    optval = 1;
    rc = nn::socket::SetSockOpt( testsock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "SetSockOpt failed setting nn::socket::Option::So_Broadcast!" );

    remoteAddr.sin_addr.S_addr = newAddr.sin_addr.S_addr;
    rc = nn::socket::Connect( testsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );

    ERROR_IF_AND_COUNT( rc < 0, "Connect Failed but Success was expected!  Errno: %d\n", nn::socket::GetLastError() );

    nn::socket::Close( testsock );


    ////////////////////////////////
    // UDP - Switch (PEER) from who I receive datagrams from
    ////////////////////////////////

WRAP_FAILING_TEST( "SIGLO-58462", "Socket API: [WIN] Recv failed with errno 0 for UDP" )
{

    NN_LOG( "UDP - Switch the 'Peer Server' from which the client recieves datagrams from!\n" );

    {
        nn::socket::SockAddrIn    server1Addr = { 0 }, server2Addr = { 0 }, cliAddr = { 0 };
        int            server_sock1 = -1, server_sock2 = -1, client_sock = -1;
        char           recvbuff[ 128 ];
        bool           dataFound = true;
        ssize_t        rc1 = 0;

        // Client
        memset(&cliAddr, 0, sizeof(cliAddr));
        cliAddr.sin_port        = nn::socket::InetHtons( 6666 );
        cliAddr.sin_family      = nn::socket::Family::Af_Inet;

        rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "127.0.0.1", &cliAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        client_sock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ERROR_IF( client_sock < 0, "Socket failed!" );

        rc = nn::socket::Bind( client_sock, (const nn::socket::SockAddr *) &cliAddr, sizeof( cliAddr ) );
        ERROR_IF( rc < 0, "Bind Failed!" );

        rc = nn::socket::Fcntl( client_sock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        // Server 1:
        memset(&server1Addr, 0, sizeof(server1Addr));
        server1Addr.sin_port        = nn::socket::InetHtons( 6667 );
        server1Addr.sin_family      = nn::socket::Family::Af_Inet;

        rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "127.0.0.1", &server1Addr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        server_sock1 = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ERROR_IF( server_sock1 < 0, "Socket failed!" );

        rc = nn::socket::Bind( server_sock1, (const nn::socket::SockAddr *) &server1Addr, sizeof( server1Addr ) );
        ERROR_IF( rc < 0, "Bind Failed!" );

        rc = nn::socket::Connect( server_sock1, (const nn::socket::SockAddr *) &cliAddr, sizeof( cliAddr ) );
        ERROR_IF_AND_COUNT( rc < 0, "Connect Failed!" );

        // Server 2:
        memset(&server2Addr, 0, sizeof(server2Addr));
        server2Addr.sin_port        = nn::socket::InetHtons( 6668 );
        server2Addr.sin_family      = nn::socket::Family::Af_Inet;

        rc = nn::socket::InetPton( nn::socket::Family::Pf_Inet, "127.0.0.1", &server2Addr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        server_sock2 = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ERROR_IF( server_sock2 < 0, "Socket failed!" );

        rc = nn::socket::Bind( server_sock2, (const nn::socket::SockAddr *) &server2Addr, sizeof( server2Addr ) );
        ERROR_IF( rc < 0, "Bind Failed!" );

        rc = nn::socket::Connect( server_sock2, (const nn::socket::SockAddr *) &cliAddr, sizeof( cliAddr ) );
        ERROR_IF_AND_COUNT( rc < 0, "Connect Failed!" );

        ///////////////////////////////
        // Server 1: Transmits successfully to Client; Server 2: Datagrams silently dropped
        ///////////////////////////////

        // Client wants to recieve from Server 1
        rc = nn::socket::Connect( client_sock, (const nn::socket::SockAddr *) &server1Addr, sizeof( server1Addr ) );
        ERROR_IF_AND_COUNT( rc < 0, "Connect Failed!" );

        // Server 1:  Send IDENT
        rc1 = nn::socket::Send( server_sock1, "1", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF( rc1 < 0, "Server 1: Send Failed!" );

        // Server 2:  Send IDENT
        rc1 = nn::socket::Send( server_sock2, "2", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF( rc1 < 0, "Server 2 Send Failed!" );

        // Allow for sent data to propagate, so that first call to Recv() will give the data
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        // Client: While Recieving...
        dataFound = false;
        memset( recvbuff, 0, sizeof( recvbuff ) );

        for( ;; )
        {
             rc1 = nn::socket::Recv( client_sock, recvbuff, sizeof( recvbuff ), nn::socket::MsgFlag::Msg_DontWait );
             if ( rc1 < 0 )
             {

                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     break;
                 }

                 ERROR_IF_AND_COUNT( rc1 < 0, "Recv Failed errno: %d\n", nn::socket::GetLastError() );
             }

             if ( ( rc1 == 1 ) && ( recvbuff[0] == '1' ) )
             {
                 dataFound = true;
                 continue;
             }

             NN_LOG( "Client received data from somewhere (other than) Server 1!  rc1=%d, data=%.*s",
                        (int) rc1, (int) rc1, recvbuff );
             goto out;
        }

        ERROR_IF_AND_COUNT( dataFound == false, "Never received data from Server 1 !" );

        ///////////////////////////////
        // Server 2: Transmits successfully to Client; Server 1: Datagrams silently dropped
        ///////////////////////////////

        // Client wants to recieve from Server 2
        rc = nn::socket::Connect( client_sock, (const nn::socket::SockAddr *) &server2Addr, sizeof( server2Addr ) );
        ERROR_IF_AND_COUNT( rc < 0, "Connect Failed!" );

        // Server 1:  Send IDENT
        rc1 = nn::socket::Send( server_sock1, "1", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF( rc1 < 0, "Server 1: Send Failed!" );

        // Server 2:  Send IDENT (first attempt) - Always FAILS
        rc1 = nn::socket::Send( server_sock2, "2", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc1 > -1, "Server 2: Send Succeded but should have Failed!" );
        ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnRefused,
                            "Server 2: Errno nn::socket::Errno::EConnRefused expected but was not returned!" );

        // Server 2:  Send IDENT (second attempt) - Success
        rc1 = nn::socket::Send( server_sock2, "2", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc1 < 0, "Server 2: Send Failed!" );

        // Allow for sent data to propagate, so that first call to Recv() will give the data
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        // Client: While Recieving...
        dataFound = false;
        memset( recvbuff, 0, sizeof( recvbuff ) );

        for( ;; )
        {
             rc1 = nn::socket::Recv( client_sock, recvbuff, sizeof( recvbuff ), nn::socket::MsgFlag::Msg_DontWait );
             if ( rc1 < 0 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     break;
                 }

                 ERROR_IF_AND_COUNT( rc1 < 0, "Recv Failed" );
             }

             if ( ( rc1 == 1 ) && ( recvbuff[0] == '2' ) )
             {
                 dataFound = true;
                 continue;
             }

             NN_LOG( "Client received data from somewhere (other than) Server 2!  rc1=%d, data=%.*s",
                        (int) rc1, (int) rc1, recvbuff );
             goto out;
        }

        ERROR_IF_AND_COUNT( dataFound == false, "Never received data from Server 2 !" );

        ///////////////////////////////
        // Work around nullptr bug
        ///////////////////////////////

        // Client
        nn::socket::Close( client_sock );

        client_sock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ERROR_IF( client_sock < 0, "Socket failed!" );

        rc = nn::socket::Bind( client_sock, (const nn::socket::SockAddr *) &cliAddr, sizeof( cliAddr ) );
        ERROR_IF( rc < 0, "Bind Failed!" );

        rc = nn::socket::Fcntl( client_sock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
        ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

        ///////////////////////////////
        // Server 1 / Server 2 (BOTH): Transmit successfully to Client
        ///////////////////////////////

        // Server 1:  Send IDENT (first attempt) - Always FAILS
        rc1 = nn::socket::Send( server_sock1, "1", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc1 > -1, "Server 1: Send Succeded but should have Failed!" );
        ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EConnRefused,
                            "Server 1: Errno nn::socket::Errno::EConnRefused expected but was not returned!" );

        // Server 1:  Send IDENT (second attempt)
        rc1 = nn::socket::Send( server_sock1, "1", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc1 < 0, "Server 1: Send Failed!" );

        // Server 2:  Send IDENT (first attempt)
        rc1 = nn::socket::Send( server_sock2, "2", 1, nn::socket::MsgFlag::Msg_None );
        ERROR_IF_AND_COUNT( rc1 < 0, "Server 2: Send Failed!" );

        // Allow for sent data to propagate
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        // Client: While Recieving...
        idx = 0;
        memset( recvbuff, 0, sizeof( recvbuff ) );

        for( ;; )
        {
             rc1 = nn::socket::Recv( client_sock, recvbuff, sizeof( recvbuff ), nn::socket::MsgFlag::Msg_DontWait );
             if ( rc1 < 0 )
             {
                 if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
                 {
                     break;
                 }

                 ERROR_IF_AND_COUNT( rc1 < 0, "Recv Failed" );
             }

             idx++;
        }

        ERROR_IF( idx != 2, "I expected 2 Servers to reply, but reply count is <%d>", idx );

        // Cleanup
        nn::socket::Close( server_sock1 );
        nn::socket::Close( server_sock2 );
        nn::socket::Close( client_sock );

    }

}  // End Failing Test


    ////////////////////////////////
    // Connect:  Address Family testing
    ////////////////////////////////

    NN_LOG( "UDP - Address Family testing!\n" );

    testsock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( testsock < 0, "Socket failed!" );

    for( int idx2 = 0; idx2 < 256; idx2++ )
    {
        if ( idx2 == static_cast<int>(nn::socket::Family::Af_Inet) || idx2 == static_cast<int>(nn::socket::Family::Pf_Inet) )
        {
            continue;
        }

        remoteAddr.sin_family = static_cast<nn::socket::Family>(idx2);
        rc = nn::socket::Connect( testsock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );
        ERROR_IF( rc > -1, "Unsupported Address Family = <%d> found!\n\n", nn::socket::GetLastError() );
    }

    nn::socket::Close( testsock );

out:

    return;

} // NOLINT(impl/function_size)


static void
Threaded_Connect( void * inArg )
{
    int sockfds = *((int *) inArg );


    while( g_startThread == false )
    {
        NN_LOG( "Waiting to start...\n" );
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG( "Thread Starting... Wait 5 seconds\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG( "Thread Closing Socket..\n" );
    nn::socket::Close( sockfds );

    NN_LOG( "Thread Exiting\n" );

}

TEST(ApiUnit,Win_Connect_End_Point_Closed_While_Connecting)
{
    nn::Result          Result;
    nn::socket::SockAddrIn  cliAddr = { 0 };
    int                 cliSock = -1, listenSock = -1, rc = 0;
    int                 threadId = -1;

    bool                isSuccess = true, rval;


    ////////////
    // Create TCP connection
    //////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Server Listen Socket FD = %d\n", listenSock );

    // Start Testing Thread
    g_startThread = false;
    threadId = g_pThreadedTest->CreateThread( &Threaded_Connect, (void *) &listenSock, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create Client (thread)!\n" );
        goto out;
    }

    // Create a TCP Client
    cliSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( cliSock < 0, "Socket failed!  Errno: %d", errno);

    // Client
    memset(&cliAddr, 0, sizeof(cliAddr));
    cliAddr.sin_port        = nn::socket::InetHtons( 9060 );
    cliAddr.sin_family      = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &cliAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1,"InetPton() failed - errno: %d\n", nn::socket::GetLastError() );

    // Blocking Connect Call
    g_startThread = true;
    rc = nn::socket::Connect( cliSock, (nn::socket::SockAddr *) &cliAddr, sizeof( cliAddr ) );
    ERROR_IF( rc < 0, "Connect failed with errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Connect succeeded!\n" );

out:

    if ( threadId > 0 )
    {
        // Wait for Echo Server thread to die
        g_pThreadedTest->WaitThread( threadId );

        // Destroy Echo Server Thread
        g_pThreadedTest->DestroyThread( threadId  );

        threadId = -1;
    }


    if ( listenSock > 0 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

    if ( cliSock > 0 )
    {
    nn::socket::Close( cliSock );
    cliSock = -1;
    }
}


void
Win_Connect_Too_Long( void * inArg )
{
    int rc = 0;
    int sockfd = *((int *) inArg );

    NN_LOG( "Thread Begin\n" );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(15));
    rc = nn::socket::Shutdown( sockfd, nn::socket::ShutdownMethod::Shut_RdWr );
    if ( rc == 0 )
    {
        NN_LOG( "Socket Shutdown - Succeeded with RC = 0\n" );
    }
    else
    {
        NN_LOG( "Socket Shutdown - Failed with RC = %d\n", rc );
    }

    NN_LOG( "Thread End!\n" );
}


TEST(ApiUnit,Win_Connect_Too_Long)
{
    nn::socket::SockAddrIn   targetAddr = { 0 };
    int                  cliSock = -1, threadId = -1l;
    bool                 isSuccess = true;
    int                  rc;

    ////////////
    // Make TCP Connect
    ///////////

    cliSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( cliSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port        = nn::socket::InetHtons( 7000 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "1.2.3.4", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Start Thread
    threadId = g_pThreadedTest->CreateThread( &Win_Connect_Too_Long, (void *) &cliSock, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create Client (thread)!\n" );
        goto out;
    }

    // Client Blocking Connect
    NN_LOG( "Connect can block as long as [2 minutes] - be patient...!\n" );
    rc = nn::socket::Connect( cliSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    ERROR_IF( cliSock < 0, "Client Socket failed to Connect!  Errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "Connect unblocked..End Test!\n" );

out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( threadId > -1 )
    {
        g_pThreadedTest->WaitThread(    threadId );
        g_pThreadedTest->DestroyThread( threadId );
        threadId = -1;
    }

}

TEST(ApiUnit,Win_Connect_Teardown)
{
    TeardownTesting();

}

}}  // Namespace: NATF::API
