﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs
#include <cstring>    // strlen

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );

    return( true );

out:

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}

namespace
{
    const int MaxNumSockets = MAX_SOCKETS_PER_CLIENT;
}


////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Accept_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Accept_Various_Tests)
{
    nn::socket::SockAddrIn  inAddr = { 0 }, localAddr = { 0 };
    nn::socket::SockLenT    inAddrLen = sizeof( nn::socket::SockAddr ), shortAddrLen = 1;
    int                     tcpSock = -1, udpSock = -1, servSock = -1, udpServerSock = -1, acceptedSock = -1, rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    bool                    isSuccess = true;
    char                    shortBuffer[1], receiveBuf[1028];
    int                     socketsLeft;

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////
    NN_LOG( "Testing descriptor:" );

    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );

        acceptedSock = nn::socket::Accept( idx, (nn::socket::SockAddr *) &inAddr, &inAddrLen );
        if ( acceptedSock < 0 )
        {
            if ( nn::socket::GetLastError() != nn::socket::Errno::EBadf)
            {
                NN_LOG( "\nAccept() bad fds: %d should have failed with nn::socket::Errno::EBadf.\n", idx );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nAccept() succeeded but fds: %d was not TCP or UDP socket\n", idx );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf failed\n" );
    isSuccess = true;

    ////////////
    // Make TCP Server
    ///////////
    servSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( servSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

     // Server: Fcntl - nn::socket::FcntlFlag::O_NonBlock
    rc = nn::socket::Fcntl( servSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server TCP: Address
    memset( &localAddr, 0, sizeof( localAddr ) );

    localAddr.sin_family  = nn::socket::Family::Af_Inet;
    localAddr.sin_port    = nn::socket::InetHtons( 8051 );

    // Server TCP
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server TCP: Bind
    rc = nn::socket::Bind( servSock, reinterpret_cast<nn::socket::SockAddr *>( &localAddr ), sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server: Listen
    rc = nn::socket::Listen( servSock, 5 );
    ERROR_IF( rc < 0, "Listen() failed! Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Test Nullptr for address of the remote host
    ///////////
    NN_LOG( "Testing: NULL ptr\n" );

    // Make TCP Client Connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( servSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT(acceptedSock < 0, "Accept() failed but should have succeeded. Errno:%d \n", nn::socket::GetLastError());

    nn::socket::Close( tcpSock );
    tcpSock = -1;

    nn::socket::Close( acceptedSock );
    acceptedSock = -1;


    ////////////
    // Test network address 1 byte buffer - truncates data
    ///////////
    NN_LOG( "Testing: 1 byte buffer\n" );

    // Make TCP Client Connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( servSock, (nn::socket::SockAddr *) &shortBuffer, &shortAddrLen );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Accept() failed but should have succeded. \n");

    nn::socket::Close( tcpSock );
    tcpSock = -1;

    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    ////////////
    // Test nn::socket::Errno::EInval
    ///////////

    NN_LOG( "Testing nn::socket::Errno::EInval\n" );

    // Make UDP Server
    udpServerSock = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServerSock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Server UDP: Address
    memset( &localAddr, 0, sizeof( localAddr ) );

    localAddr.sin_family  = nn::socket::Family::Af_Inet;
    localAddr.sin_port    = nn::socket::InetHtons( 8051 );

    // Server UDP
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server UDP: Bind
    rc = nn::socket::Bind( udpServerSock, reinterpret_cast<nn::socket::SockAddr *>( &localAddr ), sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed! Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Make UDP Client Connection
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8051 );
    ERROR_IF( udpSock < -1, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // Set UDP server to nonblocking socket - UDP socket does not call Listen()
    rc = nn::socket::Fcntl( udpServerSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( udpServerSock, (nn::socket::SockAddr *) &inAddr, &inAddrLen );
    ERROR_IF( acceptedSock != -1, "Accept() succeeded but should of failed.\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "Accept() should fail with nn::socket::Errno::EInval. Errno=<%d>\n", myError );

    nn::socket::Close( udpSock );
    udpSock = -1;

    nn::socket::Close( udpServerSock );
    udpServerSock = -1;


    ////////////
    // Test nn::socket::Errno::EWouldBlock/nn::socket::Errno::EAgain
    ///////////

    NN_LOG( "Testing - nn::socket::Errno::EWouldBlock/nn::socket::Errno::EAgain\n" );

    // Set TCP server to nonblocking socket - TCP socket calls Listen()
    rc = nn::socket::Fcntl( servSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( servSock, (nn::socket::SockAddr *) &inAddr, &inAddrLen );
    ERROR_IF( acceptedSock != -1, "Accept() succeeded but should of failed.\n" );
    myError = nn::socket::GetLastError();

    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EWouldBlock && myError != nn::socket::Errno::EAgain,
                        "Accept() should fail with nn::socket::Errno::EWouldBlock/nn::socket::Errno::EAgain. Errno=<%d>\n",
                        myError
                        );


    ////////////
    // Test Accepting closed client connection - Accept passes, Recv() returns 0
    ///////////
    NN_LOG( "Testing Accept() closed client connection\n" );

    // Make Client connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // Close Client connection
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    nn::os::SleepThread( nn::TimeSpan::FromSeconds(2) );

    // Server: Accept client connection
    acceptedSock = nn::socket::Accept( servSock, (nn::socket::SockAddr *) &localAddr, &inAddrLen );

    // Client is in a CLOSE_WAIT state and still can be accepted
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Accept() expected to succeed, Failed instead with Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Recv( acceptedSock, receiveBuf, sizeof(receiveBuf), nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 0, "Recv() expected to return 0. Actual:%d. Client connection is closed.", rc );

    nn::socket::Close( acceptedSock );
    acceptedSock = -1;


    ////////////
    // Exceed NX socket limit
    ///////////
    NN_LOG( "Testing NX socket limit\n" );
    socketsLeft = MaxNumSockets;

    isSuccess = false;
    NN_LOG( "Client fds: ");
    do
    {
        // Make Client connection
        tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
        myError = nn::socket::GetLastError();
        if ( tcpSock < 0 )
        {
            if ( socketsLeft > 0 )
            {
                NN_LOG("\nFailed calling MakeTCPConnection when socketsLeft:%d!  Errno: %d\n", socketsLeft, nn::socket::GetLastError() );
            }
            else if( myError == nn::socket::Errno::EBadf || myError == nn::socket::Errno::EMFile ) // Currently returns nn::socket::Errno::EMFile
            {
                isSuccess= true;
            }
            else
            {
                NN_LOG("\n MakeTCPConnection() expected to fail with nn::socket::Errno::EBadf/nn::socket::Errno::EMFile. Errno: %d\n", myError );
            }
        }
        else
        {
            socketsLeft = MaxNumSockets - (tcpSock + 1);
        }

        NN_LOG( "Client fds: %d, ", tcpSock);

        // Server: Accept client connection
        acceptedSock = nn::socket::Accept( servSock, (nn::socket::SockAddr *) &localAddr, &inAddrLen );
        myError = nn::socket::GetLastError();
        if ( acceptedSock < 0 )
        {
            if ( socketsLeft > 0 )
            {
                NN_LOG("\nAccept() expected to succeed, Failed with Errno: %d\n", myError );
            }
            else if( myError == nn::socket::Errno::EBadf || myError == nn::socket::Errno::EMFile ) // Currently returns nn::socket::Errno::EMFile
            {
                isSuccess = true;
            }
            else
            {
                NN_LOG("\nAccept() expected to fail with nn::socket::Errno::EBadf/nn::socket::Errno::EMFile. Errno: %d\n", myError );
            }
        }
        else
        {
            socketsLeft = MaxNumSockets - (acceptedSock + 1);
        }
        NN_LOG( "Accepted fds: %d, ", acceptedSock);
        NN_LOG( "socketsLeft: %d \n", socketsLeft );

    } while ( socketsLeft > 0 && tcpSock != -1 && acceptedSock != -1 );
    NN_LOG( "\n" );

    ERROR_IF_AND_COUNT( !isSuccess, "Exceeding socket limit test failed!\n" );
    isSuccess = true;

    // Close all sockets
    for ( idx = 0; idx < 32; idx++ )
    {
        nn::socket::Close( idx );
    }


out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( servSock > -1 )
    {
        nn::socket::Close( servSock );
        servSock = -1;
    }

    if ( udpServerSock > -1 )
    {
        nn::socket::Close( udpServerSock );
        udpServerSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

}// NOLINT(impl/function_size)


TEST(ApiUnit,Accept_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Accept_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
