﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_UnitCommonSocket.h"

namespace NATF {
namespace API {

const char* SocketContainerActorStrings[] = {
    "EMBRYONIC",
    "CLIENT",
    "SERVER"
};

const char* SocketContainerActorAsString(const SocketContainerActor role)
{
    return SocketContainerActorStrings[role];
};

SocketContainer::SocketContainer() :
    m_AddressFamilyFlags(ADDRESS_FAMILY_FLAGS_DEFAULT),
    m_SocketTypeFlags(SOCK_TYPE_FLAGS_DEFAULT),
    m_ProtocolFlags(PROTOCOL_FLAGS_DEFAULT),
    m_Socket(-1),
    m_Backlog(0),
    m_SocketControlFlags(static_cast<SocketControlFlags>(0)),
    m_ContainerTypeFlags(SOCKET_CONTAINER_TYPE_EMBRYONIC),
    m_CurrentState(SOCKET_CONTAINER_TYPE_EMBRYONIC),
    m_Role(SOCKET_CONTAINER_ACTOR_EMBRYONIC)
{
    //UNIT_TEST_TRACE("");
}

bool SocketContainer::operator==(const SocketContainer& o) const
{
    if (m_AddressFamilyFlags    == o.m_AddressFamilyFlags
        && m_SocketTypeFlags    == o.m_SocketTypeFlags
        && m_ProtocolFlags      == o.m_ProtocolFlags
        && m_Socket             == o.m_Socket
        && m_Backlog            == o.m_Backlog
        && m_SocketControlFlags == o.m_SocketControlFlags
        //&& m_CurrentState       == o.m_CurrentState
        && m_ContainerTypeFlags == o.m_ContainerTypeFlags
        && m_Role               == o.m_Role)
    {
        return true;
    };

    return false;
}

static const uint32_t g_SOCKET_CONTAINER_HEADER_MAGIC  = 0x00000001;
static const uint32_t g_SOCKET_CONTAINER_TRAILER_MAGIC = 0xCAFEBABE;

size_t SocketContainer::SizeOf()
{
    const unsigned int NUMBER_PARAMETERS = 9;
    return sizeof(g_SOCKET_CONTAINER_HEADER_MAGIC) +
        sizeof(g_SOCKET_CONTAINER_TRAILER_MAGIC) +
        sizeof(uint32_t) * NUMBER_PARAMETERS;
}

int SocketContainer::ToNetworkBuffer(uint8_t *& pBufferInOut, size_t bufferSize) const
{
    int rc = -1;
    uint32_t networkValue = 0;

    if (bufferSize < SizeOf())
    {
        goto bail;
    }

    networkValue = nn::socket::InetHtonl(g_SOCKET_CONTAINER_HEADER_MAGIC);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_AddressFamilyFlags);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_SocketTypeFlags);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_ProtocolFlags);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_Socket);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_Backlog);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_SocketControlFlags);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_ContainerTypeFlags);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_CurrentState);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(m_Role);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    networkValue = nn::socket::InetHtonl(g_SOCKET_CONTAINER_TRAILER_MAGIC);
    memcpy(pBufferInOut, &networkValue, sizeof(uint32_t));
    pBufferInOut += sizeof(uint32_t);

    rc = 0;

bail:
    return rc;
}

int SocketContainer::FromNetworkBuffer(SocketContainer& socketContainerOut, uint8_t * pBufferIn, size_t bufferSize)
{
    int rc = -1;
    uint32_t hostValue = 0;
    uint32_t headerMagic = 0, trailerMagic = 0;
    if (bufferSize < SizeOf())
    {
        goto bail;
    }

    headerMagic = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    pBufferIn += sizeof(uint32_t);
    if (headerMagic != g_SOCKET_CONTAINER_HEADER_MAGIC)
    {
        UNIT_TEST_TRACE("incompatible version of socket container header magic; expected %d, got %d",
                        g_SOCKET_CONTAINER_HEADER_MAGIC, headerMagic);
        goto bail;
    };

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_AddressFamilyFlags = static_cast<AddressFamilyFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_SocketTypeFlags = static_cast<SocketTypeFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_ProtocolFlags = static_cast<ProtocolFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_Socket = static_cast<uint32_t>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_Backlog = static_cast<uint32_t>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_SocketControlFlags = static_cast<SocketControlFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_ContainerTypeFlags = static_cast<SocketContainerTypeFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_CurrentState = static_cast<SocketContainerTypeFlags>(hostValue);
    pBufferIn += sizeof(uint32_t);

    hostValue = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    socketContainerOut.m_Role = static_cast<SocketContainerActor>(hostValue);
    pBufferIn += sizeof(uint32_t);

    trailerMagic = nn::socket::InetHtonl(*reinterpret_cast<uint32_t*>(pBufferIn));
    pBufferIn += sizeof(uint32_t);
    if (trailerMagic != g_SOCKET_CONTAINER_TRAILER_MAGIC)
    {
        UNIT_TEST_TRACE("incompatible version of socket container trailer magic; expected %d, got %d",
                        g_SOCKET_CONTAINER_TRAILER_MAGIC, trailerMagic);
        goto bail;
    };

    rc = 0;

bail:
    return rc;

}

}} // namespace NATF::API
