﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkLog.h>
#include <nn/socket.h>

#include <cstring>
#include <algorithm> //std::max

#include "testNet_ApiCommon.h"
#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_SelectUnitData.h"
#include "Complex/testNet_SelectUnitClient.h"

namespace NATF {
namespace API {

/**
 * @brief select unit client
 */
SelectUnitClient::SelectUnitClient(SelectUnitData* pSelectUnitData) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(SocketContainerActorAsString(SOCKET_CONTAINER_ACTOR_CLIENT), 500),  // TODO: use actual value
    SelectUnitNetworkCommon(pSelectUnitData, SOCKET_CONTAINER_ACTOR_CLIENT, this)
{
    //UNIT_TEST_TRACE("")
    nn::util::SNPrintf(m_SelectUnitClientName, sizeof(m_SelectUnitClientName), "s:%d", m_pSelectUnitData->GetPort());
};

SelectUnitClient::~SelectUnitClient()
{
    //UNIT_TEST_TRACE("")
};

void SelectUnitClient::Run()
{
    RunLoop();
}

void SelectUnitClient::InitializeSockets(SocketContainer& container)
{
    //UNIT_TEST_TRACE("")
    ssize_t rc = -1;
    nn::socket::SockAddrIn sin;
    uint16_t port = m_pSelectUnitData->GetPort();
    UNIT_TEST_TRACE("client using port: %d", port);

    memset(&sin, (int)0, sizeof(nn::socket::SockAddrIn));

    uint32_t loopback = nn::socket::InetHtonl(nn::socket::InAddr_Loopback);
    memcpy(&sin.sin_addr, &loopback, sizeof(loopback));
    sin.sin_port = nn::socket::InetHtons(port);

    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;

    if (SOCK_TYPE_FLAGS_SOCK_STREAM == container.m_SocketTypeFlags)
    {
        sin.sin_family = nn::socket::Family::Af_Inet;

        rc = nn::socket::Connect(container.m_Socket, (nn::socket::SockAddr *) &sin, sizeof(sin));
        errorNumber = nn::socket::GetLastError();
        if (rc != -1 && container.m_SocketControlFlags == SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING)
        {
            nn::socket::Fcntl(container.m_Socket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
        }
        else if (SVALIDATE_FAIL(m_pValidator, (rc == -1),
                                "socket %d not connected, error: %d", container.m_Socket, errorNumber))
        {
            goto bail;
        }
        else
        {
            UNIT_TEST_TRACE("socket %d connected", container.m_Socket);
        };

        if (0 != (container.m_ContainerTypeFlags & SOCKET_CONTAINER_TYPE_READ)
            || 0 != (container.m_ContainerTypeFlags & SOCKET_CONTAINER_TYPE_EXCEPTION))
        {
            SendContainer(container, "initial write", nn::socket::MsgFlag::Msg_None);
        }
        else
        {
            goto bail;
        }

    }
    else if (SOCK_TYPE_FLAGS_SOCK_DGRAM == container.m_SocketTypeFlags)
    {
        NN_ASSERT(false);
    }
    else
    {
        NN_ASSERT(false);
    };

bail:
    return;
};

bool SelectUnitClient::EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore)
{
    //UNIT_TEST_TRACE("")
    bool rc = false;
    GetSiftedList();
    for (auto iter = m_SocketContainerList.begin(); iter != m_SocketContainerList.end(); ++iter)
    {
        InitializeSockets(*iter);
    }
    return rc;
}

int SelectUnitClient::OnNewConnectionEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    // not a server
    NN_ASSERT(false);
    container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;
    return 0;
};

int SelectUnitClient::OnEmbryonicReadEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    // not a server
    NN_ASSERT(false);
    container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;
    return 0;
};

int SelectUnitClient::OnReadEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    ssize_t rc = -1;
    SocketContainer junk;
    UNIT_TEST_TRACE("read: event encountered on socket: %d", container.m_Socket);

    rc = ReceiveContainer(junk, container, "read", nn::socket::MsgFlag::Msg_Oob);
    ++eventsHandled;

    if (rc <= 0)
    {
        goto bail;
    };

    container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;

bail:
    return eventsHandled;
};

int SelectUnitClient::OnWriteEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    ssize_t rc = -1;
    SocketContainer junk;

    UNIT_TEST_TRACE("write: event encountered on socket: %d", container.m_Socket);

    rc = SendContainer(container, "delayed write", nn::socket::MsgFlag::Msg_None);
    ++eventsHandled;
    container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;

    if (rc <= 0)
    {
        goto bail;
    };

    //ReceiveContainer(junk, container, "write / receiveContainer", 0);

bail:
    return eventsHandled;
};

int SelectUnitClient::OnExceptionEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    ssize_t rc = -1;
    uint8_t buf[1024] = { '\0' };
    SocketContainer junk;
    UNIT_TEST_TRACE("exception: event encountered on socket: %d", container.m_Socket);

    /* TODO: winsock is off by 1 */
    rc = ReceiveBytes(container, buf, 1, nn::socket::MsgFlag::Msg_Oob);
    //rc = ReceiveContainer(junk, container, "exception", nn::socket::MsgFlag::Msg_Oob);
    ++eventsHandled;

    if (rc <= 0)
    {
        goto bail;
    };
    container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;

bail:
    return eventsHandled;
};

}}; // NATF::API
