﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Complex/testNet_BroadcastDatagram.h"
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm.h>
#include <nn/nn_SdkLog.h>
#include <algorithm>
#include <functional>
#include <random>

namespace NATF {
namespace API {

char g_MessageToSend[256] = { 0 };

uint64_t CreateRandomBroadcastToken() NN_NOEXCEPT
{
    uint64_t tickValue = nn::os::GetSystemTick().GetInt64Value();
    std::mt19937 mt(static_cast<uint64_t>(tickValue));
    return std::uniform_int_distribution<uint64_t>(0, INT64_MAX)(mt);
};

BroadcastDatagramWorkerThread::BroadcastDatagramWorkerThread(SimpleValidator* pSimpleValidator,
                                                             bool isProducer,
                                                             unsigned datagramCount,
                                                             unsigned producerSleepTime,
                                                             uint16_t port,
                                                             bool useOnesCompliment) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(__FUNCTION__, ( producerSleepTime * datagramCount * 1050) ),
    m_pValidator(pSimpleValidator),
    m_IsProducer(isProducer),
    m_DatagramCount(datagramCount),
    m_ProducerSleepTimeInMilliseconds(producerSleepTime),
    m_Port(port),
    m_Sockfd(-1),
    m_UseOnesCompliment(useOnesCompliment)
{
    NN_ASSERT(NULL != m_pValidator);
    memset(&m_Addr, '\0', sizeof(m_Addr));
    m_pValidator->addReference();
};

BroadcastDatagramWorkerThread::~BroadcastDatagramWorkerThread()
{
    m_pValidator->releaseReference();
    if ( -1 != m_Sockfd)
    {
        nn::socket::Close(m_Sockfd);
    };
};

/** @brief run function, waits for testTimeout or signal */
void BroadcastDatagramWorkerThread::Run()
{
    nn::socket::InAddr junk;
    nn::nifm::GetCurrentIpConfigInfo(&m_IpAddress, &m_IpAddressMask, &junk, &junk, &junk);
    m_Addr.sin_family = nn::socket::Family::Af_Inet;
    m_Addr.sin_port = nn::socket::InetHtons(m_Port);

    if ( true == m_IsProducer )
    {
        // see IP_SENDIF for send broadcast on interface addr
        m_Addr.sin_addr.S_addr = m_IpAddress.S_addr |= ~m_IpAddressMask.S_addr;
        NN_SDK_LOG("Producer Address: %s:%d\n",
                   nn::socket::InetNtoa(m_Addr.sin_addr),
                   nn::socket::InetNtohs(m_Addr.sin_port));
        Produce();
    }
    else
    {
        // see IP_RECVIF for recv broadcast on interface addr
        m_Addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        NN_SDK_LOG("Consumer Address: %s:%d\n",
                   nn::socket::InetNtoa(m_Addr.sin_addr),
                   nn::socket::InetNtohs(m_Addr.sin_port));
        Consume();
    };

    return;
};

void BroadcastDatagramWorkerThread::Produce()
{
    int rc = -1;
    int enableFlag = 1;
    size_t lengthOfString = 0;

    if ( -1 == (m_Sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp)))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to create socket; rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;
    }
    else if ( 0 != (rc = nn::socket::SetSockOpt(m_Sockfd,
                                                nn::socket::Level::Sol_Socket,
                                                nn::socket::Option::So_Broadcast,
                                                &enableFlag,
                                                sizeof(enableFlag))))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to SetSockOpt; rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;

    }
    else if (true == m_UseOnesCompliment &&
             0 != (rc = nn::socket::SetSockOpt(m_Sockfd,
                                               nn::socket::Level::Sol_Ip,
                                               nn::socket::Option::Ip_OnesBcast,
                                               &enableFlag,
                                               sizeof(enableFlag))))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to SetSockOpt(nn::socket::Option::Ip_OnesBcast); rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;

    };

    for ( ; 0 != m_DatagramCount; --m_DatagramCount)
    {
        sprintf(g_MessageToSend, "%s-%u-%lu",
                nn::socket::InetNtoa(m_IpAddress),
                m_DatagramCount,
                static_cast<unsigned long>(CreateRandomBroadcastToken()));
        lengthOfString = strlen(g_MessageToSend);

        if ( lengthOfString != (rc = nn::socket::SendTo(m_Sockfd,
                                                        g_MessageToSend,
                                                        lengthOfString,
                                                        nn::socket::MsgFlag::Msg_None,
                                                        reinterpret_cast<const nn::socket::SockAddr *>(&m_Addr),
                                                        sizeof(m_Addr))))
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Unable to SendTo; rc: %d, errno: %s (%d), socket: %d\n",
                           rc, strerror(errno), errno, m_Sockfd);
            goto bail;
        };

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_ProducerSleepTimeInMilliseconds));
    };

    NN_SDK_LOG("sendto sent: %d bytes\n", rc);

bail:
    if ( -1 == rc )
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "A failure occurred; rc=%d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
    };

    if ( -1 != m_Sockfd)
    {
        nn::socket::Close(m_Sockfd);
        m_Sockfd = -1;
    };

    return;
}

void BroadcastDatagramWorkerThread::Consume()
{
    int rc = 0;
    unsigned option = 1;
    char buffer[250];
    nn::socket::SockLenT sinLength = sizeof(nn::socket::SockAddrIn);

    if ( -1 == (m_Sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet,
                                              nn::socket::Type::Sock_Dgram,
                                              nn::socket::Protocol::IpProto_Udp)))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to create socket; rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;
    }
    else if (-1 == (rc = nn::socket::SetSockOpt(m_Sockfd,
                                                nn::socket::Level::Sol_Socket,
                                                nn::socket::Option::So_ReuseAddr,
                                                &option,
                                                sizeof(option))))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to SetSockOpt; rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;
    }
    else if ( -1 == (rc = nn::socket::SetSockOpt(m_Sockfd,
                                                 nn::socket::Level::Sol_Socket,
                                                 nn::socket::Option::So_Broadcast,
                                                 reinterpret_cast<void *>(&option),
                                                 sizeof( option ))))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Unable to SetSockOpt(nn::socket::Option::So_Broadcast); rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;

    }
    else if (-1 == (rc = nn::socket::Bind(m_Sockfd,
                                          reinterpret_cast<nn::socket::SockAddr*>(&m_Addr),
                                          sinLength)))
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "Bind; rc: %d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
        goto bail;
    };
    // add recvto here

    for( ;; )
    {
        nn::socket::FdSet readfds;
        nn::socket::FdSetZero(&readfds);
        nn::socket::FdSetSet(m_Sockfd, &readfds);
        nn::socket::TimeVal tv;
        tv.tv_sec = s_ConsumerTimeoutSeconds;
        tv.tv_usec = 0;

        memset(buffer, 0, sizeof(buffer));
        if ( 0 == (rc = nn::socket::Select(m_Sockfd + 1, &readfds, NULL, NULL, &tv)))
        {
            // this test should timeout because broadcast datagrams
            // ought NOT be received by the Consumer. Therefore, if we time out
            // the test passes but if we get data then what we need to do is
            // check that data against what was sent
            UNIT_TEST_TRACE("Did not receive a broadcast datagram within 5 seconds, test passes");
            goto bail;
        }
        else if ( -1 == rc)
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Select; rc: %d, errno: %s (%d), socket: %d\n",
                           rc, strerror(errno), errno, m_Sockfd);
        }
        else if ( 0 >= (rc = nn::socket::RecvFrom(m_Sockfd,
                                             &buffer,
                                             sizeof(buffer),
                                             nn::socket::MsgFlag::Msg_None,
                                             reinterpret_cast<nn::socket::SockAddr*>(&m_Addr),
                                             &sinLength)))
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "RecvFrom; rc: %d, errno: %s (%d), socket: %d\n",
                           rc, strerror(errno), errno, m_Sockfd);
        }
        else if ( 0 == memcmp(g_MessageToSend, buffer, rc) )
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Received %d bytes (\"%s\") from but IFF_SIMPLEX should not be enabled \n",
                           rc,
                           buffer);
        }
    };

bail:
    if ( -1 == rc )
    {
        SVALIDATE_FAIL(m_pValidator,
                       true,
                       "A failure occurred; rc=%d, errno: %s (%d), socket: %d\n",
                       rc, strerror(errno), errno, m_Sockfd);
    };

    if ( -1 != m_Sockfd)
    {
        nn::socket::Close(m_Sockfd);
        m_Sockfd = -1;
    };

    return;
} //NOLINT(impl/function_size)

void RunBroadcastDatagramUnitTest(unsigned datagramCount,
                                  unsigned producerSleepTimeInMilliseconds,
                                  unsigned port,
                                  bool useOnesCompliment,
                                  unsigned numberOfConsumers)
{
    bool nifmSetupSuccess = false;

    if (false == (nifmSetupSuccess = NATF::API::TestSetup(TestSetupOptions_Nifm | TestSetupOptions_Socket)))
    {
        NN_LOG("\nError: NATF::API::TestSetup(TestSetupOptions_Nifm | TestSetupOptions_Socket) failed.\n");
        goto bail;
    }
    else
    {
        // validator
        AutoReleaseObject<SimpleValidator>
            validator(new SimpleValidator(), false);

        // producer
        AutoReleaseObject<BroadcastDatagramWorkerThread>
            producer(new BroadcastDatagramWorkerThread(&(*validator),
                                                       true,
                                                       datagramCount,
                                                       producerSleepTimeInMilliseconds,
                                                       port,
                                                       useOnesCompliment));
        // consumer array
        AutoReleaseObject<BroadcastDatagramWorkerThread> consumerThreads[numberOfConsumers];

        // create & start the consumer workers
        for (unsigned idx=0; idx<numberOfConsumers; ++idx)
        {
            consumerThreads[idx] = new BroadcastDatagramWorkerThread(&(*validator),
                                                                     false,
                                                                     datagramCount,
                                                                     producerSleepTimeInMilliseconds,
                                                                     port,
                                                                     useOnesCompliment);
            (*consumerThreads[idx]).Start();
        };

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        // start the producer and wait until it is done
        (*producer).Start();
        (*producer).WaitForDone();

        // now wait until the consumer threads are finished
        for (unsigned idx=0; idx < numberOfConsumers; ++idx)
        {
            (*consumerThreads[idx]).WaitForDone();
        };

        // fail
        if ( true == (*validator).DidFail() )
        {
            ADD_FAILURE();
        };
    };

bail:

    if ( true == nifmSetupSuccess && false == NATF::API::TestTeardown() )
    {
        NN_LOG("\nError: NATF::API::TestTeardown() failed. \n");;
    };

    return;
};

TEST(BroadcastDatagram, oneSimpleDatagram)
{
    UNIT_TEST_TRACE("");
    unsigned datagramCount = 1;
    unsigned producerSleepTimeInMilliseconds = 50;
    unsigned port = 5555;
    bool useOnesCompliment = false;
    unsigned numberOfConsumers = 1;

    RunBroadcastDatagramUnitTest(datagramCount,
                                 producerSleepTimeInMilliseconds,
                                 port,
                                 useOnesCompliment,
                                 numberOfConsumers);
};

}}
