﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // for memcpy
#include <cstdio>  // for sscanf

#include "Utils/md5.h"

/* F, G, H and I are basic MD5 functions. */
#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

/* ROTATE_LEFT rotates x left n bits. */
#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

/* FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4.
Rotation is separate from addition to prevent recomputation. */
#define FF(a, b, c, d, x, s, ac) { \
    (a) += F ((b), (c), (d)) + (x) + (UINT4)(ac); \
    (a) = ROTATE_LEFT ((a), (s)); \
    (a) += (b); \
    }
#define GG(a, b, c, d, x, s, ac) { \
    (a) += G ((b), (c), (d)) + (x) + (UINT4)(ac); \
    (a) = ROTATE_LEFT ((a), (s)); \
    (a) += (b); \
    }
#define HH(a, b, c, d, x, s, ac) { \
    (a) += H ((b), (c), (d)) + (x) + (UINT4)(ac); \
    (a) = ROTATE_LEFT ((a), (s)); \
    (a) += (b); \
    }
#define II(a, b, c, d, x, s, ac) { \
    (a) += I ((b), (c), (d)) + (x) + (UINT4)(ac); \
    (a) = ROTATE_LEFT ((a), (s)); \
    (a) += (b); \
    }

namespace
{
    const int S11 = 7;
    const int S12 = 12;
    const int S13 = 17;
    const int S14 = 22;
    const int S21 = 5;
    const int S22 = 9;
    const int S23 = 14;
    const int S24 = 20;
    const int S31 = 4;
    const int S32 = 11;
    const int S33 = 16;
    const int S34 = 23;
    const int S41 = 6;
    const int S42 = 10;
    const int S43 = 15;
    const int S44 = 21;

    const unsigned char PADDING[64] = {
        0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
    };

    /* Encodes input (UINT4) into output (unsigned char). Assumes len is
    a multiple of 4. */
    void Encode(unsigned char *output, UINT4 *input, unsigned int len) NN_NOEXCEPT
    {
        unsigned int i, j;

        for(i = 0, j = 0; j < len; i++, j += 4)
        {
            output[j]     = (unsigned char)( input[i] & 0xff);
            output[j + 1] = (unsigned char)((input[i] >> 8) & 0xff);
            output[j + 2] = (unsigned char)((input[i] >> 16) & 0xff);
            output[j + 3] = (unsigned char)((input[i] >> 24) & 0xff);
        }
    }

    /* Decodes input (unsigned char) into output (UINT4). Assumes len is
    a multiple of 4.*/
    void Decode(UINT4 *output, const unsigned char *input, unsigned int len) NN_NOEXCEPT
    {
        unsigned int i, j;

        for(i = 0, j = 0; j < len; i++, j += 4)
        {
            output[i] = ((UINT4)input[j]) | (((UINT4)input[j + 1]) << 8) |
            (((UINT4)input[j + 2]) << 16) | (((UINT4)input[j + 3]) << 24);
        }
    }
} // Unnamed namespace

MD5Hash::Result::Result() NN_NOEXCEPT
{ memset(m_pHash, 0, sizeof(m_pHash)); }

MD5Hash::Result::Result(unsigned char n1,  unsigned char n2,  unsigned char n3,  unsigned char n4,
                        unsigned char n5,  unsigned char n6,  unsigned char n7,  unsigned char n8,
                        unsigned char n9,  unsigned char n10, unsigned char n11, unsigned char n12,
                        unsigned char n13, unsigned char n14, unsigned char n15, unsigned char n16 ) NN_NOEXCEPT
{
    m_pHash[0]  = n1;  m_pHash[1]  = n2;  m_pHash[2]  = n3;  m_pHash[3]  = n4;
    m_pHash[4]  = n5;  m_pHash[5]  = n6;  m_pHash[6]  = n7;  m_pHash[7]  = n8;
    m_pHash[8]  = n9;  m_pHash[9]  = n10; m_pHash[10] = n11; m_pHash[11] = n12;
    m_pHash[12] = n13; m_pHash[13] = n14; m_pHash[14] = n15; m_pHash[15] = n16;
}

void MD5Hash::Result::Set(unsigned char n1,  unsigned char n2,  unsigned char n3,  unsigned char n4,
                          unsigned char n5,  unsigned char n6,  unsigned char n7,  unsigned char n8,
                          unsigned char n9,  unsigned char n10, unsigned char n11, unsigned char n12,
                          unsigned char n13, unsigned char n14, unsigned char n15, unsigned char n16 ) NN_NOEXCEPT
{
    m_pHash[0]  = n1;  m_pHash[1]  = n2;  m_pHash[2]  = n3;  m_pHash[3]  = n4;
    m_pHash[4]  = n5;  m_pHash[5]  = n6;  m_pHash[6]  = n7;  m_pHash[7]  = n8;
    m_pHash[8]  = n9;  m_pHash[9]  = n10; m_pHash[10] = n11; m_pHash[11] = n12;
    m_pHash[12] = n13; m_pHash[13] = n14; m_pHash[14] = n15; m_pHash[15] = n16;
}


// Expected format: "a1 b2 c3 b4..."
MD5Hash::Result::Result(const char* pHashString) NN_NOEXCEPT
{
    // Convert string hash to buffer values
    for(unsigned i = 0; i < MD5Hash::HASH_SIZE; ++i)
    {
        unsigned nValue = 0;
        sscanf(&pHashString[i * 3], "%x", &nValue);
        m_pHash[i] = (unsigned char)nValue;
    }
}

void MD5Hash::Result::operator=(const Result& rhs)
{
    memcpy(m_pHash, rhs.m_pHash, sizeof(m_pHash));
}

bool MD5Hash::Result::operator==(const Result& rhs)
{
    return (memcmp(m_pHash, rhs.m_pHash, sizeof(m_pHash)) == 0);
}

MD5Hash::MD5Hash() NN_NOEXCEPT
{
    memset(_buffer, 0, 64);

    _count[0] = _count[1] = 0;

    // Load magic initialization constants.
    _state[0] = 0x67452301;
    _state[1] = 0xefcdab89;
    _state[2] = 0x98badcfe;
    _state[3] = 0x10325476;
}

void MD5Hash::Update (const unsigned char *input, unsigned int inputLen) NN_NOEXCEPT
{
    unsigned int i, bufindex, partLen;

    /* Compute number of bytes mod 64 */
    bufindex = (unsigned int)((_count[0] >> 3) & 0x3F);

    /* Update number of bits */
    if((_count[0] += ((UINT4)inputLen << 3)) < ((UINT4)inputLen << 3))
    {
        _count[1]++;
    }
    _count[1] += ((UINT4)inputLen >> 29);

    partLen = 64 - bufindex;

    /* Transform as many times as possible. */
    if(inputLen >= partLen)
    {
        memcpy(&_buffer[bufindex], input, partLen);
        _Transform(_buffer);

        for(i = partLen; i + 63 < inputLen; i += 64)
        {
            _Transform(&input[i]);
        }

        bufindex = 0;
    }
    else
    {
        i = 0;
    }

    /* Buffer remaining input */
    memcpy(&_buffer[bufindex], &input[i], inputLen - i);
}

void MD5Hash::Final(MD5Hash::Result& hashResult) NN_NOEXCEPT
{
    unsigned char bits[8];
    unsigned int count, padLen;

    /* Save number of bits */
    Encode(bits, _count, 8);

    /* Pad out to 56 mod 64. */
    count = (unsigned int)((_count[0] >> 3) & 0x3f);
    padLen = (count < 56) ? (56 - count) : (120 - count);
    Update(PADDING, padLen);

    /* Append length (before padding) */
    Update(bits, 8);

    /* Store state in digest */
    Encode(hashResult.m_pHash, _state, 16);

    /* Zeroize sensitive information. */
    memset ((void *)this, 0, sizeof (*this));
}

/* MD5 basic transformation. Transforms state based on block. */
void MD5Hash::_Transform(const unsigned char block[64]) NN_NOEXCEPT
{
    UINT4 a = _state[0], b = _state[1], c = _state[2], d = _state[3], x[16];

    Decode (x, block, 64);

    /* Round 1 */
    FF (a, b, c, d, x[ 0], S11, 0xd76aa478); /* 1 */
    FF (d, a, b, c, x[ 1], S12, 0xe8c7b756); /* 2 */
    FF (c, d, a, b, x[ 2], S13, 0x242070db); /* 3 */
    FF (b, c, d, a, x[ 3], S14, 0xc1bdceee); /* 4 */
    FF (a, b, c, d, x[ 4], S11, 0xf57c0faf); /* 5 */
    FF (d, a, b, c, x[ 5], S12, 0x4787c62a); /* 6 */
    FF (c, d, a, b, x[ 6], S13, 0xa8304613); /* 7 */
    FF (b, c, d, a, x[ 7], S14, 0xfd469501); /* 8 */
    FF (a, b, c, d, x[ 8], S11, 0x698098d8); /* 9 */
    FF (d, a, b, c, x[ 9], S12, 0x8b44f7af); /* 10 */
    FF (c, d, a, b, x[10], S13, 0xffff5bb1); /* 11 */
    FF (b, c, d, a, x[11], S14, 0x895cd7be); /* 12 */
    FF (a, b, c, d, x[12], S11, 0x6b901122); /* 13 */
    FF (d, a, b, c, x[13], S12, 0xfd987193); /* 14 */
    FF (c, d, a, b, x[14], S13, 0xa679438e); /* 15 */
    FF (b, c, d, a, x[15], S14, 0x49b40821); /* 16 */

    /* Round 2 */
    GG (a, b, c, d, x[ 1], S21, 0xf61e2562); /* 17 */
    GG (d, a, b, c, x[ 6], S22, 0xc040b340); /* 18 */
    GG (c, d, a, b, x[11], S23, 0x265e5a51); /* 19 */
    GG (b, c, d, a, x[ 0], S24, 0xe9b6c7aa); /* 20 */
    GG (a, b, c, d, x[ 5], S21, 0xd62f105d); /* 21 */
    GG (d, a, b, c, x[10], S22,  0x2441453); /* 22 */
    GG (c, d, a, b, x[15], S23, 0xd8a1e681); /* 23 */
    GG (b, c, d, a, x[ 4], S24, 0xe7d3fbc8); /* 24 */
    GG (a, b, c, d, x[ 9], S21, 0x21e1cde6); /* 25 */
    GG (d, a, b, c, x[14], S22, 0xc33707d6); /* 26 */
    GG (c, d, a, b, x[ 3], S23, 0xf4d50d87); /* 27 */
    GG (b, c, d, a, x[ 8], S24, 0x455a14ed); /* 28 */
    GG (a, b, c, d, x[13], S21, 0xa9e3e905); /* 29 */
    GG (d, a, b, c, x[ 2], S22, 0xfcefa3f8); /* 30 */
    GG (c, d, a, b, x[ 7], S23, 0x676f02d9); /* 31 */
    GG (b, c, d, a, x[12], S24, 0x8d2a4c8a); /* 32 */

    /* Round 3 */
    HH (a, b, c, d, x[ 5], S31, 0xfffa3942); /* 33 */
    HH (d, a, b, c, x[ 8], S32, 0x8771f681); /* 34 */
    HH (c, d, a, b, x[11], S33, 0x6d9d6122); /* 35 */
    HH (b, c, d, a, x[14], S34, 0xfde5380c); /* 36 */
    HH (a, b, c, d, x[ 1], S31, 0xa4beea44); /* 37 */
    HH (d, a, b, c, x[ 4], S32, 0x4bdecfa9); /* 38 */
    HH (c, d, a, b, x[ 7], S33, 0xf6bb4b60); /* 39 */
    HH (b, c, d, a, x[10], S34, 0xbebfbc70); /* 40 */
    HH (a, b, c, d, x[13], S31, 0x289b7ec6); /* 41 */
    HH (d, a, b, c, x[ 0], S32, 0xeaa127fa); /* 42 */
    HH (c, d, a, b, x[ 3], S33, 0xd4ef3085); /* 43 */
    HH (b, c, d, a, x[ 6], S34,  0x4881d05); /* 44 */
    HH (a, b, c, d, x[ 9], S31, 0xd9d4d039); /* 45 */
    HH (d, a, b, c, x[12], S32, 0xe6db99e5); /* 46 */
    HH (c, d, a, b, x[15], S33, 0x1fa27cf8); /* 47 */
    HH (b, c, d, a, x[ 2], S34, 0xc4ac5665); /* 48 */

    /* Round 4 */
    II (a, b, c, d, x[ 0], S41, 0xf4292244); /* 49 */
    II (d, a, b, c, x[ 7], S42, 0x432aff97); /* 50 */
    II (c, d, a, b, x[14], S43, 0xab9423a7); /* 51 */
    II (b, c, d, a, x[ 5], S44, 0xfc93a039); /* 52 */
    II (a, b, c, d, x[12], S41, 0x655b59c3); /* 53 */
    II (d, a, b, c, x[ 3], S42, 0x8f0ccc92); /* 54 */
    II (c, d, a, b, x[10], S43, 0xffeff47d); /* 55 */
    II (b, c, d, a, x[ 1], S44, 0x85845dd1); /* 56 */
    II (a, b, c, d, x[ 8], S41, 0x6fa87e4f); /* 57 */
    II (d, a, b, c, x[15], S42, 0xfe2ce6e0); /* 58 */
    II (c, d, a, b, x[ 6], S43, 0xa3014314); /* 59 */
    II (b, c, d, a, x[13], S44, 0x4e0811a1); /* 60 */
    II (a, b, c, d, x[ 4], S41, 0xf7537e82); /* 61 */
    II (d, a, b, c, x[11], S42, 0xbd3af235); /* 62 */
    II (c, d, a, b, x[ 2], S43, 0x2ad7d2bb); /* 63 */
    II (b, c, d, a, x[ 9], S44, 0xeb86d391); /* 64 */

    _state[0] += a;
    _state[1] += b;
    _state[2] += c;
    _state[3] += d;

    /* Zeroize sensitive information. */
    memset((void *)x, 0, sizeof (x));
}
