﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utils/ExchangeInfo.h"
#include <Utils/HTCSUtil.h>
#include <Utils/GTestUtil.h>

#include <nn/nifm/nifm_ApiIpAddress.h>

#include <cstdlib>
#include <memory>

#include "NetTest_Port.h"

namespace NATF {
namespace Utils {

    class ExchangeInfo::HtcsExchangeInfoSession: public nnt::net::SimpleHTCSPlug
    {

    private: // constants
        static const int BufSize = 1024;

    public: // constants
        static const int HtcsConnTimeoutMs = 60000;

    public:
        HtcsExchangeInfoSession( tics::CrossBar *pCrossBar, ExchangeInfo& exchangeInfo ):
                 nnt::net::SimpleHTCSPlug( pCrossBar,
                                           ExchangeInfo::SessionServerHost, ExchangeInfo::SessionServerPort,
                                           ExchangeInfo::SdevServerName,    ExchangeInfo::TicsSessionName ),
            m_buffer( BufSize ),
            m_pCrossBar( pCrossBar ),
            m_exchangeInfo(exchangeInfo),
            m_expectedStrLen(0)
        {
        }

        void OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP) NN_OVERRIDE;
        int  OnSendComplete(::tics::Buffer* buffer, long offset, long len) NN_OVERRIDE;
        int  OnReceiveComplete(::tics::Buffer* buffer, long offset, long len) NN_OVERRIDE;
        void PvtDetach();

        int OnDetach() NN_OVERRIDE
        {
            PvtDetach();
            return 0;
        }

        int OnRemoteDetach( ::tics::DisconnectReason ) NN_OVERRIDE
        {
            PvtDetach();
            return 0;
        }

    private:
        ::tics::RawMemBuffer m_buffer;
        tics::CrossBar *m_pCrossBar;
        ExchangeInfo& m_exchangeInfo;
        uint32_t m_expectedStrLen;
    };


    void ExchangeInfo::HtcsExchangeInfoSession::OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP)
    {
        NN_NETTEST_LOG("Htcs Session Started!\n");
        tics::Plug::OnSessionStarted( type, connectedEP );

        size_t strLen = strlen(m_exchangeInfo.m_pLocalIpStr);
        *reinterpret_cast<uint8_t*>(m_buffer.GetPayLoad()) = static_cast<uint8_t>(strLen);
        memcpy(reinterpret_cast<uint8_t*>(m_buffer.GetPayLoad()) + 1, &m_exchangeInfo.m_pLocalIpStr, strLen);

        BeginSend(&m_buffer, 0,  static_cast<long>(strLen + 1), true);
    }

    int ExchangeInfo::HtcsExchangeInfoSession::OnSendComplete(::tics::Buffer* buffer, long offset, long len)
    {
        NN_NETTEST_LOG("Send complete, now starting to receive.\n");
        BeginReceive( 1, IpBufLen, &m_buffer, 0 );
        return 0;
    }

    int ExchangeInfo::HtcsExchangeInfoSession::OnReceiveComplete(::tics::Buffer* buffer, long offset, long len)
    {
        if( m_expectedStrLen == 0 )
        {
            m_expectedStrLen = *reinterpret_cast<uint8_t*>(m_buffer.GetPayLoad());

            NN_NETTEST_LOG("Received expected ip string length: %d\n", m_expectedStrLen);
            if( m_expectedStrLen >= sizeof(m_exchangeInfo.m_pRemoteIpStr) )
            {
                NN_NETTEST_LOG("Expected ip string length too large for buffer. Buffer Len: %d\n", sizeof(m_exchangeInfo.m_pRemoteIpStr));
                BeginDetach();
                return 0;
            }

            BeginReceive( m_expectedStrLen, m_expectedStrLen, &m_buffer, 0 );
            return 0;
        }
        else if( static_cast<uint32_t>(len) != m_expectedStrLen )
        {
            NN_NETTEST_LOG("Error: Did not receive expected string length for ip address. Recvd Len: %d, Expected Len: %d\n\n", len, m_expectedStrLen);
        }
        else
        {
            const char* pIpStr = reinterpret_cast<char*>(m_buffer.GetPayLoad());
            memcpy(m_exchangeInfo.m_pRemoteIpStr, pIpStr, m_expectedStrLen);
            m_exchangeInfo.m_pRemoteIpStr[m_expectedStrLen] = '\0';
        }

        // disconnect regardless
        BeginDetach();
        return 0;
    }

    void ExchangeInfo::HtcsExchangeInfoSession::PvtDetach()
    {
        NN_NETTEST_LOG("\nv3Detach event caught in session, will shutdown\n");
        m_pCrossBar->BeginShutdown(::tics::CrossBar::ExitDone);
    }

    bool ExchangeInfo::Exchange() NN_NOEXCEPT
    {
        nn::Result result;
        bool isSuccess = true;
        nn::socket::InAddr localAddress;

        memset(&localAddress, 0, sizeof(localAddress));
        result = nn::nifm::GetCurrentPrimaryIpAddress( &localAddress );
        EXCHANGE_INFO_FAIL_IF( result.IsFailure(), "Failed to get address.\n\n" );

        memset(m_pRemoteIpStr, 0, sizeof(m_pRemoteIpStr));
        NETTEST_SNPRINTF(m_pLocalIpStr, sizeof(m_pLocalIpStr), "%s", nn::socket::InetNtoa(localAddress));
        NN_NETTEST_LOG("Ip Address: %s\n\n", m_pLocalIpStr);

        try
        {

            tics::CrossBar *pCb;
            HtcsExchangeInfoSession *pSession;

            static char __tics_heap[ 16 * 1024 * 1024 ];

            // Initialization
            ::tics::Initialize( (uintptr_t)__tics_heap, sizeof(__tics_heap) );


            NN_NETTEST_LOG("\nStarting: ExchangeInfo v6\n");


            pCb = new tics::CrossBar();

            pSession = new HtcsExchangeInfoSession( pCb, *this );

            EXCHANGE_INFO_FAIL_IF( !pSession->GetFactory().ParseStateOk(), "ERROR: Parser did not initialize successfully\n", pCb);

            pSession->GetFactory().SetAutoTeardown(); // disconnect when found
            pSession->GetFactory().SetTimeout( HtcsExchangeInfoSession::HtcsConnTimeoutMs );

            // blocking - will run all HTCS communication (host to discovery session plus session itself)
            pCb->Run();

            // order matters here
            delete pSession;
        }
        catch( const ::tics::portability::stl::string &xcpt )
        {
            NN_NETTEST_LOG("\nError: Exception [%s]\n\n", xcpt.c_str() );
            EXCHANGE_INFO_FAIL();
        }
        catch(...)
        {
            NN_NETTEST_LOG("\nError: unknown exception thrown during test\n\n");
            EXCHANGE_INFO_FAIL();
        }

out:
        return isSuccess;
    }

}} // Namespaces NATF::Utils
