﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/SoHttpDownloadMultiTest.h"
#include <nn/nifm.h>

#include <new> // For placement new
#include <cstring>
#include <cstdlib> // For Malloc

namespace NATF {
namespace Tests {

    const Utils::InitApiFlags SoHttpDownloadMulti::InitFlags = Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket;

    // Constructor
    SoHttpDownloadMulti::SoHttpDownloadMulti(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, uint32_t connectionCount, const char* pIp, uint16_t portNum, const char* pResource, const MD5Hash::Result& hashResult, bool useSelect, bool doInit, bool doNnLinger) NN_NOEXCEPT
        : BaseTest(pTestName, false, static_cast<uint32_t>(doInit) * InitFlags, netProfile),
          m_connectionCount(connectionCount),
          m_pModules(nullptr),
          m_expectedHash(hashResult),
          m_portNum(portNum),
          m_timeoutMs(timeoutMs),
          m_useSelect(useSelect),
          m_DoNnLinger(doNnLinger)
    {
        if( pIp )
        {
            NETTEST_SNPRINTF(m_pIp, sizeof(m_pIp), "%s", pIp);
        }
        else
        {
            m_pIp[0] = '\0';
        }

        if( pResource )
        {
            NETTEST_SNPRINTF(m_pResource, sizeof(m_pResource), "%s", pResource);
        }
        else
        {
            m_pResource[0] = '\0';
        }
    }

    // Destructor
    SoHttpDownloadMulti::~SoHttpDownloadMulti() NN_NOEXCEPT
    {
        FreeModules();
    }

    // Cleanup
    bool SoHttpDownloadMulti::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        return true;
    }

    // Config
    bool SoHttpDownloadMulti::Config() NN_NOEXCEPT
    {
        if( m_connectionCount > 999 )
        {
            LogError(" * Number of connection must be more than less than 999.\n\n");
            return false;
        }

        if( !AllocateModules() )
        {
            return false;
        }

        for(unsigned i = 0; i < m_connectionCount; ++i)
        {
            char pThreadName[] = "SoHttpDown_000";
            size_t nameLength = strlen(pThreadName);
            NETTEST_SNPRINTF(&pThreadName[nameLength - 3], 3, "%d", i + 1);


            TestThread* pServerThread = CreateTestThread(pThreadName, m_timeoutMs);
            if( !pServerThread )
            {
                return false;
            }

            pServerThread->AddModule(m_pModules[i]);
        }

        return true;
    }

    // AllocateModules
    bool SoHttpDownloadMulti::AllocateModules() NN_NOEXCEPT
    {
        // Allocate Server modules
        m_pModules = (Modules::SoHttpDownload*)malloc(sizeof(Modules::SoHttpDownload) * m_connectionCount);
        if( !m_pModules )
        {
            LogError(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }
        else
        {
            // Initialize all server modules with placement new.
            for(unsigned i = 0; i < m_connectionCount; ++i)
            {
                new (&m_pModules[i]) Modules::SoHttpDownload(m_pIp, m_portNum, m_pResource, m_expectedHash, m_useSelect, m_DoNnLinger);
            }
        }

        return true;
    }

    // FreeModules
    void SoHttpDownloadMulti::FreeModules() NN_NOEXCEPT
    {
        if( m_pModules )
        {
            // Call destructors for modules.
            Log("Calling destructors on modules...\n");
            for(unsigned i = 0; i < m_connectionCount; ++i)
            {
                (&m_pModules[i])->~SoHttpDownload();
            }

            Log("Deallocating modules...\n");
            free(m_pModules);
            m_pModules = nullptr;
        }
    }

} // namespace Tests
} // namespace NATF
