﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/ResolverGetNameInfoTest.h"
#include <nn/nifm.h>

// =======================================================
//
//#define VERBOSE
#define SKIP_IN_DEVELOP_TESTS
#define SKIP_CRASHING_TESTS
#define SKIP_FAILING_TESTS
//
// =======================================================

// MARK_UNTRACKED_TEST
#if defined VERBOSE
#define MARK_UNTRACKED_TEST(message) do { \
    Log("RUN (UNTRACKED): %s\n", __FUNCTION__); \
    Log("** Expecting the test will not pass because:\n"); \
    Log("** %s\n", (message)); \
    Log("** Now running untracked ...\n"); \
    ::g_numUntrackedTests++; \
} while (false)
#else
#define MARK_UNTRACKED_TEST(message) do { \
    Log("RUN (UNTRACKED): %s\n", __FUNCTION__); \
    ::g_numUntrackedTests++; \
} while (false)
#endif

// MARK_PASSING_TEST
#define MARK_PASSING_TEST do { \
    Log("RUN (PASSING): %s\n", __FUNCTION__); \
    ::g_numPassingTests++; \
} while (false)

// MARK_AND_ABORT_IN_DEVELOP_TEST
#if defined SKIP_IN_DEVELOP_TESTS && defined VERBOSE
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) do { \
    Log("SKIP (IN DEVELOP): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will still be in develop because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numInDevelopTests++; \
    return; \
} while (false)
#elif defined SKIP_IN_DEVELOP_TESTS
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) do { \
    Log("SKIP (IN DEVELOP): %s\n\n", __FUNCTION__); \
    ::g_numInDevelopTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// MARK_AND_ABORT_CRASHING_TEST
#if defined SKIP_CRASHING_TESTS && defined VERBOSE
#define MARK_AND_ABORT_CRASHING_TEST(message) do { \
    Log("SKIP (CRASHING): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will crash the process because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numCrashingTests++; \
    return; \
} while (false)
#elif defined SKIP_CRASHING_TESTS
#define MARK_AND_ABORT_CRASHING_TEST(message) do { \
    Log("SKIP (CRASHING): %s\n\n", __FUNCTION__); \
    ::g_numCrashingTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_CRASHING_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// MARK_AND_ABORT_FAILING_TEST
#if defined SKIP_FAILING_TESTS && defined VERBOSE
#define MARK_AND_ABORT_FAILING_TEST(message) do { \
    Log("SKIP (FAILING): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will fail because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numFailingTests++; \
    return; \
} while (false)
#elif defined SKIP_FAILING_TESTS
#define MARK_AND_ABORT_FAILING_TEST(message) do { \
    Log("SKIP (FAILING): %s\n\n", __FUNCTION__); \
    ::g_numFailingTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_FAILING_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// DEFAULT_GAI_RES_OVERRIDE
#define DEFAULT_GAI_RES_OVERRIDE {                                  \
                                  {                                 \
                                    nn::socket::AddrInfoFlag::Ai_None,  \
                                    nn::socket::Family::Af_Unspec,      \
                                    nn::socket::Type::Sock_Default,  \
                                    nn::socket::Protocol::IpProto_Ip,      \
                                    0, nullptr, nullptr, nullptr    \
                                  },                                \
                                  0                                 \
                                }

namespace
{
    const int g_nameLenMax = 256;

    int g_numUntrackedTests = 0;
    int g_numPassingTests = 0;
    int g_numInDevelopTests = 0;
    int g_numCrashingTests = 0;
    int g_numFailingTests = 0;
    nn::socket::AddrInfoFlag g_InvalidAiFlags = nn::socket::AddrInfoFlag::Ai_None;

    char g_GoodHostNameNintendo[g_nameLenMax];
    char g_GoodHostNameExample[g_nameLenMax];
    char g_GoodIpAddress[g_nameLenMax];
    char g_GoodServiceLocation[g_nameLenMax];
    char g_BadHostNameGoogle[g_nameLenMax];
    char g_BadIpAddress[g_nameLenMax];
    char g_BadServiceLocation[g_nameLenMax];
}

namespace NATF {
namespace Tests {

    // Constructor
    ResolverGetNameInfoTest::ResolverGetNameInfoTest(const char* testName, const nn::util::Uuid& netProfile) NN_NOEXCEPT
    :   BaseTest(testName, false, Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket, netProfile),
        m_pResolverApiModule(nullptr)
    {
        // Set invalid ai flags
        ::g_InvalidAiFlags = nn::socket::AddrInfoFlag::Ai_None;
#ifdef AI_PASSIVE
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_Passive;
#endif
#ifdef AI_CANONNAME
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_CanonName;
#endif
#ifdef AI_NUMERICSERV
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_NumericServ;
#endif
#ifdef AI_ALL
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_ALL);
#endif
#ifdef AI_V4MAPPED_CFG
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_V4MAPPED_CFG);
#endif
#ifdef AI_ADDRCONFIG
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_AddrConfig;
#endif
#ifdef AI_V4MAPPED
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_V4MAPPED);
#endif
        ::g_InvalidAiFlags = ~::g_InvalidAiFlags;

        // Set test node information
        strcpy(::g_GoodHostNameNintendo, "www.nintendo.com");
        strcpy(::g_GoodHostNameExample, "www.example.com");
        strcpy(::g_GoodIpAddress, "205.166.76.26");
        strcpy(::g_GoodServiceLocation, "http");
        strcpy(::g_BadHostNameGoogle, "https://www.google.com"); // can't contain url
        strcpy(::g_BadIpAddress, "1..2.3.4"); // poorly formed ip address
        strcpy(::g_BadServiceLocation, "nmzxcjasdfuioewqio"); // no such service
    }

    // Destructor
    ResolverGetNameInfoTest::~ResolverGetNameInfoTest() NN_NOEXCEPT
    {}

    // Cleanup
    bool ResolverGetNameInfoTest::Cleanup() NN_NOEXCEPT
    {
        Log("SUMMARY: Untracked tests = %d\n", ::g_numUntrackedTests);
        Log("SUMMARY: Passing tests = %d\n", ::g_numPassingTests);
        Log("SUMMARY: In develop tests = %d\n", ::g_numInDevelopTests);
        Log("SUMMARY: Crashing tests = %d\n", ::g_numCrashingTests);
        Log("SUMMARY: Failing tests = %d\n", ::g_numFailingTests);
        Log("SUMMARY: Total tests = %d\n", ::g_numUntrackedTests
            + ::g_numPassingTests + ::g_numCrashingTests + ::g_numFailingTests
            + ::g_numInDevelopTests);

        delete(m_pResolverApiModule);

        return true;
    }

    // Config
    bool ResolverGetNameInfoTest::Config() NN_NOEXCEPT
    {
        if(nullptr == m_pResolverApiModule)
        {
            Log(" * Aborting: m_pResolverApiModule not initialized.\n\n");
            return true;
        }

        TestThread* pThread1 = CreateTestThread("Thread_1", 300);
        if( !pThread1 )
        {
            return false;
        }

        pThread1->AddModule(*m_pResolverApiModule);

        return true;
    }

    // CheckNoError
    void ResolverGetNameInfoTest::CheckNoError() NN_NOEXCEPT
    {
        MARK_PASSING_TEST;

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and it can be located, then the name is
        // provided as null-terminated char string and 0 is returned.

        const size_t numTests = 3;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Success;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax] = "";
        char hostExpecBuffer[::g_nameLenMax] = "nintendo.com";
        char servOutBuffer[::g_nameLenMax] = "";
        char servExpecBuffer[::g_nameLenMax] = "http";
        const char* testDesc[numTests] =
        {
            "only service requested",
            "only host requested",
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host not requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // host requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { hostExpecBuffer,  sizeof(hostExpecBuffer) },
            },
            { // host requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { hostExpecBuffer,  sizeof(hostExpecBuffer) }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,   4  },
                { servOutBuffer,    ::g_nameLenMax   },
                { servExpecBuffer,  ::g_nameLenMax }
            },
            { // serv not requested
                { g_GoodServiceLocation,   sizeof(g_GoodServiceLocation)  },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // serv requested
                { g_GoodServiceLocation,   sizeof(g_GoodServiceLocation)  },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { servExpecBuffer,  sizeof(servExpecBuffer) }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiAgain
    void ResolverGetNameInfoTest::CheckEaiAgain() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("Need to determine how to inject dns timeout");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name), the name could not be resolved at this time
        // (future attempts may succeed) and flags nn::socket::NameInfoFlag::Ni_NameReqd is not set, then
        // the numeric form of the address is provided as null-terminated char
        // string and nn::socket::AiErrno::EAi_Again is returned.

        // TODO

        Log("TEST FAILED\n\n");
        return;
    }

    // CheckEaiBadflags
    void ResolverGetNameInfoTest::CheckEaiBadflags() NN_NOEXCEPT
    {
        MARK_AND_ABORT_FAILING_TEST("During 'both host and service requested' ...\n"
            "nn::socket::GetNameInfo: returned Success (0), expected Invalid value for ai_flags (3)");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and flags is invalid, then no results are
        // provided and EAI_BAD_FLAGS is returned.

        const size_t numTests = 1;
        nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_BadFlags;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            }
        };

        // Note: Cannot check which values exist at compile time (i.e. #ifdef),
        // so this calculation is done after inspecting netdb.h directly
        flagsIn = nn::socket::NameInfoFlag::Ni_None;
        flagsIn |= nn::socket::NameInfoFlag::Ni_NoFqdn;
        flagsIn |= nn::socket::NameInfoFlag::Ni_NumericHost;
        flagsIn |= nn::socket::NameInfoFlag::Ni_NameReqd;
        flagsIn |= nn::socket::NameInfoFlag::Ni_NumericServ;
        Log("NI_NUMERICSCOPE not defined\n");
        flagsIn |= nn::socket::NameInfoFlag::Ni_Dgram;
        flagsIn = ~flagsIn; // set to invalid value

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);

    }

    // CheckEaiFail
    void ResolverGetNameInfoTest::CheckEaiFail() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("Need to determine which non-recoverable errors to inject");

        // On exit, if a non-recoverable error occurred during name resolution,
        // then no results are provided and nn::socket::AiErrno::EAi_Fail is returned

        // TODO: Need to determine which non-recoverable errors to inject:
        // (1) Couldn't find database file (i.e. HOSTS, SERVICES, PROTOCOLS)
        // (2) DNS request returned by server with severe error (i.e. DNSMASQ)

        // SEE:
        // http://stackoverflow.com/questions/4368877/net-proccess-failing-dns-lookup-with-windows-7
        // http://www.thekelleys.org.uk/dnsmasq/doc.html

        Log("TEST FAILED\n\n");
        return;
    }

    // CheckEaiFamilyWithUnrecognizedAddressFamily
    void ResolverGetNameInfoTest::CheckEaiFamilyWithUnrecognizedAddressFamily() NN_NOEXCEPT
    {
        MARK_AND_ABORT_CRASHING_TEST("During 'both host and service requested' ...\n"
            "SDK Assertion Failure: 'false' at 'C:\\Users\\andrch07\\work\\s1220\\Programs\\Chris\\Sources\\Libraries\\socket\\resolver\\client\\resolver_ClientShim.cpp':457 in GetNameInfo\n"
            "GetNameInfoRequest failure");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the address family was not recognized,
        // then no results are provided and nn::socket::AiErrno::EAi_Family is returned.

        const size_t numTests = 1;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Family;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Family;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = static_cast<nn::socket::Family>(static_cast<int>(nn::socket::Family::Af_Max) + 1); // should be unsupported
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiFamilyWithInvalidAddressLength
    void ResolverGetNameInfoTest::CheckEaiFamilyWithInvalidAddressLength() NN_NOEXCEPT
    {
        MARK_AND_ABORT_CRASHING_TEST("During 'both host and service requested' ...\n"
            "SDK Assertion Failure: 'false' at 'C:\\Users\\andrch07\\work\\s1220\\Programs\\Chris\\Sources\\Libraries\\socket\\resolver\\client\\resolver_ClientShim.cpp':457 in GetNameInfo\n"
            "GetNameInfoRequest failure");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the address length was invalid for the
        // specified family, then no results are provided and nn::socket::AiErrno::EAi_Family is
        // returned.

        const size_t numTests = 1;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Family;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;
        Modules::ResolverApiModule::gaiResOverrideType gaiResOverride = DEFAULT_GAI_RES_OVERRIDE;
        gaiResOverride.info.ai_addrlen = 0; // should be invalid (normally 16)
        gaiResOverride.flags |= Modules::ResolverApiModule::gaiResOverrideAiAddrlen;

        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, &gaiResOverride, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiMemory
    void ResolverGetNameInfoTest::CheckEaiMemory() NN_NOEXCEPT
    {
        MARK_AND_ABORT_FAILING_TEST("Next call returned (null) (4), expected Memory allocation failure (6)");

        // On exit, if there was a memory allocation failure, then no results
        // are provided and nn::socket::AiErrno::EAi_Memory is returned.

        const size_t numTests = 1;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Success;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);

        m_pResolverApiModule->SetRunMode(Modules::ResolverApiModule::RUN_MODE_UNTIL_EAI_MEMORY);
    }

    // CheckEaiNonameWithNoArgs
    void ResolverGetNameInfoTest::CheckEaiNonameWithNoArgs() NN_NOEXCEPT
    {
        MARK_AND_ABORT_CRASHING_TEST("During 'neither host nor service requested (both ptr and len invalid)' ...\n"
            "SDK Assertion Failure: 'false' at 'C:\\Users\\andrch07\\work\\s1220\\Programs\\Chris\\Sources\\Libraries\\socket\\resolver\\client\\resolver_ClientShim.cpp':457 in GetNameInfo\n"
            "GetNameInfoRequest failure");

        // On exit, if no arguments are requested (i.e. neither host name nor
        // service name), then no results are provided and nn::socket::AiErrno::EAi_NoName is
        // returned.

        const size_t numTests = 3;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_NoName;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];
        const char* testDesc[numTests] =
        {
            "neither host nor service requested (both ptr and len invalid)",
            "neither host nor service requested (only ptr invalid)",
            "neither host nor service requested (only len invalid)"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // both ptr and len invalid
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // only ptr invalid
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { nullptr,          sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            },
            { // only len invalid
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    0                       },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // both ptr and len invalid
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // only ptr invalid
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { nullptr,          sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            },
            { // only len invalid
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    0                       },
                { nullptr,          0                       }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);

    }

    // CheckEaiNonameWithNiNamereqd
    void ResolverGetNameInfoTest::CheckEaiNonameWithNiNamereqd() NN_NOEXCEPT
    {

        MARK_AND_ABORT_CRASHING_TEST("During 'both host and service requested' ...\n"
            "SDK Assertion Failure: 'false' at 'C:\\Users\\andrch07\\work\\s1220\\Programs\\Chris\\Sources\\Libraries\\socket\\resolver\\client\\resolver_ClientShim.cpp':457 in GetNameInfo\n"
            "GetNameInfoRequest failure");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name), it cannot be located and flags nn::socket::NameInfoFlag::Ni_NameReqd is
        // set, then the numeric form of the address is provided as
        // null-terminated char string and nn::socket::AiErrno::EAi_NoName is returned.

        const size_t numTests = 1;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_NameReqd;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_NoName;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_NoData;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax];
        char hostExpecBuffer[::g_nameLenMax] = ""; // TODO
        char servOutBuffer[::g_nameLenMax];
        char servExpecBuffer[::g_nameLenMax] = ""; // TODO
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_BadHostNameGoogle,    sizeof(g_BadHostNameGoogle)   },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { hostExpecBuffer,  sizeof(hostExpecBuffer) }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { servExpecBuffer,  sizeof(servExpecBuffer) }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiOverflow
    void ResolverGetNameInfoTest::CheckEaiOverflow() NN_NOEXCEPT
    {
        MARK_AND_ABORT_FAILING_TEST("During 'both host and service requested' ...\n"
            "nn::socket::GetNameInfo: returned Memory allocation failure (6), expected Argument buffer overflow (14)");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the argument buffer overflowed (i.e.
        // buffer was too small), then no results are provided and nn::socket::AiErrno::EAi_Overflow
        // is returned.

        const size_t numTests = 1;
        const nn::socket::NameInfoFlag flagsIn = nn::socket::NameInfoFlag::Ni_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Overflow;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        const int nameLen = 1; // should be too small
        char hostOutBuffer[nameLen];
        char servOutBuffer[nameLen];
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { g_GoodHostNameNintendo,       sizeof(g_GoodHostNameNintendo)      },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { nullptr,          0                       }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,        sizeof(g_GoodServiceLocation)       },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { nullptr,          0                       }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiSystem
    void ResolverGetNameInfoTest::CheckEaiSystem() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("Need to determine which system errors to inject");

        // On exit, if there was a system error, then no results are
        // provided, nn::socket::AiErrno::EAi_System is returned and the error code is avilable via
        // errno.

        // TODO

        Log("TEST FAILED\n\n");
        return;
    }

} // namespace Tests
} // namespace NATF
