﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/ResolverGetAddrInfoTest.h"
#include <nn/nifm.h>

// =======================================================
//
//#define VERBOSE
#define SKIP_IN_DEVELOP_TESTS
#define SKIP_CRASHING_TESTS
#define SKIP_FAILING_TESTS
//
// =======================================================

// MARK_UNTRACKED_TEST
#if defined VERBOSE
#define MARK_UNTRACKED_TEST(message) do { \
    Log("RUN (UNTRACKED): %s\n", __FUNCTION__); \
    Log("** Expecting the test will not pass because:\n"); \
    Log("** %s\n", (message)); \
    Log("** Now running untracked ...\n"); \
    ::g_numUntrackedTests++; \
} while (false)
#else
#define MARK_UNTRACKED_TEST(message) do { \
    Log("RUN (UNTRACKED): %s\n", __FUNCTION__); \
    ::g_numUntrackedTests++; \
} while (false)
#endif

// MARK_PASSING_TEST
#define MARK_PASSING_TEST do { \
    Log("RUN (PASSING): %s\n", __FUNCTION__); \
    ::g_numPassingTests++; \
} while (false)

// MARK_AND_ABORT_IN_DEVELOP_TEST
#if defined SKIP_IN_DEVELOP_TESTS && defined VERBOSE
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) do { \
    Log("SKIP (IN DEVELOP): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will still be in develop because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numInDevelopTests++; \
    return; \
} while (false)
#elif defined SKIP_IN_DEVELOP_TESTS
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) do { \
    Log("SKIP (IN DEVELOP): %s\n\n", __FUNCTION__); \
    ::g_numInDevelopTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_IN_DEVELOP_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// MARK_AND_ABORT_CRASHING_TEST
#if defined SKIP_CRASHING_TESTS && defined VERBOSE
#define MARK_AND_ABORT_CRASHING_TEST(message) do { \
    Log("SKIP (CRASHING): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will crash the process because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numCrashingTests++; \
    return; \
} while (false)
#elif defined SKIP_CRASHING_TESTS
#define MARK_AND_ABORT_CRASHING_TEST(message) do { \
    Log("SKIP (CRASHING): %s\n\n", __FUNCTION__); \
    ::g_numCrashingTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_CRASHING_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// MARK_AND_ABORT_FAILING_TEST
#if defined SKIP_FAILING_TESTS && defined VERBOSE
#define MARK_AND_ABORT_FAILING_TEST(message) do { \
    Log("SKIP (FAILING): %s\n", __FUNCTION__); \
    Log("** When enabled, the test will fail because:\n"); \
    Log("** %s\n\n", (message)); \
    ::g_numFailingTests++; \
    return; \
} while (false)
#elif defined SKIP_FAILING_TESTS
#define MARK_AND_ABORT_FAILING_TEST(message) do { \
    Log("SKIP (FAILING): %s\n\n", __FUNCTION__); \
    ::g_numFailingTests++; \
    return; \
} while (false)
#else
#define MARK_AND_ABORT_FAILING_TEST(message) MARK_UNTRACKED_TEST(message)
#endif

// DEFAULT_GAI_RES_OVERRIDE
#define DEFAULT_GAI_RES_OVERRIDE {{0}, 0}

namespace
{
    const int g_nameLenMax = 256;

    int g_numUntrackedTests = 0;
    int g_numPassingTests = 0;
    int g_numInDevelopTests = 0;
    int g_numCrashingTests = 0;
    int g_numFailingTests = 0;
    nn::socket::AddrInfoFlag g_InvalidAiFlags;

    char g_GoodHostNameNintendo[g_nameLenMax];
    char g_GoodHostNameExample[g_nameLenMax];
    char g_GoodIpAddress[g_nameLenMax];
    char g_GoodServiceLocation[g_nameLenMax];
    char g_BadHostNameGoogle[g_nameLenMax];
    char g_BadIpAddress[g_nameLenMax];
    char g_BadServiceLocation[g_nameLenMax];
}

namespace NATF {
namespace Tests {

    // Constructor
    ResolverGetAddrInfoTest::ResolverGetAddrInfoTest(const char* testName, const nn::util::Uuid& netProfile) NN_NOEXCEPT
    :   BaseTest(testName, false, Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket, netProfile),
        m_pResolverApiModule(nullptr)
    {
        // Set invalid ai flags
        ::g_InvalidAiFlags = nn::socket::AddrInfoFlag::Ai_None;
#ifdef AI_PASSIVE
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_Passive;
#endif
#ifdef AI_CANONNAME
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_CanonName;
#endif
#ifdef AI_NUMERICSERV
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_NumericServ;
#endif
#ifdef AI_ALL
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_ALL);
#endif
#ifdef AI_V4MAPPED_CFG
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_V4MAPPED_CFG);
#endif
#ifdef AI_ADDRCONFIG
        ::g_InvalidAiFlags |= nn::socket::AddrInfoFlag::Ai_AddrConfig;
#endif
#ifdef AI_V4MAPPED
        ::g_InvalidAiFlags |= static_cast<nn::socket::AddrInfoFlag>(AI_V4MAPPED);
#endif
        ::g_InvalidAiFlags = ~::g_InvalidAiFlags;

        // Set test node information
        strcpy(::g_GoodHostNameNintendo, "www.nintendo.com");
        strcpy(::g_GoodHostNameExample, "www.example.com");
        strcpy(::g_GoodIpAddress, "205.166.76.26");
        strcpy(::g_GoodServiceLocation, "http");
        strcpy(::g_BadHostNameGoogle, "https://www.google.com"); // can't contain url
        strcpy(::g_BadIpAddress, "1..2.3.4"); // poorly formed ip address
        strcpy(::g_BadServiceLocation, "nmzxcjasdfuioewqio"); // no such service
    }

    // Destructor
    ResolverGetAddrInfoTest::~ResolverGetAddrInfoTest() NN_NOEXCEPT
    {}

    // Cleanup
    bool ResolverGetAddrInfoTest::Cleanup() NN_NOEXCEPT
    {
        Log("SUMMARY: Untracked tests = %d\n", ::g_numUntrackedTests);
        Log("SUMMARY: Passing tests = %d\n", ::g_numPassingTests);
        Log("SUMMARY: In develop tests = %d\n", ::g_numInDevelopTests);
        Log("SUMMARY: Crashing tests = %d\n", ::g_numCrashingTests);
        Log("SUMMARY: Failing tests = %d\n", ::g_numFailingTests);
        Log("SUMMARY: Total tests = %d\n", ::g_numUntrackedTests
            + ::g_numPassingTests + ::g_numCrashingTests + ::g_numFailingTests
            + ::g_numInDevelopTests);

        delete(m_pResolverApiModule);

        return true;
    }

    // Config
    bool ResolverGetAddrInfoTest::Config() NN_NOEXCEPT
    {
        if(nullptr == m_pResolverApiModule)
        {
            Log(" * Aborting: m_pResolverApiModule not initialized.\n\n");
            return true;
        }

        TestThread* pThread1 = CreateTestThread("Thread_1", 300);
        if( !pThread1 )
        {
            return false;
        }

        pThread1->AddModule(*m_pResolverApiModule);

        return true;
    }

    // CheckNoError
    void ResolverGetAddrInfoTest::CheckNoError() NN_NOEXCEPT
    {
        MARK_PASSING_TEST;

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and it can be located, then the result is
        // provided as nn::socket::AddrInfo structure and 0 is returned.

        const size_t numTests = 3;
        const nn::socket::AddrInfoFlag flagsIn = nn::socket::AddrInfoFlag::Ai_None;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Success;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_Success;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        char hostOutBuffer[::g_nameLenMax] = "";
        char hostExpecBuffer[::g_nameLenMax] = "nintendo.com";
        char servOutBuffer[::g_nameLenMax] = "";
        char servExpecBuffer[::g_nameLenMax] = "http";
        const char* testDesc[numTests] =
        {
            "only service requested",
            "only host requested",
            "both host and service requested"
        };
        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host not requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // host requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { hostExpecBuffer,  sizeof(hostExpecBuffer) },
            },
            { // host requested
                { g_GoodHostNameNintendo,   sizeof(g_GoodHostNameNintendo)  },
                { hostOutBuffer,    sizeof(hostOutBuffer)   },
                { hostExpecBuffer,  sizeof(hostExpecBuffer) }
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { g_GoodServiceLocation,   4  },
                { servOutBuffer,    ::g_nameLenMax   },
                { servExpecBuffer,  ::g_nameLenMax }
            },
            { // serv not requested
                { g_GoodServiceLocation,   sizeof(g_GoodServiceLocation)  },
                { nullptr,          0                       },
                { nullptr,          0                       }
            },
            { // serv requested
                { g_GoodServiceLocation,   sizeof(g_GoodServiceLocation)  },
                { servOutBuffer,    sizeof(servOutBuffer)   },
                { servExpecBuffer,  sizeof(servExpecBuffer) }
            }
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiAddrfamily
    void ResolverGetAddrInfoTest::CheckEaiAddrfamily() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if the host is requested and it does not have any
        // network addresses in the requested address family, then
        // EAI_ADDRFAMILY is returned.
    }

    // CheckEaiAgain
    void ResolverGetAddrInfoTest::CheckEaiAgain() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the name could not be resolved at this time
        // (future attempts may succeed), then nn::socket::AiErrno::EAi_Again is returned.
    }

    // CheckEaiBadflagsWithInvalidFlags
    void ResolverGetAddrInfoTest::CheckEaiBadflagsWithInvalidFlags() NN_NOEXCEPT
    {
        MARK_AND_ABORT_FAILING_TEST("nn::socket::GetNameInfo: returned Invalid value for ai_flags (8),  expected Invalid value for ai_flags (3)\n");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and hints.ai_flags contains invalid flags,
        // then nn::socket::AiErrno::EAi_BadFlags is returned.

        const size_t numTests = 1;
        const nn::socket::AiErrno gniRvExpec = nn::socket::AiErrno::EAi_Success;
        const nn::socket::AiErrno gaiRvExpec = nn::socket::AiErrno::EAi_BadFlags;
        nn::socket::AddrInfo gaiHints = {};
        gaiHints.ai_family = nn::socket::Family::Af_Inet;
        gaiHints.ai_socktype = nn::socket::Type::Sock_Stream;

        nn::socket::AddrInfoFlag flagsIn = ::g_InvalidAiFlags;
        char hostOutBuffer[::g_nameLenMax];
        char servOutBuffer[::g_nameLenMax];

        const Modules::ResolverApiModule::argTestType host[numTests] =
        {
            { // host requested
                { ::g_GoodHostNameNintendo, sizeof(::g_GoodHostNameNintendo) },
                { hostOutBuffer, sizeof(hostOutBuffer) },  // requested
                { nullptr, 0 }  // not requested
            }
        };
        const Modules::ResolverApiModule::argTestType serv[numTests] =
        {
            { // serv requested
                { ::g_GoodServiceLocation, sizeof(::g_GoodServiceLocation) },
                { servOutBuffer, sizeof(servOutBuffer) },  // requested
                { nullptr, 0 }  // not requested
            }
        };
        const char* testDesc[numTests] =
        {
            "both host and service requested"
        };

        m_pResolverApiModule = new Modules::ResolverApiModule(
            numTests, &testDesc[0], &host[0], &serv[0],
            &gaiHints, nullptr, gaiRvExpec, static_cast<int>(flagsIn), gniRvExpec);
    }

    // CheckEaiBadflagsWithAiCanonname
    void ResolverGetAddrInfoTest::CheckEaiBadflagsWithAiCanonname() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name), hints.ai_flags included nn::socket::AddrInfoFlag::Ai_CanonName
        // and name was null, then nn::socket::AiErrno::EAi_BadFlags is returned.
    }

    // CheckEaiFail
    void ResolverGetAddrInfoTest::CheckEaiFail() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and a non-recoverable error occurred
        // during name resolution, then nn::socket::AiErrno::EAi_Fail is returned.
    }

    // CheckEaiFamily
    void ResolverGetAddrInfoTest::CheckEaiFamily() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the requested address family is
        // not supported, then nn::socket::AiErrno::EAi_Family is returned.
    }

    // CheckEaiMemory
    void ResolverGetAddrInfoTest::CheckEaiMemory() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if there is a memory allocation failure, then nn::socket::AiErrno::EAi_Memory
        // is returned.
    }

    // CheckEaiNodata
    void ResolverGetAddrInfoTest::CheckEaiNodata() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and the specified network host exists,
        // but it does not have any network addresses defined, then nn::socket::AiErrno::EAi_NoData
        // is returned.
    }

    // CheckEaiNonameWithUnknownArgs
    void ResolverGetAddrInfoTest::CheckEaiNonameWithUnknownArgs() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and either the node or service is not known,
        // then nn::socket::AiErrno::EAi_NoName is returned.
    }

    // CheckEaiNonameWithNullArgs
    void ResolverGetAddrInfoTest::CheckEaiNonameWithNullArgs() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and both the node and service are null, then
        // nn::socket::AiErrno::EAi_NoName is returned.
    }

    // CheckEaiNonameWithAiNumericserv
    void ResolverGetAddrInfoTest::CheckEaiNonameWithAiNumericserv() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) and nn::socket::AddrInfoFlag::Ai_NumericServ was specified in
        // hints.ai_flags and service was not a numeric port-number string,
        // then nn::socket::AiErrno::EAi_NoName is returned.
    }

    // CheckEaiService
    void ResolverGetAddrInfoTest::CheckEaiService() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) but the requested service is not available
        // for the requested socket type, then nn::socket::AiErrno::EAi_Service is returned.
    }

    // CheckEaiSocktype
    void ResolverGetAddrInfoTest::CheckEaiSocktype() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if at least one argument was requested (i.e. either host
        // name or service name) but the requested socket type is not
        // supported, then nn::socket::AiErrno::EAi_Service is returned.
    }

    // CheckEaiSystem
    void ResolverGetAddrInfoTest::CheckEaiSystem() NN_NOEXCEPT
    {
        MARK_AND_ABORT_IN_DEVELOP_TEST("TODO: Determine test scenario");

        // On exit, if there is another system error, then nn::socket::AiErrno::EAi_System is
        // returned and error details are provided by errno.
    }

} // namespace Tests
} // namespace NATF
