﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/IperfTest.h"
#include <nn/nifm.h>

#include <new> // For placement new
#include <cstring>
#include <cstdlib> // For Malloc

namespace
{
    const char* const IperfProgramName = "iperf";
    const uint32_t MaxIpStrBufLen = 16;
}

namespace NATF {
namespace Tests {

    // Constructor
    Iperf::Iperf(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, bool doIpExchange, const char* pIperfArgString) NN_NOEXCEPT
        : BaseTest(pTestName, doIpExchange, Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket, netProfile),
          m_pIperf(nullptr),
          m_timeoutMs(timeoutMs),
          m_pIperfArgString(pIperfArgString),
          m_pArgV(nullptr)
    {}

    // Destructor
    Iperf::~Iperf() NN_NOEXCEPT
    {
        Cleanup();
    }

    // Cleanup
    bool Iperf::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        FreeArgV();
        return true;
    }

    // Config
    bool Iperf::Config() NN_NOEXCEPT
    {
        if( !AllocateModules() )
        {
            return false;
        }

        TestThread* pThread = CreateTestThread("", m_timeoutMs);
        if( !pThread )
        {
            return false;
        }

        pThread->AddModule(*m_pIperf);

        return true;
    }

    // AllocateModules
    bool Iperf::AllocateModules() NN_NOEXCEPT
    {
        int argC = AllocateArgV(m_pIperfArgString);
        if( argC < 0 )
        {
            LogError(" Failed to construct argv for iperf!\n\n");
            return false;
        }

        // Allocate Server modules
        m_pIperf = new Modules::Iperf(argC, m_pArgV);
        if( !m_pIperf )
        {
            LogError(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }

        return true;
    }

    // FreeModules
    void Iperf::FreeModules() NN_NOEXCEPT
    {
        if( m_pIperf )
        {
            Log("Deallocating modules...\n");
            delete m_pIperf;
            m_pIperf = nullptr;
        }
    }

    void Iperf::FreeArgV() NN_NOEXCEPT
    {
        if( m_pArgV )
        {
            Log("Deallocating ArgV.");
            char** pCurr = m_pArgV;
            while( *pCurr )
            {
                NN_LOG(".");
                delete [] *pCurr;
                ++pCurr;
            }

            delete [] m_pArgV;
            m_pArgV = nullptr;
            NN_LOG("\n");
        }
    }

    int Iperf::AllocateArgV(const char* pArgStringIn) NN_NOEXCEPT
    {
        int argC = 1; // Start at one because of the program name.
        const char* pCurr = pArgStringIn;
        bool isParsingArg = false;

        // Count the number of arguments
        while( *pCurr != '\0' )
        {
            if( isParsingArg )
            {
                if( *pCurr == ' ' )
                {
                    isParsingArg = false;
                }
            }
            else
            {
                if( *pCurr != ' ' )
                {
                    ++argC;
                    isParsingArg = true;
                }
            }

            ++pCurr;
        }

        m_pArgV = new char*[argC + 1]; // One more for null termination
        if( !m_pArgV )
        {
            LogError("\nError: Failed to allocate memory!\n\n");
            return -1;
        }

        memset(m_pArgV, 0, sizeof(char*) * (argC + 1)); // One extra for null terminator

        m_pArgV[0] = new char[strlen(IperfProgramName) + 1];
        if( !m_pArgV[0] )
        {
            LogError("\nError: Failed to allocate memory!\n\n");
            FreeArgV();
            return -1;
        }

        strcpy(m_pArgV[0], IperfProgramName);

        int iArg = 1; // Start at one because we already filled arg[0] with the program name
        const char* pBeginArg = nullptr;
        isParsingArg = false;
        pCurr = pArgStringIn;

        // Allocate and copy over each argument
        while( *pCurr != '\0' )
        {
            if( isParsingArg )
            {
                if( *pCurr == ' ' )
                {
                    bool isHostName = false;
                    size_t hostNameLen = strlen(BaseTest::RemoteHostName);
                    size_t argLen = (pCurr - pBeginArg);
                    size_t bufLen = argLen + 1;

                    if( hostNameLen == argLen && memcmp(BaseTest::RemoteHostName, pBeginArg, hostNameLen) == 0 )
                    {
                        bufLen = MaxIpStrBufLen;
                        isHostName = true;
                    }

                    m_pArgV[iArg] = new char[bufLen];
                    if( !m_pArgV[iArg] )
                    {
                        LogError("\nError: Failed to allocate memory!\n\n");
                        FreeArgV();
                        return -1;
                    }

                    memcpy(m_pArgV[iArg], pBeginArg, static_cast<uint32_t>(argLen));
                    m_pArgV[iArg][argLen] = '\0';

                    if( isHostName )
                    {
                        if( !WriteHostIp(m_pArgV[iArg], static_cast<uint32_t>(bufLen)) )
                        {
                            LogError("Failed to write remote host ip\n\n");
                            FreeArgV();
                            return -1;
                        }
                    }

                    ++iArg;
                    pBeginArg = pCurr;
                    isParsingArg = false;
                }
            }
            else if( *pCurr != ' ' )
            {
                pBeginArg = pCurr;
                isParsingArg = true;
            }

            ++pCurr;
        }

        if( pBeginArg )
        {
            m_pArgV[iArg] = new char[pCurr - pBeginArg + 1];
            if( !m_pArgV[iArg] )
            {
                LogError("\nError: Failed to allocate memory!\n\n");
                FreeArgV();
                return -1;
            }

            memcpy(m_pArgV[iArg], pBeginArg, pCurr - pBeginArg);
            m_pArgV[iArg][pCurr - pBeginArg] = '\0';
        }

        return argC;
    }
}} // namespace NATF::Tests
