﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/CurlHttpUploadDownloadTest.h"
#include "curl/curl.h"
#include <nn/nifm.h>
#include <nn/fs.h>

#include "cstdlib" // For malloc

namespace NATF {
namespace Tests {

    const Utils::InitApiFlags CurlHttpUploadDownload::InitFlags = Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket | Utils::InitApiFlags::InitApiFlags_Curl;

    // Constructor
    CurlHttpUploadDownload::CurlHttpUploadDownload(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, bool doInit, const char* pDownloadUrl, const char* pUploadUrl, const Modules::LibCurl::Params& moduleParams) NN_NOEXCEPT
        :
        BaseTest(pTestName, false, static_cast<uint32_t>(doInit) * InitFlags, netProfile),
        m_timeoutMs(timeoutMs),
        m_pHttpDown(nullptr),
        m_pHttpUp(nullptr),
        m_params(moduleParams)
    {
        if( pDownloadUrl )
        {
            NETTEST_SNPRINTF(m_pDownloadUrl, sizeof(m_pDownloadUrl), "%s",  pDownloadUrl);
        }
        else
        {
            m_pDownloadUrl[0] = '\0';
        }

        if( pUploadUrl )
        {
            NETTEST_SNPRINTF(m_pUploadUrl, sizeof(m_pUploadUrl), "%s",  pUploadUrl);
        }
        else
        {
            m_pUploadUrl[0] = '\0';
        }
    }

    // Destructor
    CurlHttpUploadDownload::~CurlHttpUploadDownload() NN_NOEXCEPT {}

    // AllocateModules
    bool CurlHttpUploadDownload::AllocateModules() NN_NOEXCEPT
    {
        // Allocate download module
        Modules::LibCurl::Params downloadParams(m_params);
        downloadParams.httpMethod = Modules::LibCurl::HttpMethod_Get;
        downloadParams.uploadSize = 0;

        strncpy(downloadParams.pUrl, m_pDownloadUrl, sizeof(downloadParams.pUrl) - 1);
        downloadParams.pUrl[sizeof(downloadParams.pUrl) - 1] = '\0';
        downloadParams.publishUpSpeedToTeamCity = false;
        m_pHttpDown = new Modules::LibCurl(downloadParams);
        if( !m_pHttpDown )
        {
            Log(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }

        // Allocate upload module
        Modules::LibCurl::Params uploadParams(m_params);
        uploadParams.httpMethod = Modules::LibCurl::HttpMethod_Put;
        uploadParams.expectedHash = MD5Hash::Result();
        strncpy(uploadParams.pUrl, m_pUploadUrl, sizeof(uploadParams.pUrl) - 1);
        uploadParams.pUrl[sizeof(uploadParams.pUrl) - 1] = '\0';
        uploadParams.publishDownSpeedToTeamCity = false;
        m_pHttpUp = new Modules::LibCurl(uploadParams);
        if( !m_pHttpUp )
        {
            Log(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }

        return true;
    }

    // Init
    bool CurlHttpUploadDownload::Init() NN_NOEXCEPT
    {
        nn::Result result = nn::fs::MountHostRoot();
        if( result.IsFailure() )
        {
            Log(" * Warning: Failed to mount host file system!\n\n");
        }

        return true;
    }

    // Cleanup
    bool CurlHttpUploadDownload::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        nn::fs::UnmountHostRoot();

        return true;
    }

    // FreeModules
    void CurlHttpUploadDownload::FreeModules() NN_NOEXCEPT
    {
        if( m_pHttpDown )
        {
            delete m_pHttpDown;
            m_pHttpDown = nullptr;
        }

        if( m_pHttpUp )
        {
            delete m_pHttpUp;
            m_pHttpUp = nullptr;
        }
    }

    // Config
    bool CurlHttpUploadDownload::Config() NN_NOEXCEPT
    {
        if(!AllocateModules())
        {
            return false;
        }

        TestThread* pThread1 = CreateTestThread("DownloadThread", m_timeoutMs);
        if( !pThread1 )
        {
            return false;
        }

        pThread1->AddModule(*m_pHttpDown);

        TestThread* pThread2 = CreateTestThread("UploadThread", m_timeoutMs);
        if( !pThread2 )
        {
            return false;
        }

        pThread2->AddModule(*m_pHttpUp);

        return true;
    }

} // namespace Tests
} // namespace NATF
