﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Modules/SoHttpDownloadModule.h"

#include <cstdio>
#include <cstdlib>
#include <cstring>

//#define LOG_RESOURCE

namespace
{
    const unsigned         BufferSize    = 1024 * 2;
    const char* const      g_pHttpCmd    = "GET";
    const char* const      g_pHttpVers   = "HTTP/1.0";
    const uint32_t         PollTimeoutMs = 10000;
} // un-named namespace

namespace NATF {
namespace Modules {

    // SendHttpRequest
    bool SoHttpDownload::SendHttpRequest(int socketFd, const char* pResource) const NN_NOEXCEPT
    {
        bool isSuccess = true;
        unsigned totalWritten;
        static const unsigned MaxRequestSize = 1024;
        char pGetRequest[MaxRequestSize];

        // Copy http request into write buffer.
        NETTEST_SNPRINTF(pGetRequest, MaxRequestSize, "%s %s %s\r\n\r\n", g_pHttpCmd, pResource, g_pHttpVers);
        size_t requestSize = strlen(pGetRequest);
        Log("HTTP Request: %s", pGetRequest);

        // While sending a buffer, only a portion may have successfully sent.
        // You may have to call the write funtion a few times to send any remaining data.
        totalWritten = 0;
        do
        {
            ssize_t rval;

            Log("Writing data...\n");
            ssize_t nBytesWritten = rval = NetTest::Send(socketFd, &pGetRequest[totalWritten], (unsigned)requestSize - totalWritten, nn::socket::MsgFlag::Msg_None);
            if( rval <= 0 )
            {
                LogError("Error: Send: rval: %d\n", (int)rval);
                isSuccess = false;
                goto out;
            }
            else // Write was successful(or at least part), keep track of how much has been sent.
            {
                totalWritten += (unsigned)nBytesWritten;
            }
        } while( totalWritten < (unsigned)requestSize );

    out:

        return isSuccess;
    }

    // CreateSocketWithOptions
    int SoHttpDownload::CreateSocketWithOptions() const NN_NOEXCEPT
    {
        int  socketFd  = -1;
        int  rval      = -1;
        bool doCleanup = false;

        // Create socket for tcp connection.
        socketFd = NetTest::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);

        if(socketFd >= 0)
        {
            if(m_IsDoNnLinger)
            {
                Log("Applying nn::socket::Option::So_Nn_Linger.\n");
                nn::socket::Linger lingerOpt = {true, 1};
                rval = NetTest::SetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Nn_Linger, &lingerOpt, sizeof(lingerOpt));
                if (rval < 0)
                {
                    Log("nn::socket::Option::So_Linger option not accepted with rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
                }
            }
        }
        else
        {
            LogError("Error: Failed to create client socket. rval: %d, SOErr:%d\n", socketFd, NetTest::GetLastError());
            doCleanup = true;
            goto out;
        }

    out:

        if( doCleanup )
        {
            if( socketFd >= 0 )
            {
                rval = NetTest::Close(socketFd);
                if( rval != 0 )
                {
                    LogError("Error: Failed to close socket. rval: %d, SOErr: %d\n", rval, NetTest::GetLastError());
                }
                else
                {
                    socketFd = -1;
                }
            }
        }

        return socketFd;
    }

    // ConnectToServer
    bool SoHttpDownload::ConnectToServer(int socketFd, const char* pServerIp, unsigned short portNum) const NN_NOEXCEPT
    {
        bool isSuccess   = true;
        int  rval        = -1;
        NetTest::SockAddrIn serverAddr;

        memset(&serverAddr, 0, sizeof(serverAddr));

        if( !pServerIp )
        {
            LogError("Error: pServerIp is NULL\n");
            isSuccess = false;
            goto out;
        }

        rval = NetTest::InetAddrPton(nn::socket::Family::Af_Inet, pServerIp, &serverAddr.sin_addr);
        if( !rval )
        {
            LogError("Error: pServerIp could not be converted into ip address!\n");
            isSuccess = false;
            goto out;
        }

        // Initialize server address struct
        serverAddr.sin_family = nn::socket::Family::Af_Inet;
        serverAddr.sin_port   = NetTest::Htons(portNum);

        // Print server ip.
        Log("Connecting to %s\n", pServerIp);

        rval = NetTest::Connect(socketFd, (NetTest::SockAddr *)&serverAddr, sizeof(serverAddr));
        if( rval < 0 )
        {
            LogError("Error: Connect: rval: %d SOErr: %d\n", rval, NetTest::GetLastError());
            isSuccess = false;
            goto out;
        }

    out:

        return isSuccess;
    }

    // ParseHeader
    char* SoHttpDownload::ParseHeader(char* pBuffer, unsigned bufSize, bool& isLastNewLine, bool& isHeaderComplete) const NN_NOEXCEPT
    {
        if( isLastNewLine && pBuffer[0] == '\n' )
        {
            isHeaderComplete = true;

            if( bufSize > 1 )
            {
                return &pBuffer[1];
            }
            else
            {
                return NULL;
            }
        }

        unsigned i = 0;
        unsigned newLinesCount = 0;
        for(; i < bufSize; ++i)
        {
            if( pBuffer[i] == '\n' || pBuffer[i] == '\r' )
            {
                ++newLinesCount;
                if( newLinesCount >= 4 )
                {
                    isHeaderComplete = true;
                    if( i >= bufSize - 1 )
                    {
                        return NULL;
                    }
                    else
                    {
                        return &pBuffer[i + 1];
                    }
                }
            }
            else
            {
                newLinesCount = 0;
            }
        }

        if( newLinesCount )
        {
            isLastNewLine = true;
        }

        return NULL;
    }

    // ReceiveResponse
    bool SoHttpDownload::ReceiveResponse(int socketFd, MD5Hash::Result& hashResult) const NN_NOEXCEPT
    {
        int rval;
        bool isSuccess        = true;
        unsigned totalRead    = 0;
        char pBuffer[BufferSize + 1];
        bool isHeaderComplete = false;
        bool isLastNewLine    = false;
        bool hasPollTimedout  = false;
        MD5Hash md5Hash;
        float throughput;
        NetTest::Tick start, end;
        NetTest::Time duration;
        int milSec;
        NetTest::SockLen paramLen;

        int value = 1024 * 64 - 1;
        rval = NetTest::SetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, sizeof(value));
        if (rval < 0)
        {
            Log("WARNING: Failed to set recv buffer! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }

        paramLen = sizeof(int);
        rval = NetTest::GetSockOpt(socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, &paramLen);
        if (rval < 0)
        {
            Log("WARNING: Failed to get recv buffer! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }
        else
        {
            Log("nn::socket::Option::So_RcvBuf: %d\n", value);
        }

        NetTest::PollFd fdPoll;
        Log("Socket Fd: %d\n\n", socketFd);
        fdPoll.fd = socketFd;
        fdPoll.events = nn::socket::PollEvent::PollRdNorm;
        fdPoll.revents = nn::socket::PollEvent::PollNone;

        // Read the response from server
        Log("Reading data...\n");
        start = NetTest::GetTick();
        do
        {
            fdPoll.revents = nn::socket::PollEvent::PollNone;
            // Wait for socket to be readable
            rval = SelectOrPoll(&fdPoll, 1, PollTimeoutMs);
            if( rval < 0 )
            {
                LogError("Error: Socket FD: %d, Poll: rval: %d errno: %d\n", fdPoll.fd, rval, NetTest::GetLastError());
                isSuccess = false;
                goto out;
            }
            else if( rval == 0 )
            {
                Log("Poll: Timed out!\n");
                hasPollTimedout = true;
                break;
            }
            else if( !(fdPoll.revents & nn::socket::PollEvent::PollRdNorm) )
            {
                LogError("Warning: Poll: Socket not readable after poll! rval: %d errno: %d\n", rval, NetTest::GetLastError());
            }

            // Socket is now readable, read the data.
            ssize_t bytesRead = rval = (int)NetTest::Recv(socketFd, pBuffer, BufferSize, nn::socket::MsgFlag::Msg_None);
            if( rval == 0 )
            {
                Log("Recv: Connection has been closed. rval: %d\n", rval);
                break;
            } else if( rval < 0 )
            {
                LogError("Error: Recv: rval: %d errno: %d\n", rval, NetTest::GetLastError());
                isSuccess = false;
                goto out;
            }

            // If bytes were read
            if( bytesRead > 0 )
            {
                totalRead += (unsigned)bytesRead;
                char* pData = pBuffer;
                if( !isHeaderComplete )
                {
                    Log("Parsing header!\n");
                    pData = ParseHeader(pBuffer, (unsigned)bytesRead, isLastNewLine, isHeaderComplete);
                    if( isHeaderComplete )
                    {
                        Log("Header download complete!\n");
                    }
                }

                if( pData )
                {
                    // Continue calculating the MD5 hash value.
                    md5Hash.Update((unsigned char*)pData, static_cast<unsigned>(bytesRead - (pData - pBuffer)));
                }

                // Print read bytes.
    #ifdef LOG_RESOURCE
                NN_LOG("%.*s", (int)bytesRead, pBuffer);
    #endif
            }

        } while( rval > 0 );

        end = NetTest::GetTick();
        duration = NetTest::TickToTime(end - start);

        Log("Finalizing MD5 Hash...\n");
        md5Hash.Final(hashResult);

        // Calculate throughput
        //  Don't count poll timeout against us.
        milSec = static_cast<int>(duration.GetMilliSeconds()) - static_cast<int>(hasPollTimedout) * PollTimeoutMs;
        throughput = totalRead * 8.0f / (milSec / 1000.0f) / 1000000.0f;
        Log("Total bytes: %d, Throughput: %d.%d Mbits/Sec\n", static_cast<int>(totalRead),
                                                              static_cast<int>(throughput),
                                                              static_cast<int>(throughput * 100.0f) % 100 );

    out:

        return isSuccess;
    }

    // GetServerPage
    bool SoHttpDownload::GetServerPage(const char* pServerIp, unsigned short serverPort, const char* pResource, MD5Hash::Result& hashResult) const NN_NOEXCEPT
    {
        bool isSuccess = true;
        int rval       = -1;
        int socketFd   = -1;

        if( !pResource || pResource[0] == '\0' )
        {
            pResource = "/";
        }

        // Create socket with custom options
        socketFd = CreateSocketWithOptions();
        NN_LOG("\nSocket: %d\n\n", socketFd);
        if( socketFd < 0 )
        {
            isSuccess = false;
            goto out;
        }

        // Connect socket to server
        if( !ConnectToServer(socketFd, pServerIp, serverPort) )
        {
            isSuccess = false;
            goto out;
        }

        // Send HTTP request through tcp connection
        if( !SendHttpRequest(socketFd, pResource) )
        {
            isSuccess = false;
            goto out;
        }

        // Receive response from our HTTP request
        if( !ReceiveResponse(socketFd, hashResult) )
        {
            isSuccess = false;
            goto out;
        }

    out:

        if( socketFd >= 0 )
        {
            // Close socket
            rval = NetTest::Close(socketFd);
            if( rval != 0 )
            {
                Log("Warning: Failed to close socket. rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
            }
            else
            {
                socketFd = -1;
            }
        }

        return isSuccess;
    }

    // Constructor
    SoHttpDownload::SoHttpDownload(const char* pIp, unsigned short port, const char* pResource, const MD5Hash::Result& expectedHash, bool useSelect, bool doNnLinger) NN_NOEXCEPT
        :
        BaseModule(useSelect),
        m_pIp(pIp),
        m_port(port),
        m_pResource(pResource),
        m_expectedHash(expectedHash),
        m_IsDoNnLinger(doNnLinger)
        {}

    // Run
    bool SoHttpDownload::Run() NN_NOEXCEPT
    {
        if( !GetServerPage(m_pIp, m_port, m_pResource, m_hashResult) )
        {
            return false;
        }

        Log("Hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n"
            , m_hashResult.m_pHash[0]
            , m_hashResult.m_pHash[1]
            , m_hashResult.m_pHash[2]
            , m_hashResult.m_pHash[3]
            , m_hashResult.m_pHash[4]
            , m_hashResult.m_pHash[5]
            , m_hashResult.m_pHash[6]
            , m_hashResult.m_pHash[7]
            , m_hashResult.m_pHash[8]
            , m_hashResult.m_pHash[9]
            , m_hashResult.m_pHash[10]
            , m_hashResult.m_pHash[11]
            , m_hashResult.m_pHash[12]
            , m_hashResult.m_pHash[13]
            , m_hashResult.m_pHash[14]
            , m_hashResult.m_pHash[15] );

        if( memcmp(m_hashResult.m_pHash, m_expectedHash.m_pHash, sizeof(m_hashResult.m_pHash)) == 0 )
        {
            Log( "MD5 has matches as expected.\n" );
            return true;
        }
        else
        {
            Log( " Error: MD5 hash does NOT match expected.\n\n" );
            return false;
        }
    }

    // GetName
    const char* SoHttpDownload::GetName() const NN_NOEXCEPT
    { return "SoHttpDownload"; }
}
}
