﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Modules/ResolverApiModule.h"

//#define SHOW_DEBUG_LOGS

namespace
{
    const int g_gaiResHistoryMax = 4096;

    int g_gaiResHistoryIndex = 0;
    nn::socket::AddrInfo *g_gaiResHistory[g_gaiResHistoryMax];
    nn::socket::AiErrno g_lastGaiRv = nn::socket::AiErrno::EAi_Success;
    nn::socket::AiErrno g_lastGniRv = nn::socket::AiErrno::EAi_Success;
    bool g_printTestDesc = true;
    bool g_freeGaiRes = true;
} // un-named namespace

namespace NATF {
namespace Modules {

    // Constructor
    ResolverApiModule::ResolverApiModule(
        int NumTests, const char **TestDesc, const argTestType *Host, const argTestType *Serv,
        nn::socket::AddrInfo *GaiHints, gaiResOverrideType *GaiResOverride,
        nn::socket::AiErrno GaiRvExpec, int FlagsIn, nn::socket::AiErrno GniRvExpec) NN_NOEXCEPT
    :   m_NumTests(NumTests), m_GaiRvExpec(GaiRvExpec), m_FlagsIn(FlagsIn), m_GniRvExpec(GniRvExpec),
        m_RunMode(RUN_MODE_DEFAULT)
    {
        // Set g_gaiResHistory
        for(int i = 0; i < ::g_gaiResHistoryMax; i++)
        {
            ::g_gaiResHistory[i] = nullptr;
        }

        // Set m_TestDesc, m_Host, m_Serv (deep copy to buffers)
        for(int i = 0; i < NumTests; i++)
        {
            strcpy(m_TestDesc[i], TestDesc[i]);

            // Set m_Host (in, out, expec)
            DeepCopyGniArgType(&m_Host[i].in, &Host[i].in, m_BufferHost[i].in);
            DeepCopyGniArgType(&m_Host[i].out, &Host[i].out, m_BufferHost[i].out);
            DeepCopyGniArgType(&m_Host[i].expec, &Host[i].expec, m_BufferHost[i].expec);

            // Set m_Serv (in, out, expec)
            DeepCopyGniArgType(&m_Serv[i].in, &Serv[i].in, m_BufferServ[i].in);
            DeepCopyGniArgType(&m_Serv[i].out, &Serv[i].out, m_BufferServ[i].out);
            DeepCopyGniArgType(&m_Serv[i].expec, &Serv[i].expec, m_BufferServ[i].expec);
        }

        // Set m_pGaiHints, m_BufferGaiHints (deep copy)
        if(nullptr == GaiHints)
        {
            m_pGaiHints = nullptr;
        }
        else
        {
            m_pGaiHints = &m_BufferGaiHints;
            memcpy(&m_BufferGaiHints, GaiHints, sizeof(nn::socket::AddrInfo));
        }

        // Set m_pGaiResOverride, m_BufferGaiResOverride (deep copy)
        if(nullptr == GaiResOverride)
        {
            m_pGaiResOverride = nullptr;
        }
        else
        {
            m_pGaiResOverride = &m_BufferGaiResOverride;
            memcpy(&m_BufferGaiResOverride, GaiResOverride, sizeof(nn::socket::AddrInfo));
        }
    }

    // Run
    bool ResolverApiModule::Run() NN_NOEXCEPT
    {
        bool isSuccess = false;

        if(RUN_MODE_DEFAULT == m_RunMode)
        {
            isSuccess = RunTestScenario();
        }
        else if(RUN_MODE_UNTIL_EAI_MEMORY == m_RunMode)
        {
            isSuccess = RunTestScenarioUntilEaiMemory();
        }

        return isSuccess;
    }

    // GetName
    const char* ResolverApiModule::GetName() const NN_NOEXCEPT
    {
        return "ResolverApiModule";
    }

    // RunTestScenario
    bool ResolverApiModule::RunTestScenario() NN_NOEXCEPT
    {
        const size_t numArgs = 2;
        const int HOST = 0;
        const int SERV = 1;
        const char* desc[numArgs] = { "m_Host", "m_Serv" };
        bool isSuccess = true;

        for(int j = 0; j < m_NumTests; j++)
        {
            const gniArgType *argsIn[numArgs] = { &m_Host[j].in, &m_Serv[j].in };
            const gniArgType *argsOut[numArgs] = { &m_Host[j].out, &m_Serv[j].out };
            const gniArgType *argsExpec[numArgs] = { &m_Host[j].expec, &m_Serv[j].expec };
            nn::socket::AddrInfo *gaiRes = nullptr;
            char argBufferEmpty[MAX_CHARS_IN_STR];
            bool isSuccessInLoop = true;

            // Check whether print test description is enabled
            if(true == ::g_printTestDesc)
            {
                Log("%s\n", m_TestDesc[j]);
            }

            // Validate buffer size
            for(int i = 0; i < numArgs; ++i)
            {
                if(nullptr != argsOut[i]->ptr
                    && (argsOut[i]->len < 0 || argsOut[i]->len > MAX_CHARS_IN_STR))
                {
                    Log("buffer size invalid (%d), expected from %d to %d\n",
                        argsOut[i]->len, 0, MAX_CHARS_IN_STR);
                    isSuccessInLoop = false; // ASSERT_TRUE fails
                    goto bailInLoop;
                }
            }

#ifdef SHOW_DEBUG_LOGS
            // DEBUG
            LogParamsGetAddrInfo(argsIn[HOST]->ptr, argsIn[SERV]->ptr,
                m_pGaiHints, &gaiRes);
#endif

            // Validate nn::socket::GetAddrInfo return value
            // Note: This section allocates memory for gaiRes in nn::socket::GetAddrInfo
            if((::g_lastGaiRv = nn::socket::GetAddrInfo(argsIn[HOST]->ptr, argsIn[SERV]->ptr,
                m_pGaiHints, &gaiRes)) != m_GaiRvExpec)
            {
                char logMessage[256] = "nn::socket::GetAddrInfo: ";
                sprintf(logMessage + strlen(logMessage), "returned %s (%d), ",
                    nn::socket::GAIStrError(::g_lastGaiRv), ::g_lastGaiRv);
                sprintf(logMessage + strlen(logMessage), "expected %s (%d)\n",
                    nn::socket::GAIStrError(m_GaiRvExpec), m_GaiRvExpec);
                Log(logMessage);
                isSuccessInLoop = false; // ADD_FAILURE
            }

#ifdef SHOW_DEBUG_LOGS
            // DEBUG
            LogResultsGetAddrInfo(&gaiRes, ::g_lastGaiRv, m_GaiRvExpec);
#endif

            // Override nn::socket::GetAddrInfo result to inject test inputs
            OverrideGaiRes(gaiRes);

            // Traverse on linked list of matching network addresses
            memset(argBufferEmpty, 0xFF, MAX_CHARS_IN_STR);
            for(nn::socket::AddrInfo *p = gaiRes; p != nullptr; p = p->ai_next)
            {
                // Clear m_Host and m_Serv names
                for(int i = 0; i < numArgs; ++i)
                {
                    if(nullptr != argsOut[i]->ptr && 0 != argsOut[i]->len)
                    {
                        memcpy(argsOut[i]->ptr, argBufferEmpty, argsOut[i]->len);
                        if(nullptr != memchr(argsOut[i]->ptr, '\0', argsOut[i]->len))
                        {
                            isSuccessInLoop = false; // ASSERT_TRUE fails
                            goto bailInLoop;
                        }
                    }
                }

#ifdef SHOW_DEBUG_LOGS
                // DEBUG
                LogParamsGetNameInfo(p->ai_addr, p->ai_addrlen,
                    argsOut[HOST]->ptr, argsOut[HOST]->len,
                    argsOut[SERV]->ptr, argsOut[SERV]->len, m_FlagsIn);
#endif

                // Validate getnaminfo return value
                if((::g_lastGniRv = nn::socket::GetNameInfo(p->ai_addr, p->ai_addrlen,
                    argsOut[HOST]->ptr, argsOut[HOST]->len,
                    argsOut[SERV]->ptr, argsOut[SERV]->len, static_cast<nn::socket::NameInfoFlag>(m_FlagsIn))) != m_GniRvExpec)
                {
                    char logMessage[256] = "nn::socket::GetNameInfo: ";
                    sprintf(logMessage + strlen(logMessage), "returned %s (%d), ",
                        nn::socket::GAIStrError(::g_lastGniRv), ::g_lastGniRv);
                    sprintf(logMessage + strlen(logMessage), "expected %s (%d)\n",
                        nn::socket::GAIStrError(m_GniRvExpec), m_GniRvExpec);
                    Log(logMessage);
                    isSuccessInLoop = false; // ADD_FAILURE
                }

#ifdef SHOW_DEBUG_LOGS
                // DEBUG
                LogResultsGetNameInfo(argsOut[HOST]->ptr, argsOut[SERV]->ptr,
                    ::g_lastGniRv, m_GniRvExpec);
#endif

                // Verify m_Host and m_Serv names will be set when required
                for(int i = 0; i < numArgs; ++i)
                {
                    if(nullptr != argsOut[i]->ptr && 0 == argsOut[i]->len
                        && nullptr != argsExpec[i]->ptr && 0 != argsExpec[i]->len
                        && 0 != memcmp(argsOut[i]->ptr, argBufferEmpty, argsOut[i]->len))
                    {
                        Log("nn::socket::GetNameInfo %s: returned %s, expected nothing\n",
                            desc[i], argsOut[i]->ptr);
                        isSuccessInLoop = false; // ADD_FAILURE
                    }
                    if(nullptr != argsOut[i]->ptr && 0 != argsOut[i]->len
                        && nullptr != argsExpec[i]->ptr && 0 != argsExpec[i]->len
                        && 0 == memcmp(argsOut[i]->ptr, argBufferEmpty, argsOut[i]->len))
                    {
                        Log("nn::socket::GetNameInfo %s: returned nothing, expected %s\n",
                            desc[i], argsExpec[i]->ptr);
                        isSuccessInLoop = false; // ADD_FAILURE
                    }
                    if(nullptr != argsOut[i]->ptr && 0 != argsOut[i]->len
                        && nullptr != argsExpec[i]->ptr && 0 != argsExpec[i]->len
                        && nullptr == strstr(argsOut[i]->ptr, argsExpec[i]->ptr))
                    {
                        Log("%s: nn::socket::GetNameInfo %s: returned %s, expected %s\n",
                            desc[i], argsOut[i]->ptr, argsExpec[i]->ptr);
                        isSuccessInLoop = false; // ADD_FAILURE
                    }
                }
            }

bailInLoop:
            // Free allocated memory
            isSuccess &= isSuccessInLoop;
            if(nullptr != gaiRes
                && false == FreeOrStoreGaiRes(gaiRes, ::g_freeGaiRes))
            {
                Log("FreeOrStoreGaiRes: reached max results in history (%d)\n",
                    ::g_gaiResHistoryMax);
                isSuccess = false;
                goto bail;
            }
        }

bail:
        // Log result to indicate test scenario success
        bool logResultsEnabled = ::g_freeGaiRes;
        if(true == logResultsEnabled)
        {
            if(true == isSuccess)
            {
                Log("TEST PASSED\n\n");
            }
            else
            {
                Log("TEST FAILED\n\n");
            }
        }

        return isSuccess;
    } // NOLINT(impl/function_size)

    // RunTestScenarioUntilError
    bool ResolverApiModule::RunTestScenarioUntilEaiMemory() NN_NOEXCEPT
    {
        bool isSuccess = false;
        ::g_freeGaiRes = false; // set false for this test
        for(int i = 0; i < ::g_gaiResHistoryMax; i++)
        {
            // Stop printing test description after first iteration
            if(i == 1)
            {
                ::g_printTestDesc = false; // set false for this test
            }

            // Run test scenario until nn::socket::GetAddrInfo return value is not 0
            if(false == RunTestScenario())
            {
                Log("Previous line mismatch does not indicate failure, see below ...\n");
                Log("Called GetAddrInfo() %d times with return value %s (%d)\n",
                    i + 1, nn::socket::GAIStrError(m_GaiRvExpec), m_GaiRvExpec);
                if(nn::socket::AiErrno::EAi_Memory == ::g_lastGaiRv)
                {
                    Log("Next call returned %s (%d)\n",
                        nn::socket::GAIStrError(nn::socket::AiErrno::EAi_Memory), nn::socket::AiErrno::EAi_Memory);
                    isSuccess = true;
                }
                else
                {
                    Log("Next call returned %s (%d), expected %s (%d)\n",
                        nn::socket::GAIStrError(::g_lastGaiRv), ::g_lastGaiRv,
                        nn::socket::GAIStrError(nn::socket::AiErrno::EAi_Memory), nn::socket::AiErrno::EAi_Memory);
                    isSuccess = false;
                }
                break;
            }
        }
        ::g_printTestDesc = true; // restore default for next test
        ::g_freeGaiRes = true; // restore default for this test
        FreeGaiResHistory();
        return isSuccess;
    }

    // SetRunMode
    void ResolverApiModule::SetRunMode(int RunMode) NN_NOEXCEPT
    {
        m_RunMode = RunMode;
    }

    // OverrideGaiRes
    void ResolverApiModule::OverrideGaiRes(nn::socket::AddrInfo *gaiRes) NN_NOEXCEPT
    {
        // Validate override pointer
        if(nullptr == m_pGaiResOverride)
        {
            return;
        }

        // Override nn::socket::GetAddrInfo result with injected data
        if(m_pGaiResOverride->flags & gaiResOverrideAiFlags)
        {
            gaiRes->ai_flags = m_pGaiResOverride->info.ai_flags;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiFamily)
        {
            gaiRes->ai_family = m_pGaiResOverride->info.ai_family;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiSocktype)
        {
            gaiRes->ai_socktype = m_pGaiResOverride->info.ai_socktype;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiProtocol)
        {
            gaiRes->ai_protocol = m_pGaiResOverride->info.ai_protocol;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiAddrlen)
        {
            gaiRes->ai_addrlen = m_pGaiResOverride->info.ai_addrlen;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiAddr)
        {
            gaiRes->ai_addr = m_pGaiResOverride->info.ai_addr;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiCanonname)
        {
            gaiRes->ai_canonname = m_pGaiResOverride->info.ai_canonname;
        }
        if(m_pGaiResOverride->flags & gaiResOverrideAiNext)
        {
            gaiRes->ai_next = m_pGaiResOverride->info.ai_next;
        }
    }

    // FreeOrStoreGaiRes
    bool ResolverApiModule::FreeOrStoreGaiRes(
        nn::socket::AddrInfo *gaiRes, bool freeGaiRes) NN_NOEXCEPT
    {
        // Validate input pointer
        if(nullptr == gaiRes)
        {
            return false;
        }

        // Free or store nn::socket::GetAddrInfo result
        bool isSuccess = true;
        if(true == freeGaiRes)
        {
            nn::socket::FreeAddrInfo(gaiRes);
        }
        else if(::g_gaiResHistoryIndex < ::g_gaiResHistoryMax)
        {
            g_gaiResHistory[::g_gaiResHistoryIndex++] = gaiRes;
        }
        else
        {
            isSuccess = false;
        }
        return isSuccess;
    }

    // FreeGaiResHistory
    bool ResolverApiModule::FreeGaiResHistory() NN_NOEXCEPT
    {
        // Validate current index
        if(::g_gaiResHistoryIndex >= ::g_gaiResHistoryMax)
        {
            return false;
        }

        // Free nn::socket::GetAddrInfo result history
        for(int i = 0; i <= ::g_gaiResHistoryIndex; i++)
        {
            nn::socket::FreeAddrInfo(::g_gaiResHistory[i]);
        }
        ::g_gaiResHistoryIndex = 0;
        return true;
    }

    // DeepCopyGniArgType
    void ResolverApiModule::DeepCopyGniArgType(gniArgType* dst,
        const gniArgType* src, char buf[])
    {
        dst->len = src->len;
        if(nullptr == src->ptr)
        {
            dst->ptr = nullptr;
        }
        else
        {
            dst->ptr = &buf[0];
            strcpy(&buf[0], src->ptr);
        }
    }

    // LogParamsGetAddrInfo
    void ResolverApiModule::LogParamsGetAddrInfo(const char* node,
        const char* service, nn::socket::AddrInfo* hints, nn::socket::AddrInfo** res)
    {
        Log("%s: GetAddrInfo WILL BE CALLED SOON ...\n", __FUNCTION__);
        Log("%s: node = %p\n", __FUNCTION__, node);
        Log("%s: node = %s\n", __FUNCTION__, node);
        Log("%s: service = %p\n", __FUNCTION__, service);
        Log("%s: service = %s\n", __FUNCTION__, service);
        Log("%s: hints = %p\n", __FUNCTION__, hints);
        Log("%s: hints->ai_flags = %d\n", __FUNCTION__, hints->ai_flags);
        Log("%s: hints->ai_family = %d\n", __FUNCTION__, hints->ai_family);
        Log("%s: hints->ai_socktype = %d\n", __FUNCTION__, hints->ai_socktype);
        Log("%s: hints->ai_protocol = %d\n", __FUNCTION__, hints->ai_protocol);
        Log("%s: hints->ai_addrlen = %d\n", __FUNCTION__, hints->ai_addrlen);
        Log("%s: hints->ai_canonname = %p\n", __FUNCTION__, hints->ai_canonname);
        Log("%s: hints->ai_canonname = %s\n", __FUNCTION__, hints->ai_canonname);
        Log("%s: hints->ai_addr = %p\n", __FUNCTION__, hints->ai_addr);
        Log("%s: hints->ai_next = %p\n", __FUNCTION__, hints->ai_next);
        Log("%s: res (out) = %p\n", __FUNCTION__, res);
    }

    // LogResultsGetAddrInfo
    void ResolverApiModule::LogResultsGetAddrInfo(nn::socket::AddrInfo **res,
        int gaiRv, int gaiRvExpec)
    {
        Log("%s: GetAddrInfo WAS JUST CALLED ...\n", __FUNCTION__);
        Log("%s: res = %p\n", __FUNCTION__, res);
        Log("%s: *res = %p\n", __FUNCTION__, *res);
        Log("%s: *res->ai_flags = %d\n", __FUNCTION__, (*res)->ai_flags);
        Log("%s: *res->ai_family = %d\n", __FUNCTION__, (*res)->ai_family);
        Log("%s: *res->ai_socktype = %d\n", __FUNCTION__, (*res)->ai_socktype);
        Log("%s: *res->ai_protocol = %d\n", __FUNCTION__, (*res)->ai_protocol);
        Log("%s: *res->ai_addrlen = %d\n", __FUNCTION__, (*res)->ai_addrlen);
        Log("%s: *res->ai_canonname = %p\n", __FUNCTION__, (*res)->ai_canonname);
        Log("%s: *res->ai_canonname = %s\n", __FUNCTION__, (*res)->ai_canonname);
        Log("%s: *res->ai_addr = %p\n", __FUNCTION__, (*res)->ai_addr);
        Log("%s: *res->ai_next = %p\n", __FUNCTION__, (*res)->ai_next);
        Log("%s: gaiRv = %d\n", __FUNCTION__, gaiRv);
        Log("%s: gaiRvExpec = %d\n", __FUNCTION__, gaiRvExpec);
    }

    // LogParamsGetNameInfo
    void ResolverApiModule::LogParamsGetNameInfo(nn::socket::SockAddr *sa, nn::socket::SockLenT salen,
        char* host, int hostLen,
        char* serv, int servLen, int flags)
    {
        Log("%s: GetNameInfo WILL BE CALLED SOON ...\n", __FUNCTION__);
        Log("%s: sa = %p\n", __FUNCTION__, sa);
        Log("%s: salen = %d\n", __FUNCTION__, salen);
        Log("%s: host (out) = %p\n", __FUNCTION__, host);
        Log("%s: host (out) = %s\n", __FUNCTION__, host);
        Log("%s: hostLen = %d\n", __FUNCTION__, hostLen);
        Log("%s: serv (out) = %p\n", __FUNCTION__, serv);
        Log("%s: serv (out) = %s\n", __FUNCTION__, serv);
        Log("%s: servLen = %d\n", __FUNCTION__, servLen);
        Log("%s: flags = %d\n", __FUNCTION__, flags);
    }

    // LogResultsGetNameInfo
    void ResolverApiModule::LogResultsGetNameInfo(char* host, char* serv,
        int gniRv, int gniRvExpec)
    {
        Log("%s: GetNameInfo WAS JUST CALLED ...\n", __FUNCTION__);
        Log("%s: host (out) = %p\n", __FUNCTION__, host);
        Log("%s: host (out) = %s\n", __FUNCTION__, host);
        Log("%s: serv (out) = %p\n", __FUNCTION__, serv);
        Log("%s: serv (out) = %s\n", __FUNCTION__, serv);
        Log("%s: gniRv = %d\n", __FUNCTION__, gniRv);
        Log("%s: gniRvExpec = %d\n", __FUNCTION__, gniRvExpec);
    }

} // namespace Modules
} // namespace NATF
