﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Modules/HttpServerModule_Win.h"
#include "Utils/md5.h"

#define MHD_PLATFORM_H
#include <microhttpd.h>

namespace
{
const char g_pKey[] =
"-----BEGIN RSA PRIVATE KEY-----\n"
"MIICXQIBAAKBgQDyqPxK+6Kusv+5vkVwuKce4zBwAQscPkibHVFKrWAdP4ARSLj6\n"
"XiI8gKeI0O2G75t6HrrYKaa1eg9GxJbsf+dc8aWv3tS0MNfAG9LgdvYukKsBkqrd\n"
"to2HJBrh/IC2y6mZt5/ZyLyWhyADyxQwzLMEw1vdAgj0zq0hzjh8w33OUwIDAQAB\n"
"AoGATyinHMLXwAuudmZGes4R3naYf4EnobADuW4n7ad/OitT2M+TYEFDA9+oR254\n"
"fdM+MelCZG+57WuC/j/1R/M1gq7AwtpCgCjyHxuUUeObBIXWOw6xxmMGV62LG7Wh\n"
"rpUHL54Xy6QKQ/KLnl5IFkIAgY1YWNOVBK7NhN27TQLNyYkCQQD+IqOu52hrQydT\n"
"spoNFycMrnvYOrwlBIhsdz9/3jSW1rVIYddBafHSpAhsxl+K/gvvwAzGKoCW1q3I\n"
"i0FCwxx3AkEA9HDKmnVqYol+9OSjGJcfKqws9pNGYU49hjYpeh2RDtsZWI7N2vcE\n"
"AuK8KCmTlm2rgogKq2yM9ZTECzsVeJPABQJBAKpQaqQf/7HK6oNEMcvOFWSBVww9\n"
"qLOiSw9uB4dZUqM4N3vP3mO/bco+FktX0tYilbtSWP1BpLXyH7tD4DOu8NcCQFNh\n"
"Qgrr8E0KX68DaorRycTRlEjLtd8tRBlK0l0gRsHNhNjLZXbCebDOTKZagEh4wl81\n"
"LoaI8ZA3naPDFLDbnU0CQQDIwh93toYxhOVvvgNGyCc+o2J5S9XLrs3H3aKBCVci\n"
"uMcOkA42R5fEELWH/VgN+L6m452mHcSP7eE53D0ITdmy\n"
"-----END RSA PRIVATE KEY-----";

const char g_pPem[] =
"-----BEGIN CERTIFICATE-----\n"
"MIIC/jCCAmegAwIBAgIJAJGPfSMvywYzMA0GCSqGSIb3DQEBBQUAMF4xCzAJBgNV\n"
"BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMRcw\n"
"FQYDVQQKEw5NeSBDb21wYW55IEx0ZDEPMA0GA1UEAxMGSG9zdFBDMB4XDTE2MDMx\n"
"MDE3MDM1NVoXDTE3MDMxMDE3MDM1NVowXjELMAkGA1UEBhMCVVMxEzARBgNVBAgT\n"
"Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxFzAVBgNVBAoTDk15IENvbXBh\n"
"bnkgTHRkMQ8wDQYDVQQDEwZIb3N0UEMwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJ\n"
"AoGBAPKo/Er7oq6y/7m+RXC4px7jMHABCxw+SJsdUUqtYB0/gBFIuPpeIjyAp4jQ\n"
"7Ybvm3oeutgpprV6D0bElux/51zxpa/e1LQw18Ab0uB29i6QqwGSqt22jYckGuH8\n"
"gLbLqZm3n9nIvJaHIAPLFDDMswTDW90CCPTOrSHOOHzDfc5TAgMBAAGjgcMwgcAw\n"
"HQYDVR0OBBYEFAhp+OYX8HRGaaRUS+zvVTCioGapMIGQBgNVHSMEgYgwgYWAFAhp\n"
"+OYX8HRGaaRUS+zvVTCioGapoWKkYDBeMQswCQYDVQQGEwJVUzETMBEGA1UECBMK\n"
"V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEXMBUGA1UEChMOTXkgQ29tcGFu\n"
"eSBMdGQxDzANBgNVBAMTBkhvc3RQQ4IJAJGPfSMvywYzMAwGA1UdEwQFMAMBAf8w\n"
"DQYJKoZIhvcNAQEFBQADgYEAD6JX6ktNv1OZIfkpVbIFIeBlp246nAFSDxeWg2kr\n"
"XCnti0EkACp9qiU3+U1r19oEAbxhwQBfTKdWKZidJASXQ0yw6izZUjaITM0nIIgl\n"
"xFaF5buA16n9HTaYCYk3NEwsIsr+AUrH/ADXqfXwJwfKU5100Mp9uCQSoCqw2KQG\n"
"ZpA=\n"
"-----END CERTIFICATE-----";


const char g_pClientAuth[] =
"-----BEGIN CERTIFICATE-----\n"
"MIID5jCCAs6gAwIBAgIJAKHkhxS6kkvxMA0GCSqGSIb3DQEBBQUAMFUxCzAJBgNV\n"
"BAYTAlVTMQswCQYDVQQIEwJXQTEQMA4GA1UEBxMHUmVkbW9uZDELMAkGA1UEChMC\n"
"TkExCzAJBgNVBAsTAk5BMQ0wCwYDVQQDEwRuYXRmMB4XDTE2MDYwODIxMTcyM1oX\n"
"DTI2MDYwNjIxMTcyM1owVTELMAkGA1UEBhMCVVMxCzAJBgNVBAgTAldBMRAwDgYD\n"
"VQQHEwdSZWRtb25kMQswCQYDVQQKEwJOQTELMAkGA1UECxMCTkExDTALBgNVBAMT\n"
"BG5hdGYwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDGHc7hin40ZzG0\n"
"0XZVMQx8LzEtXSNJUHWg3TqOsPj1g1HuSLg5fmoTt/FrGsWiqXF05VhTzc/LVPS9\n"
"JIxsyhrNP3nuj9zi0vg97sZuT9Bmwrnl0iIR+81EDpK+IH2iussIaFnw7TyrkW65\n"
"HX8//0Mt2vwCRQkFXMsKZI9UpNZel2NN6K4GeAjemXgCNNJ3ZbBNQ6FAYBglhrXh\n"
"lFqktQSFTdtBV2U00dmQrHJEWyAiq1jF27LywrvZOfWok01WyGGcQedXA7RvxZ/k\n"
"DZXqVv1rDdtsIsBEzgesfYnsDhM8fpGCyAf5/LwyaI6rtXidU39eJsMjBEapzbNl\n"
"1UepOonNAgMBAAGjgbgwgbUwHQYDVR0OBBYEFJnkri784bRZ4vKp3ONjp5s3KjjN\n"
"MIGFBgNVHSMEfjB8gBSZ5K4u/OG0WeLyqdzjY6ebNyo4zaFZpFcwVTELMAkGA1UE\n"
"BhMCVVMxCzAJBgNVBAgTAldBMRAwDgYDVQQHEwdSZWRtb25kMQswCQYDVQQKEwJO\n"
"QTELMAkGA1UECxMCTkExDTALBgNVBAMTBG5hdGaCCQCh5IcUupJL8TAMBgNVHRME\n"
"BTADAQH/MA0GCSqGSIb3DQEBBQUAA4IBAQBqrJQU3Yb0NV6t3pZJ4KYfZoVMl1CD\n"
"ZcieuTkfanHYmYBGvRyKzG8DVgLqJawEd5lj7UmFWhmO7FiBp7ionRkBuImhDbPm\n"
"C4C2M86cJ9gZzfWYX8NmKmK/OVAHIBjz6mdCSqcrGdYKaEEz5iN1ZyIwYBwQO34l\n"
"nAej3KTFnZjtAN1q9MvsAVwxRR005JQUZlWkAd05fv8b6JjWnKjWXGKcI+gZpCYH\n"
"T9+6aumJWFRI4PqY5F5hAP2gI44prOHkTb7wslyRN/pkql64HUOvfdb7QU9JgCIF\n"
"Jf5Nuj9Sf3w5jzbCxLfDIE3IqE8ht+GEi66Kk7FM8Cchuzeo5Q2Jn5oX\n"
"-----END CERTIFICATE-----";
}

namespace NATF {
namespace Modules {

class HttpServer::Connection
{
public:

    static const uint32_t MaxMethodLen = 8;

    Connection(HttpServer* pThis, const char* pMethod, size_t contentLength) NN_NOEXCEPT :
        m_pThis(pThis),
        m_contentLength(contentLength),
        m_bytesRead(0),
        m_postProcessor(nullptr)
    {
        NETTEST_SNPRINTF(m_pMethod, MaxMethodLen, "%s", pMethod);
    }

    ~Connection() NN_NOEXCEPT
    {
        if( m_postProcessor )
        {
            MHD_destroy_post_processor(m_postProcessor);
        }
    }

    bool Finalize(MHD_Connection *connection) NN_NOEXCEPT
    {
        if( strcmp(m_pMethod, "PUT") == 0 || strcmp(m_pMethod, "POST") == 0 )
        {
            m_hash.Final(m_hashResult);

            m_pThis->Log(" Upload complete!\n");
            m_pThis->Log("Hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n\n"
            , m_hashResult.m_pHash[0]
            , m_hashResult.m_pHash[1]
            , m_hashResult.m_pHash[2]
            , m_hashResult.m_pHash[3]
            , m_hashResult.m_pHash[4]
            , m_hashResult.m_pHash[5]
            , m_hashResult.m_pHash[6]
            , m_hashResult.m_pHash[7]
            , m_hashResult.m_pHash[8]
            , m_hashResult.m_pHash[9]
            , m_hashResult.m_pHash[10]
            , m_hashResult.m_pHash[11]
            , m_hashResult.m_pHash[12]
            , m_hashResult.m_pHash[13]
            , m_hashResult.m_pHash[14]
            , m_hashResult.m_pHash[15] );

            if( m_hashResult == m_expectedHashResult )
            {
                m_pThis->Log("MD5 hashes match!\n\n");
            }
            else
            {
                m_pThis->LogError("MD5 hashes miss-match\n\n");

                m_pThis->Log("Expected Hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n\n"
                , m_expectedHashResult.m_pHash[0]
                , m_expectedHashResult.m_pHash[1]
                , m_expectedHashResult.m_pHash[2]
                , m_expectedHashResult.m_pHash[3]
                , m_expectedHashResult.m_pHash[4]
                , m_expectedHashResult.m_pHash[5]
                , m_expectedHashResult.m_pHash[6]
                , m_expectedHashResult.m_pHash[7]
                , m_expectedHashResult.m_pHash[8]
                , m_expectedHashResult.m_pHash[9]
                , m_expectedHashResult.m_pHash[10]
                , m_expectedHashResult.m_pHash[11]
                , m_expectedHashResult.m_pHash[12]
                , m_expectedHashResult.m_pHash[13]
                , m_expectedHashResult.m_pHash[14]
                , m_expectedHashResult.m_pHash[15] );

                m_pThis->Fail();
                return false;
            }

            char pPage[256];
            NETTEST_SNPRINTF(pPage, sizeof(pPage), "<html><body><p>200 OK. Upload successful! MD5 Hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x</p></body></html>"
            , m_hashResult.m_pHash[0]
            , m_hashResult.m_pHash[1]
            , m_hashResult.m_pHash[2]
            , m_hashResult.m_pHash[3]
            , m_hashResult.m_pHash[4]
            , m_hashResult.m_pHash[5]
            , m_hashResult.m_pHash[6]
            , m_hashResult.m_pHash[7]
            , m_hashResult.m_pHash[8]
            , m_hashResult.m_pHash[9]
            , m_hashResult.m_pHash[10]
            , m_hashResult.m_pHash[11]
            , m_hashResult.m_pHash[12]
            , m_hashResult.m_pHash[13]
            , m_hashResult.m_pHash[14]
            , m_hashResult.m_pHash[15] );

            struct MHD_Response* pResponse = MHD_create_response_from_buffer(strlen(pPage), pPage, MHD_RESPMEM_MUST_COPY);
            if( nullptr == pResponse )
            {
                m_pThis->LogError(" * Failed to create response.\n\n");
                m_pThis->Fail();
                return false;
            }

            int ret = MHD_queue_response(connection, MHD_HTTP_OK, pResponse);
            MHD_destroy_response(pResponse);

            if( MHD_YES != ret )
            {
                m_pThis->LogError(" Failed to Respond to: %s Err: %d\n", m_pMethod, ret);
                m_pThis->Fail();
                return false;
            }
            else
            {
                m_pThis->Log(" Responding to: %s.\n", m_pMethod);
            }

            return true;
        }

        return true;
    }

    void HandlePutRequest(MHD_Connection *connection, const char *resource,
                              const char *method, const char *version,
                              const char *upload_data, size_t *upload_data_size) NN_NOEXCEPT
    {
        if( *upload_data_size )
        {
            m_hash.Update((const uint8_t*)upload_data, (unsigned int)(*upload_data_size));
            m_bytesRead += *upload_data_size;
            *upload_data_size = 0;
        }
    }

    static int PostIterator( void *cls,
                             enum MHD_ValueKind kind,
                             const char *key,
                             const char *filename,
                             const char *content_type,
                             const char *transfer_encoding,
                             const char *data,
                             uint64_t off,
                             size_t size) NN_NOEXCEPT
    {
        Connection* pThis = reinterpret_cast<Connection*>(cls);

        pThis->m_hash.Update((const uint8_t*)data, (unsigned int)size);
        pThis->m_bytesRead += size;
        return MHD_YES;
    }


    MD5Hash m_hash;
    MD5Hash::Result m_hashResult;
    MD5Hash::Result m_expectedHashResult;
    char m_pMethod[MaxMethodLen];
    HttpServer* m_pThis;
    size_t m_contentLength;
    size_t m_bytesRead;
    MHD_PostProcessor* m_postProcessor;
};

// Helper class
class HttpServer::Helper
{
public:

    struct ResponseData
    {
        HttpServer* pThis;
        int32_t totalSize;
        int32_t bytesSent;
        int32_t percent10Done;
    };

    static void RequestCompletedCallback(void *cls,
                                         struct MHD_Connection *connection,
                                         void **con_cls,
                                         enum MHD_RequestTerminationCode toe) NN_NOEXCEPT
    {
        if( !con_cls || !*con_cls )
        {
            NN_LOG("RequestCompletedCallback: User connection data already destroyed.\n");
            return;
        }

        Connection* pConnection = reinterpret_cast<Connection*>(*con_cls);
        pConnection->Finalize(connection);

        *con_cls = nullptr;
        delete pConnection;
    }

    // Called whenever the download needs more data to send.
    static ssize_t GetResponseCallback(void *cls, int64_t pos, char* pBuffer, size_t maxWrite) NN_NOEXCEPT
    {
        ResponseData* pResponseData = (ResponseData*)cls;

        char val = static_cast<char>(pos);
        for(uint32_t i = 0; i < maxWrite; ++i)
        {
            pBuffer[i] = val;
            ++val;
        }

        pResponseData->bytesSent += static_cast<int32_t>(maxWrite);
        int32_t percent10Done = static_cast<int32_t>((float)pResponseData->bytesSent / (float)pResponseData->totalSize * 10.0f);

        // Have we completed at least another 10 percent?
        if( percent10Done > pResponseData->percent10Done )
        {
            pResponseData->percent10Done = percent10Done;
            pResponseData->pThis->Log(" Sent: %d%%\n", pResponseData->percent10Done * 10);
        }

        return maxWrite;
    }

    static void ReaderFreeCallback(void *cls) NN_NOEXCEPT
    {
        delete ((ResponseData*)cls);
    }

    static int HandleUploadRequest(MHD_Connection *connection, const char *upload_data, size_t *upload_data_size, void **con_cls, const char *resource,
                              const char *method, const char *version, HttpServer* pThis) NN_NOEXCEPT
    {
        if( nullptr == con_cls )
        {
            pThis->LogError(" con_cls is null!\n\n");
            pThis->Fail();
            return MHD_NO;
        }

        if( *con_cls )
        {
            size_t initialDataSize = *upload_data_size;
            Connection* pConnection = reinterpret_cast<Connection*>(*con_cls);
            pThis->Log("BytesRead: %d, upload: %d\n", pConnection->m_bytesRead, initialDataSize);

            if( pConnection->m_postProcessor )
            {
                MHD_post_process(pConnection->m_postProcessor, upload_data, *upload_data_size);
            }
            else
            {
                pConnection->HandlePutRequest(connection, resource, method, version, upload_data, upload_data_size);
            }

            *upload_data_size = 0;

            if( 0 == initialDataSize )
            {
                pConnection->Finalize(connection);

                *con_cls = nullptr;
                delete pConnection;
            }

            return MHD_YES;
        }
        else
        {
            unsigned n1, n2, n3, n4, n5, n6, n7, n8, n9, n10, n11, n12,n13, n14, n15, n16;

            pThis->Log("New upload session!\n");

            int rval = sscanf(resource, "/%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x", &n1,  &n2,  &n3,  &n4, &n5,  &n6,  &n7,  &n8, &n9,  &n10, &n11, &n12, &n13, &n14, &n15, &n16 );
            if( rval > 1 && rval != 16 )
            {
                pThis->LogError(" Error: Url for PUT method must be md5 hash. rval: %d, resource: %s Example Url: http://192.168.0.100/8e,53,46,38,38,ad,c8,59,87,3b,bb,1a,17,2e,1a,b1\n\n", rval, resource);
                pThis->Fail();

                return MHD_NO;
            }

            int contentLength = 0;
            const char* pContentLength = MHD_lookup_connection_value(connection, MHD_HEADER_KIND, MHD_HTTP_HEADER_CONTENT_LENGTH);
            if( pContentLength )
            {
                rval = sscanf(pContentLength, "%d", &contentLength);
                if( rval != 1 || contentLength <= 0 )
                {
                    pThis->LogError(" Error: Failed to parse Content-Length: in from header\n\n");
                    pThis->Fail();

                    return MHD_NO;
                }
            }

            Connection* pNewConnection = new Connection(pThis, method, static_cast<size_t>(contentLength));
            if( nullptr == pNewConnection )
            {
                pThis->LogError(" * Failed to allocate new connection.\n\n");
                pThis->Fail();
                return MHD_NO;
            }

            pNewConnection->m_expectedHashResult.Set((unsigned char)n1, (unsigned char)n2,  (unsigned char)n3,  (unsigned char)n4,  (unsigned char)n5,  (unsigned char)n6,  (unsigned char)n7,  (unsigned char)n8,
                                                     (unsigned char)n9, (unsigned char)n10, (unsigned char)n11, (unsigned char)n12, (unsigned char)n13, (unsigned char)n14, (unsigned char)n15, (unsigned char)n16);

            if( strcmp(method, "POST") == 0 )
            {
                pNewConnection->m_postProcessor = MHD_create_post_processor(connection,  64 * 1024, Connection::PostIterator, pNewConnection);
                if( !pNewConnection->m_postProcessor )
                {
                    delete pNewConnection;
                    pThis->LogError(" * Failed to create POST processor.\n\n");
                    pThis->Fail();
                    return MHD_NO;
                }

                MHD_post_process(pNewConnection->m_postProcessor, upload_data, *upload_data_size);
            }
            else
            {
                if( *upload_data_size )
                {
                    pNewConnection->HandlePutRequest(connection, resource, method, version, upload_data, upload_data_size);
                }
            }

            *upload_data_size = 0;
            *con_cls = pNewConnection;

            return MHD_YES;
        }
    }

    static int HandleGetRequest(MHD_Connection *connection, const char *resource,
                              const char *method, const char *version, HttpServer* pThis) NN_NOEXCEPT
    {
        struct MHD_Response* pResponse;
        int ret;
        int pageSize = 0;

        int rval = sscanf(resource, "/kb%d", &pageSize);
        if( rval == 1 )
        {
            pageSize *= 1024;
        }
        else
        {
            rval = sscanf(resource, "/mb%d", &pageSize);
            if( rval == 1 )
            {
                pageSize *= 1024 * 1024;
            }
            else
            {
                pThis->LogError(" * Failed to parse resource: %s. Examples: /kb20 or /mb100\n\n", resource);
                const char* pPage = "<html><body><p>Error: 404. Page not found. Examples: /kb20 or /mb100</p></body></html>";
                pResponse = MHD_create_response_from_buffer(strlen(pPage), (void*)pPage, MHD_RESPMEM_PERSISTENT);
                if( nullptr == pResponse )
                {
                    pThis->LogError(" * Failed to create response.\n\n");
                    return MHD_NO;
                }

                ret = MHD_queue_response(connection, MHD_HTTP_NOT_FOUND, pResponse);
                MHD_destroy_response(pResponse);

                if( MHD_YES != ret )
                {
                    pThis->Log(" Failed to Respond to: %s %s %s\n", method, resource, version);
                }

                return ret;
            }
        }

        ResponseData* pResponseData = new ResponseData;
        if( nullptr == pResponseData )
        {
            pThis->LogError(" * Failed to allocate response data.\n\n");
            return MHD_NO;
        }

        memset(pResponseData, 0, sizeof(*pResponseData));
        pResponseData->pThis = pThis;
        pResponseData->totalSize = pageSize;

        pResponse = MHD_create_response_from_callback(pageSize, 1024, (MHD_ContentReaderCallback)&Helper::GetResponseCallback, pResponseData, &Helper::ReaderFreeCallback);
        if( nullptr == pResponse )
        {
            pThis->LogError(" * Failed to create response.\n\n");
            delete pResponseData;
            return MHD_NO;
        }

        ret = MHD_queue_response(connection, MHD_HTTP_OK, pResponse);
        MHD_destroy_response(pResponse);

        if( MHD_YES == ret )
        {
            pThis->Log(" Responding to: %s %s %s\n", method, resource, version);
        }
        else
        {
            pThis->LogError(" Failed to respond to: %s %s %s\n", method, resource, version);
            delete pResponseData;
        }

        return ret;
    }

    // Called when there is a connection request
    static int AnswerConnectionRequest(void *cls, struct MHD_Connection *connection,
                              const char *resource,
                              const char *method, const char *version,
                              const char *upload_data,
                              size_t *upload_data_size, void **con_cls) NN_NOEXCEPT
    {
        HttpServer* pThis = static_cast<HttpServer*>(cls);

        const char* pLen = MHD_lookup_connection_value(connection, MHD_HEADER_KIND, MHD_HTTP_HEADER_CONTENT_LENGTH );

        if( upload_data_size )
        {
            pThis->Log("%s %s %s UploadLen: %d, ContLen: %s\n", method, resource, version, *upload_data_size, pLen);
        }
        else
        {
            pThis->Log("%s %s %s, ContLen: %s\n", method, resource, version, pLen);
        }

        if( strcmp(method, "GET") == 0 )
        {
            int rval = Helper::HandleGetRequest(connection, resource, method, version, pThis);
            if( rval != MHD_YES )
            {
                pThis->Fail();
            }
            return rval;
        }
        else if( strcmp(method, "PUT") == 0 || strcmp(method, "POST") == 0 )
        {
            return HandleUploadRequest(connection, upload_data, upload_data_size, con_cls, resource, method, version, pThis);
        }

        pThis->LogError(" Http method not supported! Method: %s\n\n", method);
        pThis->Fail();
        return MHD_NO;
    }
};

// Constructor
HttpServer::HttpServer(uint16_t port, uint32_t durationSec, bool useSsl, bool doAuthClient, bool quitOnError) NN_NOEXCEPT :
    m_pPageBuffer(nullptr),
    m_port(port),
    m_durationSec(durationSec),
    m_useSsl(useSsl),
    m_doAuthClient(doAuthClient),
    m_quitOnError(quitOnError),
    m_isSuccess(true)
{
    NetTest::InitEvent(&m_hasFailure);
}

// Destructor
HttpServer::~HttpServer() NN_NOEXCEPT
{
    NetTest::FinalizeEvent(&m_hasFailure);
}

// Fail
void HttpServer::Fail() NN_NOEXCEPT
{
    m_isSuccess = false;

    if(m_quitOnError)
    {
        NetTest::SignalEvent(&m_hasFailure);
    }
}

// Run
bool HttpServer::Run() NN_NOEXCEPT
{
    struct MHD_Daemon *pDaemon;

    Log(" Starting page daemon.\n");
    if( m_useSsl )
    {
        Log(" Using Ssl.\n");
        if( m_doAuthClient )
        {
            Log(" Using Client Authentication.\n");
            pDaemon = MHD_start_daemon(MHD_USE_SELECT_INTERNALLY | MHD_USE_THREAD_PER_CONNECTION | MHD_USE_SSL, m_port, nullptr, nullptr, &Helper::AnswerConnectionRequest, this, MHD_OPTION_HTTPS_MEM_KEY, g_pKey, MHD_OPTION_HTTPS_MEM_CERT, g_pPem, MHD_OPTION_HTTPS_PRIORITIES, "PERFORMANCE", MHD_OPTION_HTTPS_MEM_TRUST, g_pClientAuth, MHD_OPTION_NOTIFY_COMPLETED, Helper::RequestCompletedCallback, nullptr, MHD_OPTION_END);
        }
        else
        {
            pDaemon = MHD_start_daemon(MHD_USE_SELECT_INTERNALLY | MHD_USE_THREAD_PER_CONNECTION | MHD_USE_SSL, m_port, nullptr, nullptr, &Helper::AnswerConnectionRequest, this, MHD_OPTION_HTTPS_MEM_KEY, g_pKey, MHD_OPTION_HTTPS_MEM_CERT, g_pPem, MHD_OPTION_HTTPS_PRIORITIES, "PERFORMANCE", MHD_OPTION_NOTIFY_COMPLETED, Helper::RequestCompletedCallback, nullptr, MHD_OPTION_END);
        }
    }
    else
    {
        pDaemon = MHD_start_daemon(MHD_USE_SELECT_INTERNALLY | MHD_USE_THREAD_PER_CONNECTION, m_port, nullptr, nullptr, &Helper::AnswerConnectionRequest, this, MHD_OPTION_END);
    }

    if( nullptr == pDaemon )
    {
        LogError("Failed to start page daemon\n\n");
        return false;
    }

    // Wait for daemon to do it's stuff.
    Log(" Waiting out the duration: %d sec\n", m_durationSec);
    NetTest::Time waitFrameTimeout = NetTest::TimeFromMs(m_durationSec * 1000);
    NetTest::TimedWaitEvent(&m_hasFailure, waitFrameTimeout);

    MHD_stop_daemon(pDaemon);
    return m_isSuccess;
}

// GetName
const char* HttpServer::GetName() const NN_NOEXCEPT
{ return "HttpServer"; }

}} // Namespace NATF::Modules
