﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 HTCS Utility class implementation.  See HTCSUtil.h for class description.
 *---------------------------------------------------------------------------*/

#include <HTCSUtil.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include "expat.h"



// enable this for parser state machine debugging
#define PARSE_DEBUG( ... ) // { NN_LOG(##__VA_ARGS__ );   }


#define HTCS_ARRAY_SIZE( __a)  ( sizeof(__a)  / sizeof( (__a)[0] ) )

#define HTCS_COPY_CSTRING(__targetArray, __sourcePtr, __sourceSize )            \
        do                                                                      \
        {                                                                       \
            int __vsize =                                                       \
                  ( (__sourceSize) < ( HTCS_ARRAY_SIZE(__targetArray) - 1 ) ?   \
                     (__sourceSize) :                                           \
                     ( HTCS_ARRAY_SIZE(__targetArray) - 1 )                     \
                  );                                                            \
            strncpy( (__targetArray), (__sourcePtr), __vsize );                 \
            __targetArray[ __vsize ] = 0;                                       \
        } while(NN_STATIC_CONDITION(false))


#define _PARSERPTR(__x)  static_cast<XML_Parser>(__x)

const char *nnt::net::HTCSFactory::htcsFactorySessionName = "htcsFactorySession";

// class constructor - performs required init of xml parser and others
nnt::net::HTCSFactory::HTCSFactory( tics::CrossBar *pCrossBar,
                              const char *pLocalHost, const char *pLocalPort,
                              const char *pServerPortName, const char *pTicsSessionName ):
    m_control(this),
    m_pCrossBar(pCrossBar),
    m_bUserTerminationRequested(false),
    m_bControlConnected(false),
    m_bAutoTeardown (false),
    m_otherItemNesting(0),
    m_Parser(NULL)
{
    // we keep this out of the interface, but ensure here that sizes are compatible
    NN_STATIC_ASSERT(sizeof(XML_Parser) == sizeof(m_Parser));
    NN_STATIC_ASSERT(sizeof(XML_Error)  == sizeof(m_xmlStatus));

    HTCS_COPY_CSTRING( m_serverPortName, pServerPortName, PortInfoFieldSize);
    m_ticsSessionName[0] = 0;
    if( pTicsSessionName != NULL )
    {
        HTCS_COPY_CSTRING( m_ticsSessionName, pTicsSessionName, PortInfoFieldSize);
    }

    if( Clear() )
    {
        ::tics::TCPSlot* pTcpSlot = new ::tics::TCPSlot( "tcp", "bridge_client", pLocalHost, pLocalPort);
        pCrossBar->AddSlot(htcsFactorySessionName, pTcpSlot);

        pCrossBar->RegisterForSessionStart( htcsFactorySessionName, &m_control );
    }
}

nnt::net::HTCSFactory::~HTCSFactory()
{
    XML_ParserFree( _PARSERPTR(m_Parser) );
}

bool nnt::net::HTCSFactory::ParseErrorFound()
{
    return (m_xmlStatus == XML_STATUS_ERROR) ||
           (m_parseState == XmlParseState_HighLevelParseError);
}

bool nnt::net::HTCSFactory::ParseStateOk()
{
    return (m_xmlStatus != XML_STATUS_ERROR) &&
           (!m_bUserTerminationRequested) &&
           (m_parseState != XmlParseState_HighLevelParseError);
}



bool nnt::net::HTCSFactory::Clear()
{
    // do enough state clearing that the class can be used again
    m_parseState = XmlParseState_Initial;
    m_bParserFoundServer = false;
    m_timeoutMs = TimeoutNotSet;
    m_bTimeoutFound = false;
    m_timeoutEvent.Reset();

    if( m_Parser != NULL )
    {
        XML_ParserFree( _PARSERPTR(m_Parser) );
        m_Parser = NULL;
    }

    m_Parser = XML_ParserCreate(NULL);
    if( m_Parser != NULL )
    {
        const char *headingText = "<infinitedoc>";
        XML_Parser p = _PARSERPTR(m_Parser);
        XML_SetUserData            ( p, this);
        XML_SetElementHandler      ( p, XmlStartElement, XmlEndElement);
        XML_SetCharacterDataHandler( p, XmlCharacterData);
        m_xmlStatus = XML_Parse    ( p, headingText, (int)strlen(headingText), 0);
    }
    else
    {
        m_xmlStatus = XML_STATUS_ERROR;
    }
    return ParseStateOk();

}




void nnt::net::HTCSFactory::TriggerParseError(bool bLowLevel)
{
    if( ParseStateOk() )
    {
        if( bLowLevel )
        {
            XML_Error e =  XML_GetErrorCode( _PARSERPTR(m_Parser) );
            NN_LOG("ERROR: XML Parsing error found: [line=%d][%s]!\n",
                XML_GetCurrentLineNumber(_PARSERPTR(m_Parser) ), XML_ErrorString(e) );
            m_xmlStatus = XML_STATUS_ERROR;
        }
        else
        {
            NN_LOG("ERROR: Content Parsing error found: [line=%d][state=%d]!\n",
                XML_GetCurrentLineNumber(_PARSERPTR(m_Parser)), m_parseState   );
            m_parseState = XmlParseState_HighLevelParseError;
        }
        if( IsControlConnected() )
        {
            NN_LOG("Disconnecting control\n");
            m_control.BeginDetach(); // defer teardown til we connect
        }
        else // immediate teardown
        {
            Teardown();
            // not connected, so must cause break here
        }
    }
}

void nnt::net::HTCSFactory::SetTimeout( int timeoutMs )
{
    if( m_timeoutMs == TimeoutNotSet ) // if not set yet
    {
        ::tics::portability::thread* thread;
        thread = m_pCrossBar->create_thread( TimeoutThreadHook, this );
        if( thread == NULL)
        {
            NN_LOG("ERROR: Unable to create thread for timeout!\n" );
        }
        else
        {
            m_timeoutMs = timeoutMs;
        }
    }
}


void nnt::net::HTCSFactory::TimeoutThread( )
{
    if( !(m_timeoutEvent.Wait(m_timeoutMs)) )
    {
        NN_LOG("ERROR: Timeout detected!\n" );
        // must do this from the one true thread
        m_bTimeoutFound = true;
        m_pCrossBar->PostGenericMessage( TriggerTimeout, this );
    }
}

bool nnt::net::HTCSFactory::Parse( const char *userText, int userLen )
{
    // we will stop parsing forever if we encounter an error
    if( ParseStateOk() )
    {
        int len = userLen;
        if( len < 0 )
        {
            len = (int)strlen( userText );
        }
        if( XML_Parse(_PARSERPTR(m_Parser), userText, userLen, 0) == XML_STATUS_ERROR )
        {
            // transitioned into error
            TriggerParseError(true);
        }
        return ParseStateOk();
    }
    else
    {
        return false;
    }
}


void nnt::net::HTCSFactory::Teardown()
{
    PARSE_DEBUG("Teardown starting\n");
    if( ParseErrorFound() )
    {
        PARSE_DEBUG("Trigger parse error\n");
        OnParseError();
    }
    else if( m_bParserFoundServer && m_bAutoTeardown )
    {
        // deferred notification
        OnServerFound( (m_ticsSessionName[0] != 0) ? m_ticsSessionName : NULL ,
            m_currPortIP,
            m_currPort );
    }
    PARSE_DEBUG("Teardown ended\n");
}


void nnt::net::HTCSFactory::ControlSessionStart()
{
    m_bControlConnected = true;
}
void nnt::net::HTCSFactory::ControlSessionStop()
{
    if( m_bControlConnected )
    {
        m_bControlConnected = false;
        OnControlDisconnected();
        Teardown();
    }
}

void nnt::net::HTCSFactory::StopControlParsing()
{
    m_bUserTerminationRequested = true;
    if( IsControlConnected() && (!m_bUserTerminationRequested) )
    {
        m_control.BeginDetach();
    }
}


void nnt::net::HTCSFactory::XmlStartElement(const char *name, const char **)
{
    bool bTransitionOk = false;
    PARSE_DEBUG("StartElement: %s\n", name );
    switch( m_parseState )
    {
    case XmlParseState_Initial:
        bTransitionOk = true; // ignore everything til we see what we care about
        if( !strcmp(name, "HtcsInfo") )
        {
            PARSE_DEBUG("--------- went into htcsinfo\n");
            m_parseState = XmlParseState_InHtcsInfo;
        }
        break;
    case XmlParseState_InHtcsInfo:
        bTransitionOk = true; // peacefuly ignore any other elements that start
        if( !strcmp(name, "PortMap") )
        {
            PARSE_DEBUG("--------- went into portmap\n");
            m_parseState = XmlParseState_InHtcsInfoPortMap;
        }
        break;
    case XmlParseState_InHtcsInfoPortMap:
        if( !strcmp(name, "PortMapItem") )
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItem;
            bTransitionOk = true;
            m_currPortName[0] = 0;
            m_currPortIP[0] = 0;
            m_currPort[0] = 0;
            PARSE_DEBUG("--------- PortMapItem begins\n");
        }
        // won't accept anything but PortMapItem at level 0
        break;
    case XmlParseState_InHtcsInfoPortMapItem:
        bTransitionOk = true; // will take anything
        m_otherItemNesting = 0;
        if( !strcmp(name, "HtcsPortName") )
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItemPortName;
        }
        else if( !strcmp(name, "IPAddress") )
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItemIPAddress;
        }
        else if( !strcmp(name, "TcpPortNumber") )
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItemPort;
        }
        else  // some other stuff ?
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItemOther;
        }
        break;
    case XmlParseState_InHtcsInfoPortMapItemPortName:
    case XmlParseState_InHtcsInfoPortMapItemIPAddress:
    case XmlParseState_InHtcsInfoPortMapItemPort:
    case XmlParseState_InHtcsInfoPortMapItemOther:
        ++m_otherItemNesting;
        break;
    default:
        // no action - will naturally fail parsing
        break;
    }

    if( !bTransitionOk )
    {
        TriggerParseError();
    }
}

void nnt::net::HTCSFactory::XmlCharacterData( const char *s, int len )
{
    if( m_otherItemNesting > 0 )
    {
        // not interested in any data other than at level 0
        return;
    }

    switch( m_parseState )
    {
    case XmlParseState_InHtcsInfoPortMapItemPortName:
        PARSE_DEBUG("Lenghts are %d / %d\n", HTCS_ARRAY_SIZE(m_currPortName), len );

        HTCS_COPY_CSTRING( m_currPortName, s, len );
        PARSE_DEBUG("Found port name %s\n", m_currPortName );
        break;
    case XmlParseState_InHtcsInfoPortMapItemIPAddress:
        HTCS_COPY_CSTRING( m_currPortIP, s, len );
        break;
    case XmlParseState_InHtcsInfoPortMapItemPort:
        HTCS_COPY_CSTRING( m_currPort, s, len );
        break;
    default:
        // no action - will naturally fail parsing
        break;
    }
}

void nnt::net::HTCSFactory::TriggerServerFound()
{
    bool bRegisterSlot = false;
    m_bParserFoundServer = true;

    m_timeoutEvent.Set(); // prevent timeout from triggering

    if( m_ticsSessionName[0] != 0 )
    {
        bRegisterSlot = true;
        ::tics::TCPSlot* pTcpSlot = new ::tics::TCPSlot(
            "tcp", "bridge_client", m_currPortIP, m_currPort);
        m_pCrossBar->AddSlot(m_ticsSessionName, pTcpSlot);
    }

    if( !m_bAutoTeardown )
    {
        // notify right away, and stay up -
        // user can search for more servers if they want
        OnServerFound( bRegisterSlot ? m_ticsSessionName : NULL ,
            m_currPortIP,
            m_currPort );
    }
    else
    {
        ControlSessionStop();
        // and will notify later when we get this event
    }
}


void nnt::net::HTCSFactory::XmlEndElement(const char *name)
{
    PARSE_DEBUG("EndElement: %s\n", name );
    bool bTransitionOk = false;
    switch( m_parseState )
    {
    case XmlParseState_Initial:
        bTransitionOk = true; // things may come and go, I don't care in this state
        break;
    case XmlParseState_InHtcsInfoPortMapItem:
        if( !strcmp(name, "PortMapItem") )
        {
            PARSE_DEBUG("--------- PortMapItem ends\n\n");
            m_parseState = XmlParseState_InHtcsInfoPortMap; // ready to get next item
            bTransitionOk = true;
            if( (!m_bParserFoundServer)  &&
                (m_currPortName[0] != 0) &&
                (m_currPortIP[0] != 0)   &&
                (m_currPort[0] != 0)  )
            {
                if( (!strcmp(m_serverPortName, m_currPortName)) )
                {
                    NN_LOG("   USE port [%s] [%s] [%s] \n", m_currPortName, m_currPortIP, m_currPort );
                    TriggerServerFound();
                }
                else
                {
                    NN_LOG("   skip port [%s] [%s] [%s] \n", m_currPortName, m_currPortIP, m_currPort );
                }
            }
        }
        break;
    case XmlParseState_InHtcsInfoPortMapItemPortName:
    case XmlParseState_InHtcsInfoPortMapItemIPAddress:
    case XmlParseState_InHtcsInfoPortMapItemPort:
    case XmlParseState_InHtcsInfoPortMapItemOther:
        if( m_otherItemNesting == 0)
        {
            m_parseState = XmlParseState_InHtcsInfoPortMapItem;
            bTransitionOk = true;
        }
        else
        {
            --m_otherItemNesting;
        }
        break;

    case XmlParseState_InHtcsInfoPortMap:
        if( !strcmp(name, "PortMap" ) )
        {
            bTransitionOk = true;
            m_parseState = XmlParseState_InHtcsInfo;
        }
        break;

    case XmlParseState_InHtcsInfo:
        bTransitionOk = true;
        if( !strcmp(name, "HtcsInfo") )
        {
            PARSE_DEBUG("--------- Back to initial state\n\n");
            m_parseState = XmlParseState_Initial;
        }
        break;
    default:
        // no action - will naturally fail parsing
        break;
    }
    if( !bTransitionOk )
    {
        TriggerParseError();
    }
}



void nnt::net::HTCSFactory::Control::OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP)
{
    // base invoke
    ::tics::Plug::OnSessionStarted(type, connectedEP);
    m_pParent->ControlSessionStart();
    BeginReceive( 1, xmlTextBufferSize, &m_buffer, 0 );
}

int nnt::net::HTCSFactory::Control::OnReceiveComplete(::tics::Buffer* buffer, long, long len)
{
    if( m_pParent->Parse( (const char *)(buffer->GetPayLoad()), len ) )
    {
        // register to receive a brand new buffer
        BeginReceive( 1, xmlTextBufferSize, &m_buffer, 0 );
    }
    return 0;
}






void nnt::net::SimpleHTCSPlug::OnServerFound( const char *pTicsSessionName, const char *, const char * )
{
    NN_LOG("\n Channel detected as %s - requesting TCP connection!!\n", pTicsSessionName );
    if( pTicsSessionName != NULL )
    {
        m_factory.GetCrossBar()->RegisterForSessionStart( pTicsSessionName, this );
    }
}

void nnt::net::SimpleHTCSPlug::OnParseError() // override
{
    // will cause Run() to return
    m_factory.GetCrossBar()->BeginShutdown(::tics::CrossBar::ExitDone);
}


