﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 HTCS Utility class declaration.
  - This utility class connects to TargetManager,
      -  parses the XML it outputs that describes all open HTCS ports,
      -  until it finds one specific port, named by the user.
      -  The output of this process is the port's IP address and port nuber
  - The class offers the optional service of creating a TICS TCPSlot for it, such
    that the user can benefit from the TCP plug wrapper.
 *---------------------------------------------------------------------------*/

#pragma once

#ifndef __NNT_HTCS_UTILS_H__
#define __NNT_HTCS_UTILS_H__

#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <RawMemBuffer.h>
#include <CrossBar.h>
#include <TCPslot.h>


namespace nnt { namespace net {




class HTCSFactory
{

public:
    // pLocalHost/pLocalHost: connetion to local target manager (typically localhost/8003)
    // pServerPortName:  the port registered by the server-side in the SDEV program,
    //                   via htcs::Bind.
    // pTicsSessionName: if non-null, causes this class to register a tics
    //                   session with that name for the local client connection.
    //                   This way, the user can benefit from tics to wrap
    //                   the socket for the session data.  If client does not
    //                   intend to use tics, use NULL - then client will use his
    //                   own means to connect to the IP/port reported in OnServerFound
    HTCSFactory( tics::CrossBar *pCrossBar,
                 const char *pLocalHost, const char *pLocalPort,
                 const char *pServerPortName, const char *pTicsSessionName );

    void SetTimeout( int timeoutMs );

    // ---------- customer overrides
    // when this is called the user is free to connect to the host/port as client.
    // if pTicsSessionName is not NULL, the user can use TICS for the connection by using
    // RegisterForSessionStart on a plug.
    virtual void OnServerFound( const char *pTicsSessionName, const char *pHostName, const char *pPort ) = 0;
    virtual void OnParseError() { }
    virtual void OnControlDisconnected() {}


    // ------------ customer state query
    bool ParseErrorFound();
    bool ParseStateOk();
    bool TimedOut()
    {
        return m_bTimeoutFound;
    }
    ::tics::CrossBar *GetCrossBar()
    {
        return m_pCrossBar;
    }


    // ------------- customer settings/control
    void SetAutoTeardown()
    {
        m_bAutoTeardown = true;
    }

    void ResetSearch() // use to find multiple servers - reset upon finding each one
    {
        m_bParserFoundServer = false;
    }

    bool Clear(); // when search is fully done and crossbar exits, Clear() to re-use object

    void StopControlParsing();



public: // destructor
    virtual ~HTCSFactory();


private: // constants
    static const char *htcsFactorySessionName;
    static const int PortInfoFieldSize = 64;
    static const int TimeoutNotSet = -1;


private: // callbacks
    void XmlStartElement(const char *name, const char **atts);
    void XmlEndElement(const char *name);
    void XmlCharacterData( const char *s, int len );
    void ControlSessionStart();
    void ControlSessionStop();
    void TimeoutThread();


private:
    // callbacks shims for parser C interface
    static void XmlStartElement(void *pThis, const char *name, const char **atts)
    {
        ((HTCSFactory*)(pThis))->XmlStartElement( name, atts );
    }

    static void XmlEndElement(void *pThis, const char *name)
    {
        ((HTCSFactory*)(pThis))->XmlEndElement( name );
    }
    static void XmlCharacterData(void *pThis, const char *s, int len)
    {
        ((HTCSFactory*)(pThis))->XmlCharacterData( s, len );
    }
    static void TimeoutThreadHook( void* pThis )
    {
        ((HTCSFactory*)(pThis))->TimeoutThread();
    }
    static int TriggerTimeout(void *pThis, void *, int, intmax_t )
    {
        ((HTCSFactory*)(pThis))->TriggerParseError();
        return 0;
    }


private: // types
    class Control: public tics::Plug
    {
    private: // constants
        static const int xmlTextBufferSize = 1024;

    public:
        explicit Control( HTCSFactory *pParent ):
            m_pParent( pParent ),
            m_buffer( xmlTextBufferSize )
        {
        }

        void OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP) NN_OVERRIDE;

        int OnReceiveComplete(::tics::Buffer* buffer, long offset, long len) NN_OVERRIDE;

        int OnDetach() NN_OVERRIDE
        {
            m_pParent->ControlSessionStop();
            return 0;
        }

        int OnRemoteDetach( ::tics::DisconnectReason ) NN_OVERRIDE
        {
            m_pParent->ControlSessionStop();
            return 0;
        }


    private:
        HTCSFactory *m_pParent;
        ::tics::RawMemBuffer m_buffer;

    private:
        Control(); // no default constructor
        Control( const Control & ); // no copy constructor
        Control &operator=( const Control & ); // no assignment operator
    };

    enum XmlParseState
    {
        XmlParseState_Initial,
        XmlParseState_InHtcsInfo,
        XmlParseState_InHtcsInfoPortMap,
        XmlParseState_InHtcsInfoPortMapItem,
        XmlParseState_InHtcsInfoPortMapItemPortName,
        XmlParseState_InHtcsInfoPortMapItemIPAddress,
        XmlParseState_InHtcsInfoPortMapItemPort,
        XmlParseState_InHtcsInfoPortMapItemOther,
        XmlParseState_HighLevelParseError
    };


private: // helpers
    bool Parse( const char *text, int tLen = -1 );
    void TriggerParseError(bool bLowLevel = false);
    void TriggerServerFound();
    bool IsControlConnected()
    {
        return m_bControlConnected;
    }
    void Teardown();


private: // data about port being parsed
    char m_currPortName[PortInfoFieldSize];
    char m_currPortIP[PortInfoFieldSize];
    char m_currPort[PortInfoFieldSize];
    char m_serverPortName[PortInfoFieldSize];
    char m_ticsSessionName[PortInfoFieldSize];

private: // data
    Control                  m_control;
    void                    *m_Parser;
    long                     m_xmlStatus;
    bool                     m_bUserTerminationRequested;
    bool                     m_bControlConnected;
    int                      m_otherItemNesting;
    XmlParseState            m_parseState;
    ::tics::CrossBar        *m_pCrossBar;
    bool                     m_bParserFoundServer;
    bool                     m_bAutoTeardown;
    int                      m_timeoutMs;
    ::tics::ManualResetEvent m_timeoutEvent;
    bool                     m_bTimeoutFound;

private:
    HTCSFactory(); // no default constructor
    HTCSFactory( const HTCSFactory & ); // no copy constructor
    HTCSFactory &operator=( const HTCSFactory & ); // no assignment operator

};




// simple implementation if all you ever need is one connection to target,
//  includes a facade to the factory
class SimpleHTCSPlug: public ::tics::Plug
{
private:
    class NestedFactory: public HTCSFactory
    {
    public:
        NestedFactory(SimpleHTCSPlug *pParent, tics::CrossBar *pCrossBar,
                      const char     *pLocalHost,      const char *pLocalPort,
                      const char     *pServerPortName, const char *pTicsSessionName ):
                      HTCSFactory( pCrossBar, pLocalHost, pLocalPort,
                                   pServerPortName, pTicsSessionName ),
                      m_pParent(pParent)
        {
        }

        void OnServerFound( const char *pTicsSessionName, const char *pHostName, const char *port ) NN_OVERRIDE
        {
            m_pParent->OnServerFound(pTicsSessionName, pHostName, port );
        }
        void OnParseError() NN_OVERRIDE
        {
            m_pParent->OnParseError();
        }
        void OnControlDisconnected()
        {
            m_pParent->OnControlDisconnected();
        }

    private:
        SimpleHTCSPlug *m_pParent;

    private:
        NestedFactory(); // no default constructor
        NestedFactory( const NestedFactory & ); // no copy constructor
        NestedFactory &operator=( const NestedFactory & ); // no assignment operator
    };
public:
    SimpleHTCSPlug(tics::CrossBar *pCrossBar,
                   const char     *pLocalHost,      const char *pLocalPort,
                   const char     *pServerPortName, const char *pTicsSessionName ):
        m_factory( this, pCrossBar, pLocalHost, pLocalPort,
                   pServerPortName, pTicsSessionName )
    {
    }

    virtual void OnServerFound( const char *pTicsSessionName, const char *pHostName, const char *port );
    virtual void OnParseError();
    virtual void OnControlDisconnected() {}


    HTCSFactory &GetFactory()
    {
        return m_factory;
    }

private:
    NestedFactory m_factory;

private:
    SimpleHTCSPlug(); // no default constructor
    SimpleHTCSPlug( const SimpleHTCSPlug & ); // no copy constructor
    SimpleHTCSPlug &operator=( const SimpleHTCSPlug & ); // no assignment operator
};




}  }; // nnt::net namespace



#endif // __NNT_HTCS_UTILS_H__


