﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "natf.h"

#define NETTEST_COPY_BUFFER_BASE(BUFFER, INPUT, LEN, BUFFERTYPENAME) \
do \
{ \
    if( (nullptr != (INPUT)) && (nullptr != (BUFFER)) ) \
    { \
        if( NN_STATIC_CONDITION(NATF::Modules::BufferType_String == (BUFFERTYPENAME)) ) \
        { \
            strncpy((BUFFER), (INPUT), (LEN) - 1); \
            (BUFFER)[LEN - 1] = '\0'; \
        } \
        else if( NN_STATIC_CONDITION(NATF::Modules::BufferType_Bytes == (BUFFERTYPENAME)) ) \
        { \
            memcpy((BUFFER), (INPUT), (LEN)); \
        } \
    } \
} while( NN_STATIC_CONDITION(false) )

#define NETTEST_COPY_BUFFER_STRING_ARRAY(BUFFER, INPUT) \
    NETTEST_COPY_BUFFER_BASE((BUFFER), (INPUT), sizeof(BUFFER), NATF::Modules::BufferType_String)
#define NETTEST_COPY_BUFFER_STRING_POINTER(BUFFER, INPUT, LEN) \
    NETTEST_COPY_BUFFER_BASE((BUFFER), (INPUT), (LEN), NATF::Modules::BufferType_String)
#define NETTEST_COPY_BUFFER_BYTES_ARRAY(BUFFER, INPUT) \
    NETTEST_COPY_BUFFER_BASE((BUFFER), (INPUT), sizeof(BUFFER), NATF::Modules::BufferType_Bytes)
#define NETTEST_COPY_BUFFER_BYTES_POINTER(BUFFER, INPUT, LEN) \
    NETTEST_COPY_BUFFER_BASE((BUFFER), (INPUT), (LEN), NATF::Modules::BufferType_Bytes)

namespace NATF {
namespace Modules {

enum AppType
{
    AppType_Client = 0,
    AppType_Server = 1
};

enum InitType
{
    InitType_All    = 0,
    InitType_Init   = 1,
    InitType_Socket = 2,
    InitType_AllExempt = 3,
    InitType_SocketExempt = 4
};

enum TeardownType
{
    TeardownType_Shutdown = 0,
    TeardownType_Close    = 1,
    TeardownType_All      = 2,
    TeardownType_Graceful = 3
};

enum BufferType
{
    BufferType_String  = 0,
    BufferType_Bytes   = 1
};

class SoTestDataXfer : public BaseModule
{
    NN_DISALLOW_COPY(SoTestDataXfer);

public:
    SoTestDataXfer(AppType appType, const char* pIp, const char* xferPattern, unsigned short port, nn::socket::Protocol protoType, InitType initType, int durationSec) NN_NOEXCEPT;
    virtual ~SoTestDataXfer() NN_NOEXCEPT override;

    virtual bool Run() NN_NOEXCEPT override;
    virtual const char* GetName() const NN_NOEXCEPT override;

    static SoTestDataXfer* CreateTestNode(TestThread* pThread, const char* appTypeName, const char* ipAddr, int portNumber, const char* protocolTypeName, int durationSec, const char* xferPattern) NN_NOEXCEPT;
    static SoTestDataXfer* CreateTestNode(TestThread* pThread, const char* appTypeName, const char* ipAddr, int portNumber, const char* protocolTypeName, int durationSec, const char* xferPattern, bool socketExempt) NN_NOEXCEPT;

    static const uint32_t MaxIpAddrStringLen = 16;
    static const uint32_t MaxNameStringLen = 64;
    static const uint32_t MaxPortNumber = 0xFFFF;
    static const unsigned MaxBufSize = 2 * 1024;

private:

    enum AppErr
    {
        AppErr_Ok       = 0,
        AppErr_Continue = 1,
        AppErr_Out      = 2
    };
    struct SoTestParams
    {
        bool isNonBlocking;
        int socketRecvSize;
        int socketSendSize;
        NetTest::SockAddrIn remoteAddr;
        NetTest::SockLen addrLen;

        AppType appType;
        char pIp[MaxIpAddrStringLen];
        char* xferPattern;
        unsigned short port;
        int backLog;
        int socketFd;
        TeardownType teardownType;
        nn::socket::ShutdownMethod shutdownType;
        int duration;
        unsigned bufferSize;

        InitType initType;
        nn::socket::Type socketType;
        nn::socket::Protocol socketProtocol;
        nn::socket::Family socketFamily;
        nn::socket::MsgFlag xferOptions;
        const char* pTestName;
        nn::socket::Errno expectedInitErr;
        nn::socket::Errno expectedSetupErr;
        nn::socket::Errno expectedXferErr;
        nn::socket::Errno expectedTeardownErr;
        unsigned lastRecvIdx;
    };

    bool Teardown(nn::socket::Errno expectedErr) NN_NOEXCEPT;
    bool Init(nn::socket::Errno expectedErr) NN_NOEXCEPT;
    bool Setup(nn::socket::Errno expectedErr) NN_NOEXCEPT;
    bool DataXfer(nn::socket::Errno expectedErr) NN_NOEXCEPT;
    bool SetOpt() NN_NOEXCEPT;

    AppErr ServerTcp(unsigned& totalSent, unsigned& totalRecvd, char* sendBuffer, char* recvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT;
    AppErr ClientTcp(unsigned& totalSent, unsigned& totalRecvd, char* sendBuffer, char* recvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT;
    AppErr ServerUdp(unsigned& totalSent, unsigned& totalRecvd, char* sendBuffer, char* recvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT;
    AppErr ClientUdp(unsigned& totalSent, unsigned& totalRecvd, char* sendBuffer, char* recvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT;

    struct SoTestParams m_params;
};

}
}
