﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_ApiPrivate.h>

namespace NetTest {

/* Types */

typedef nn::socket::SockAddrIn  SockAddrIn;
typedef nn::socket::SockAddr    SockAddr;
typedef nn::socket::PollFd      PollFd;
typedef nn::socket::SockLenT    SockLen;
typedef nn::socket::FdSet       FdSet;
typedef nn::socket::TimeVal     TimeVal;
typedef nn::socket::HostEnt     HostEnt;

/* APIs */

inline bool Init() NN_NOEXCEPT
{
    //static nn::socket::ConfigDefaultWithMemory s_SocketConfigWithMemory;


static nn::socket::ConfigWithMemory
<
    32,                                                            // tcpSocketCountMax - How many nn::socket::Type_Sock_Stream sockets will you use?
    32,                                                            // udpSocketCountMax - How many nn::socket::Type_Sock_Dgram sockets will you use?
    nn::socket::ConfigDefault::DefaultTcpInitialSendBufferSize,    // tcpInitialSendBufferSize
    nn::socket::ConfigDefault::DefaultTcpInitialReceiveBufferSize, // tcpInitialReceiveBufferSize
    nn::socket::ConfigDefault::DefaultTcpAutoSendBufferSizeMax,    // tcpAutoSendBufferSizeMax
    nn::socket::ConfigDefault::DefaultTcpAutoReceiveBufferSizeMax, // tcpAutoReceiveBufferSizeMax
    nn::socket::ConfigDefault::DefaultUdpSendBufferSize,           // udpSendBufferSize
    nn::socket::ConfigDefault::DefaultUdpReceiveBufferSize,        // udpReceiveBufferSize
    nn::socket::ConfigDefault::DefaultSocketBufferEfficiency,      // socketBufferEfficiency
    nn::socket::DefaultSocketAllocatorSize                         // allocatorPoolSize - Used primary by resolver API
>s_SocketConfigWithMemory(nn::socket::DefaultConcurrencyLimit);


    return nn::socket::Initialize(s_SocketConfigWithMemory).IsSuccess();
}

inline void Finalize() NN_NOEXCEPT
{ nn::socket::Finalize(); }

inline int Socket(nn::socket::Family family, nn::socket::Type socketType, nn::socket::Protocol proto) NN_NOEXCEPT
{ return nn::socket::Socket(family, socketType, proto); }

inline int SocketExempt(nn::socket::Family family, nn::socket::Type socketType, nn::socket::Protocol proto) NN_NOEXCEPT
#ifndef NN_BUILD_CONFIG_OS_WIN
{ return nn::socket::SocketExempt(family, socketType, proto); }
#else
{ return nn::socket::Socket(family, socketType, proto); }
#endif

inline int Close(int fd) NN_NOEXCEPT
{ return nn::socket::Close(fd); }

inline int Shutdown(int fd, nn::socket::ShutdownMethod how) NN_NOEXCEPT
{ return nn::socket::Shutdown(fd, how); }

inline int Bind(int fd, const SockAddr* pAddr, SockLen addrLen) NN_NOEXCEPT
{ return nn::socket::Bind(fd, pAddr, addrLen); }

inline int Listen(int fd, int backLog) NN_NOEXCEPT
{ return nn::socket::Listen(fd, backLog); }

inline int Accept(int fd, SockAddr* pAddr, SockLen* pAddrLen) NN_NOEXCEPT
{ return nn::socket::Accept(fd, pAddr, pAddrLen); }

inline int Connect(int fd, const SockAddr* pAddr, SockLen addrLen) NN_NOEXCEPT
{ return nn::socket::Connect(fd, pAddr, addrLen); }

inline ssize_t Send(int fd, const void* pBuf, size_t bufLen, nn::socket::MsgFlag flags) NN_NOEXCEPT
{ return nn::socket::Send(fd, pBuf, bufLen, flags); }

inline ssize_t SendTo(int fd, const void* pBuf, size_t bufLen, nn::socket::MsgFlag flags, SockAddr* pAddr, SockLen addrLen) NN_NOEXCEPT
{ return nn::socket::SendTo(fd, pBuf, bufLen, flags, pAddr, addrLen); }

inline ssize_t Recv(int fd, void* pBuf, size_t bufLen, nn::socket::MsgFlag flags) NN_NOEXCEPT
{ return nn::socket::Recv(fd, pBuf, bufLen, flags); }

inline ssize_t RecvFrom(int fd, void* pBuf, size_t bufLen, nn::socket::MsgFlag flags, SockAddr* pAddr, SockLen* pAddrLen) NN_NOEXCEPT
{ return nn::socket::RecvFrom(fd, pBuf, bufLen, flags, pAddr, pAddrLen); }

inline int GetSockName(int fd, SockAddr* pAddr, SockLen* pAddrLen) NN_NOEXCEPT
{ return nn::socket::GetSockName(fd, pAddr, pAddrLen); }

inline int GetPeerName(int fd, SockAddr* pAddr, SockLen* pAddrLen) NN_NOEXCEPT
{ return nn::socket::GetPeerName(fd, pAddr, pAddrLen); }

inline int GetSockOpt(int fd, nn::socket::Level level, nn::socket::Option optName, void* pVal, SockLen* pValLen) NN_NOEXCEPT
{ return nn::socket::GetSockOpt(fd, level, optName, pVal, pValLen); }

inline int SetSockOpt(int fd, nn::socket::Level level, nn::socket::Option optName, const void* pVal, SockLen valLen) NN_NOEXCEPT
{ return nn::socket::SetSockOpt(fd, level, optName, pVal, valLen); }

inline int Fcntl(int fd, int command, ...) NN_NOEXCEPT
{
    int     iocmd;
    va_list args;
    va_start(args, command);
    iocmd = va_arg(args, int);
    va_end(args);
    return nn::socket::Fcntl(fd, static_cast<nn::socket::FcntlCommand>(command), iocmd);
}

inline int Ioctl(int socket, nn::socket::IoctlCommand command, void* pData, size_t dataLength)
{ return nn::socket::Ioctl(socket,command, pData, dataLength); }

inline int Poll(PollFd* pFds, nn::socket::NfdsT fdCount, int timeout) NN_NOEXCEPT
{ return nn::socket::Poll(pFds, fdCount, timeout); }

inline int Select(int fdCount, FdSet* pReadFds, FdSet* pWriteFds, FdSet* pErrFds, TimeVal* pTimeVal) NN_NOEXCEPT
{ return nn::socket::Select(fdCount, pReadFds, pWriteFds, pErrFds, pTimeVal); }

inline nn::socket::Errno GetLastError() NN_NOEXCEPT
{ return nn::socket::GetLastError(); }

inline int InetAddrPton(nn::socket::Family family, const char* pStr, void* pDest) NN_NOEXCEPT
{ return nn::socket::InetPton(family, pStr, pDest); }

inline char* InetNtoa(nn::socket::InAddr networkAddress) NN_NOEXCEPT
{ return nn::socket::InetNtoa(networkAddress); }

inline HostEnt* GetHostByName(const char* pName) NN_NOEXCEPT
{ return nn::socket::GetHostEntByName(pName); }

inline int16_t Htons(int16_t val) NN_NOEXCEPT
{ return nn::socket::InetHtons(val); }

inline int16_t Ntohs(int16_t val) NN_NOEXCEPT
{ return nn::socket::InetNtohs(val); }

inline int32_t Htonl(int32_t val) NN_NOEXCEPT
{ return nn::socket::InetHtonl(val); }

inline int32_t Ntohl(int32_t val) NN_NOEXCEPT
{ return nn::socket::InetNtohl(val); }

} // namespace NetTest
