﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket.h>

#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_UnitCommonSocket.h"
#include "Complex/testNet_SelectUnitData.h"

namespace NATF {
namespace API {
/**
 * @brief SelectUnitNetworkCommon contains code that is common across both the SelectUnitServer and SelectUnitClient classes
 */
class SelectUnitNetworkCommon
{
public:
    virtual void SetPeer(UnitTestThreadBase* pPeerThread);

protected:
    /** @brief ctor */
    SelectUnitNetworkCommon(SelectUnitData* pSelectUnitData, SocketContainerActor role, UnitTestThreadBase* pSelfThread);

    /** @brief dtor */
    virtual ~SelectUnitNetworkCommon();

    /** @brief main loop for select threads */
    void RunLoop();

    /** @brief function to initialize sockets */
    virtual void InitializeSockets(SocketContainer& container) = 0;

    /** @brief Get the count of socket containers given a specific type  */
    virtual unsigned int GetSocketContainerTypeCountAndSet(SocketContainerTypeFlags socketContainerType, nn::socket::FdSet* & pFdSetInOut) const;

    /**
     * @brief handle fd sets
     * @param pFileDescriptorSetIn the file descriptor set to check
     * @param events the events to check
     * @param closedASocket flag set if a socket is shut down; indicator to bypass usual event check; NOT reset on use
     * @returns the number of events handled
     */
    virtual int DispatchEvents(nn::socket::FdSet* pReadFds,nn::socket::FdSet* pWriteFds, nn::socket::FdSet* pExceptFds );

    /** @brief callback when new connections arrive, place. Add embryonic connections to AddSocketContainerToWaitingQueue */
    virtual int OnNewConnectionEvent(SocketContainer& container) = 0;

    /** @brief OnEmbryonicReadevent data from the client describing the connection has arrived */
    virtual int OnEmbryonicReadEvent(SocketContainer& container) = 0;

    /** @brief OnReadEvent a select read event happened */
    virtual int OnReadEvent(SocketContainer& container) = 0;

    /** @brief OnWriteEvent a write event for the container happened */
    virtual int OnWriteEvent(SocketContainer& container) = 0;

    /** @brief OnExceptionEvent a select 'exception' (i.e. out of band data) occurred */
    virtual int OnExceptionEvent(SocketContainer& container) = 0;

    /** @brief get the maximum file descriptor from the socket container list */
    virtual int GetMaxFd();

    /** @brief get the list of sockets from the socket manager that are for our role */
    virtual void GetSiftedList();

    /**
     * @brief new sockets are added to a waiting queue
     * so that they can be added to the socket container list when the current select loop finishes
     * adding mid-stream may cause issues where new sockets are considered set (i.e.  nn::socket::FdSetIsSet ) even though they are not
     */
    void AddSocketContainerToWaitingQueue(SocketContainer& newContainer);

    /**
     * @brief convenience function that sends a socket container
     */
    ssize_t SendContainer(SocketContainer container, const char* eventName, nn::socket::MsgFlag flags);

    /**
     * @brief convenience function that receives a socket container
     */
    ssize_t ReceiveContainer(SocketContainer& containerTo, SocketContainer& containerFrom, const char* eventName, nn::socket::MsgFlag flags);

    /**
     * @brief recv count bytes, or return on error
     */
    virtual ssize_t ReceiveBytes(const SocketContainer& container, uint8_t* pBufferPointer, size_t count, nn::socket::MsgFlag flags);

    /**
     * @brief send count bytes, or return on error
     */
    virtual ssize_t SendBytes(SocketContainer& container, const uint8_t* buffer, size_t bufferLength, nn::socket::MsgFlag flags);

    /**
     * @brief shutdown process
     * @return true if you can continue false if iteratorInOut is at the end of the list
     */
    virtual bool ShutdownSocket(std::list<SocketContainer>::iterator & iteratorInOut);

    /** access lock for manager */
    mutable nn::os::MutexType m_CommonAccessLock;

    SelectUnitData* m_pSelectUnitData;

    /**
     * @brief the validator
     */
    SimpleValidator* m_pValidator;

    /** @brief read socket list, care should be used when accessing this from subclasses */
    std::list<SocketContainer> m_SocketContainerList;

    /**
     * @brief my role
     */
    const SocketContainerActor m_Role;

private:
    virtual int DispatchEventsInternal(nn::socket::FdSet* pFileDescriptorSetIn, SocketContainerTypeFlags events);

    UnitTestThreadBase* m_pSelfThread;

    UnitTestThreadBase* m_pPeerThread;

    /** @brief read socket list */
    std::list<SocketContainer> m_ShutdownQueue;

    /**
     * @brief waiting queue, purged after each call to select
     */
    std::list<SocketContainer> m_WaitingQueue;


};

}}; // NATF::API
