﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket.h>

#include <list>

#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_UnitCommonSocket.h"

namespace NATF {
namespace API {

/**
 * @brief SelectUnitData represents data for a single testcase
 */
class SelectUnitData :
        public virtual LockedReferenceCountObjectImpl,
        public virtual ISocketContainerListManager
{
public:
    /**
     * @brief Get a copy of the set for all acquired read file descriptors
     * @implements ISocketContainerListManager::GetSocketContainerListCopy
     */
    virtual void GetSocketContainerListCopy(std::list<SocketContainer>& socketContainerListOut) const;

    /**
     * @brief Get a copy of the set for all acquired read file descriptors
     * @implements ISocketContainerListManager::AddSocketContainerToList
     */
    virtual void AddSocketContainerToList(const SocketContainer& socketContainerIn);

    /**
     * @brief Get a copy of the set for all acquired read file descriptors
     * @implements ISocketContainerListManager::RemoveSocketContainerFromList
     */
    virtual void RemoveSocketContainerFromList(const SocketContainer& socketContainerIn);

    /** @brief Get the port */
    virtual uint16_t GetPort() const;

    /** @brief Set the port */
    virtual void SetPort(uint16_t port);

    /**
     * @brief Get a reference to the validator or NULL if not set
     * @note in the case that it is set the reference count is increased +1
     *       so a call to releaseReference() is necessary
     */
    virtual SimpleValidator* GetValidator();

    /**
     * Get a copy of the timeout value for this test
     */
    void GetSelectTimeout(nn::socket::TimeVal *& pSelectTimeoutInOut) const;

    /**
     * @brief constructor
     * @param flags the flags for types of sockets to create (i.e. ST_SOCK_STREAM|ST_)
     * @param numberRead the number of read socket to acquire
     * @param numberWrite the number of write socket to acquire
     * @param numberError the number of error sockets to acquire
     * @param testTypeFlags the flags for this test
     * @param validator the validator for this test
     */
    SelectUnitData(AddressFamilyFlags addressFamily,
                   SocketTypeFlags socketTypeFlags,
                   ProtocolFlags protocolFlags,
                   SocketControlFlags socketControlFlags,
                   unsigned int numberRead,
                   unsigned int numberWrite,
                   unsigned int numberError,
                   SimpleValidator* validator,
                   uint64_t selectTimeoutInMs);


protected:
    /** @brief cleanup, called after the test is stopped */
    virtual void Cleanup();

    /** @brief dtor */
    virtual ~SelectUnitData();

private:
    /** @brief nn::socket::Family::Af_Inet, etc. */
    const AddressFamilyFlags m_AddressFamilyFlags;

    /** nn::socket::Type::Sock_Stream, nn::socket::Type::Sock_Dgram, etc. */
    const SocketTypeFlags m_SocketTypeFlags;

    /** @brief protocol flags */
    ProtocolFlags m_ProtocolFlags;

    /** @brief ioctl / fcntl flags */
    SocketControlFlags m_ControlFlags;

    /** @brief the port used for the client and server */
    uint16_t m_Port;

    /**
     * acquire 'number' of sockets of type flags and add to socketMapOut
     * @param adddressFamilyFlags the 1 address family flags to pass to nn::socket::Socket
     * @param socketTypeFlags the 1 socket type flags to pass to nn::socket::Socket
     * @param protocolFlags the 1 protocol flags to pass to nn::socket::Socket
     * @param role determines whether the sockets being acquired are for use by the server, client, etc.
     * @param containerTypeFlags determines how the actor is going to use the socket
     * @param controlFlags ioctl / fcntl flags
     * @param number the number of sockets to acquire
     * @param socketMapOut sockets are placed here
     * @return the number of sockets acquired
     */
    int AcquireSockets(AddressFamilyFlags addressFamilyFlags,
                       SocketTypeFlags socketType,
                       ProtocolFlags protocolFlags,
                       SocketContainerActor role,
                       SocketContainerTypeFlags containerTypeFlags,
                       SocketControlFlags controlFlags,
                       int number,
                       std::list<SocketContainer> & socketContainerList);

    /**
     * @brief the map of sockets
     */
    std::list<SocketContainer> m_SocketContainerList;

    /**
     * @brief an instance to the validator that validates each SelectUnitData
     */
    SimpleValidator* m_Validator;

    /**
     * @brief timeout for select
     */
    const uint64_t m_SelectTimeout;

    /** @brief data are shared across client and server threads */
    mutable nn::os::MutexType m_DataAccessLock;
};
}}; // NATF::API
