﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nerd/nerd_InertialMeasurementUnit.h>
#include <core/nerd_io.h>
#include <core/nerd_print.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <string>

#include "SensorData.h"


namespace nerd {
namespace {
Imu::WorkingMemory workingMemory;
char* cacheBuffer= nullptr;;
void MountRom()
{
    size_t cacheSize=0;
    NN_LOG("Mount Rom\n");
    (void)nn::fs::QueryMountRomCacheSize(&cacheSize);
    cacheBuffer= new(std::nothrow) char[cacheSize];
    NN_ASSERT(cacheBuffer);
    nn::Result result= nn::fs::MountRom("data", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
void UnMountRom()
{
    nn::fs::Unmount("data");
    delete[] cacheBuffer;
    cacheBuffer= nullptr;
}

void StillnessPercentage(std::string path)
{
    Imu imu;
    imu.Init(workingMemory);
    auto data = LoadData(path.c_str(), 1);
    imu.StartOnlineCalibration();
    int stillness = 0;
    for (int i = 0;i < data.omegas.Size();++i)
    {
        imu.AddSample(data.localAccels[i], data.omegas[i], data.dts[i]);
        stillness += imu.GetIsStill();
    }
    imu.StopOnlineCalibration();
    float percentStill = 100.0f * stillness / (float)data.omegas.Size();
    //TEST(percentStill > 99.0f, "%s %.1f%% still > 99%%", path.c_str(), percentStill);
    EXPECT_GE(percentStill, 99.0f);
}
void AngularError(std::string path)
{
    Imu imu;
    imu.Init(workingMemory);
    auto data = LoadData(path.c_str(), 1);
    imu.StartOnlineCalibration();
    Quaternion<float> start, end;

    for (int i = 0;i < data.omegas.Size();++i)
    {
        imu.AddSample(data.localAccels[i], data.omegas[i], data.dts[i]);
        if (i == 1000)
        {
            start = imu.GetRotation();
        }
    }
    imu.StopOnlineCalibration();
    end = imu.GetRotation();
    float angularError = (end * start.Conjugated()).Log().VectorPart().Length() * 180.0f / (float)constants::PI;
    //"%s angular error= %f < 5 degrees", path.c_str(), angularError);
    EXPECT_LE(angularError, 5.0f);
}


void GravityEstimateWhenSliding(std::string path)
{
    Imu imu;
    imu.Init(workingMemory);
    auto data = LoadData(path.c_str(), 1);
    imu.StartOnlineCalibration();
    Quaternion<float> start, end;
    float3 gravityZero;
    float maxGravityEstimateError = 0.0f;
    for (int i = 0;i < data.omegas.Size();++i)
    {
        imu.AddSample(data.localAccels[i], data.omegas[i], data.dts[i]);
        auto grav = imu.GetGravity();
        if (i < 100)
        {
            gravityZero = grav;
        }
        float gravityError = 180.0f / constants::PI*imu.GetVerticalizationError();
        //float gravityError = 180.0f/constants::PI*fabs(atan2(cross(grav, gravityZero).Length(), dot(grav, gravityZero)));
        maxGravityEstimateError = std::max(gravityError, maxGravityEstimateError);
    }
    imu.StopOnlineCalibration();
    //TEST(maxGravityEstimateError < 5.0f, "%s gravity estimate error= %f < 5 degrees", path.c_str(), maxGravityEstimateError);
    EXPECT_LE(maxGravityEstimateError, 5.0f);//, "%s gravity estimate error= %f < 5 degrees", path.c_str(), maxGravityEstimateError);
}


}
}

//!< Stillness: Check that when the device is not moving we detect it's still
TEST(SanjiroTest, Stillness)
{
    std::string stillTest[] = {
        "data:/stress/stress1.csv",
        "data:/stillness/20170115221043.csv",
        "data:/stillness/20170115221208.csv",
    };
    nerd::MountRom();
    for (auto& p : stillTest) nerd::StillnessPercentage(p);
    nerd::UnMountRom();
}
//!< Paddling Movements: check that when me move the console around the angular error remains inferior to 5 degrees.
TEST(SanjiroTest, Paddling)
{
    std::string PaddleTest[] = {
        "data:/paddling/20170101180412.csv",
        "data:/paddling/20170101180432.csv",
        "data:/paddling/20170101180450.csv"
    };
    NERD_PRINT("Paddling movement\n");
    nerd::MountRom();
    for (auto& p : PaddleTest) nerd::AngularError(p);
    nerd::UnMountRom();
}

//!< Three turns around three axis: check that when me move the console around the angular error remains inferior to 5 degrees.
TEST(SanjiroTest, ThreeTurnsXYZ)
{
    std::string turnsTest[] = {
        "data:/ThreeTurnsXYZ/20170101183756.csv",
        "data:/ThreeTurnsXYZ/20170101183833.csv",
        "data:/ThreeTurnsXYZ/20170101183908.csv"
    };
    NERD_PRINT("Three turns around each axis\n");
    nerd::MountRom();
    for (auto& p : turnsTest) nerd::AngularError(p);
    nerd::UnMountRom();
}


//!< Slide the device and checks that our gravity estimate stays close to vertical (<5 degrees)
TEST(SanjiroTest, GravityEstimateWhenSliding)
{
    std::string gravityEstimate[] = {
        "data:/wobble/20170109174329.csv",
        "data:/wobble/20170109174250.csv",
        "data:/wobble/20170109170503.csv",
        "data:/wobble/20170109170352.csv"
    };
    NERD_PRINT("sliding movement\n");
    nerd::MountRom();
    for (auto& p : gravityEstimate) nerd::GravityEstimateWhenSliding(p);
    nerd::UnMountRom();
}
