﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdlib.h> // NOLINT
#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>
#include <utility>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os.h>

#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2012
#define STR_TO_ULL _strtoui64
#else
#define STR_TO_ULL strtoull
#endif

namespace {
    typedef std::string String;

    char s_Buffer[1024 * 1024];

    String GetNowString()
    {
        using namespace std::chrono;

        auto duration = high_resolution_clock::now().time_since_epoch();
        time_t t = duration_cast<seconds>(duration).count();
        tm now;
        localtime_s(&now, &t);

        char stamp[32];
        sprintf_s(stamp, sizeof(stamp),
            "%04u%02u%02u%02u%02u%02u%03u",
            now.tm_year + 1900, now.tm_mon + 1, now.tm_mday, now.tm_hour, now.tm_min, now.tm_sec,
            static_cast<uint32_t>(duration_cast<milliseconds>(duration).count() % 1000));

        return String(stamp);
    }

    String CreateTemporaryRootPath()
    {
        return String("C:\\Windows\\Temp")
            .append("\\")
            .append("SIGLO_NCM_TEST")
            .append("_")
            .append(GetNowString());
    }

    template<typename HandleT>
    class ScopedHandle
    {
    public:
        typedef void(*CloseFunction)(HandleT)NN_NOEXCEPT;
        explicit ScopedHandle(HandleT handle, CloseFunction func) NN_NOEXCEPT : m_Handle(handle), m_Func(func){}
        ~ScopedHandle()
        {
            m_Func(m_Handle);
        }

    private:
        HandleT         m_Handle;
        CloseFunction   m_Func;
    };

    class ScopedDirectoryHandle : private ScopedHandle<nn::fs::DirectoryHandle>
    {
    public:
        explicit ScopedDirectoryHandle(nn::fs::DirectoryHandle handle) NN_NOEXCEPT : ScopedHandle<nn::fs::DirectoryHandle>(handle, nn::fs::CloseDirectory){}
    };

    nn::Result ShowDirectoryTreeRecursively(const char* rootPath, int indent)
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, rootPath, static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_File | nn::fs::OpenDirectoryMode_Directory)));
        ScopedDirectoryHandle scopedDirectoryHandle(dir);

        for (;;)
        {
            nn::fs::DirectoryEntry entry;
            int64_t numEntries;
            NN_RESULT_DO(nn::fs::ReadDirectory(&numEntries, &entry, dir, 1));
            if (numEntries == 0)
            {
                break;
            }

            for (auto i = 0; i < indent; i++)
            {
                NN_LOG("  ");
            }
            switch (entry.directoryEntryType)
            {
            case nn::fs::DirectoryEntryType_Directory:
                {
                    NN_LOG("%s/\n", entry.name);
                    ShowDirectoryTreeRecursively((std::string(rootPath) + "/" + entry.name).c_str(), indent + 1);
                }
                break;
            case nn::fs::DirectoryEntryType_File:
                {
                    NN_LOG("%s %lld bytes\n", entry.name, entry.fileSize);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ShowDirectoryTree(const char* rootPath)
    {
        NN_LOG("Show directory tree %s\n", rootPath);

        return ShowDirectoryTreeRecursively(rootPath, 0);
    }

    template<typename FuncT>
    void ExecuteAllContentDirectoryHierarchy(FuncT func)
    {
        func(nn::ncm::MakeFlatContentFilePath);
        func(nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat4KCluster);
        func(nn::ncm::MakeSha256HierarchicalContentFilePath_ForFat32KCluster);
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return std::free(p);
    }

    struct ContentInfo
    {
        nn::ncm::PlaceHolderId  placeHolderId;
        nn::ncm::ContentId      contentId;
        size_t                  size;
        std::vector<int64_t>    data;
    };

    class InstallTaskTest : public testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            nn::fs::SetAllocator(Allocate, Deallocate);
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }
    };
}

void TestInstallNsp(const char* nspPath, nn::ncm::StorageId storage)
{
    nn::fs::FileHandle file;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&file, nspPath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{nn::fs::CloseFile(file);};

    nn::ncm::SubmissionPackageInstallTask task;
    NNT_EXPECT_RESULT_SUCCESS(task.Initialize(file, storage, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(task.Prepare());
    auto progress = task.GetProgress();
    NN_LOG("Prepared. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);
    EXPECT_GT(progress.totalSize, progress.installedSize);
    NNT_EXPECT_RESULT_SUCCESS(task.Execute());
    progress = task.GetProgress();
    EXPECT_EQ(progress.installedSize, progress.totalSize);
    NN_LOG("Installed. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);
    NNT_EXPECT_RESULT_SUCCESS(task.Commit());
}

TEST_F(InstallTaskTest, Unit)
{
    nn::ncm::Initialize();

    const char* nspDirectoryPath = nnt::GetHostArgv()[1];
    NN_LOG("nspDirectoryPath: %s\n", nspDirectoryPath);

    nn::fs::DirectoryHandle dir;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dir, nspDirectoryPath, nn::fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseDirectory(dir); };

    while(NN_STATIC_CONDITION(true))
    {
        int64_t readCount;
        nn::fs::DirectoryEntry entry;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readCount, &entry, dir, 1));
        if(readCount == 0)
        {
            break;
        }

        auto name = std::string(entry.name);
        if (name == "testNcm_InstallTask.nsp")
        {
            break;
        }

        auto ext = std::string(".nsp");

        if(name.substr(name.size() - ext.size()) == ext)
        {
            auto nspPath = std::string(nspDirectoryPath) + "/" + name;
            NN_LOG("nspPath: %s\n", nspPath.c_str());
            TestInstallNsp(nspPath.c_str(), nn::ncm::StorageId::BuildInSystem);
        }
    }

    nn::ncm::Finalize();
}
